/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Feb  8 12:12:00 2005 by Jeff Dalton
 * Copyright: (c) 2001, 2005, AIAI, University of Edinburgh
 */

package ix.iface.util;

import javax.swing.*;

import java.awt.event.*;
import java.awt.Component;
import java.util.*;

import ix.util.*;

/**
 * An object that manages a group of radio buttons in a convenient
 * and easy to use fashion. <p>
 *
 * Essentially it is a JPanel with a BoxLayout, and components of all
 * sorts can be added to it; however, if a JRadioButton is added, the
 * button is added to a ButtonGroup and given an ActionListener that records
 * its action command when it is selected, as well as passing the
 * ActionEvent to this box's own ActionListeners.  The most recently selected
 * action command can be obtained by calling {@link #getSelection()}.
 */
public class RadioButtonBox extends JPanel {

    /** This box's ActionListeners. */
    protected List actionListeners = new LinkedList();

    /** This box's ButtonGroup. */
    protected ButtonGroup group = new ButtonGroup();

    /** The action-command from the most recently selected button. */
    protected String selection = null;

    /** 
     * The listener this box attaches to radio buttons.
     * It arranges for ActionEvents from the buttons to
     * be passed to this box's ActionListeners.
     *
     * @see #addActionListener(ActionListener)
     */
    protected ActionListener listener = new ActionListener() {
	public void actionPerformed(ActionEvent event) {
	    selection = event.getActionCommand();
	    Debug.noteln("Radio button selection", selection);
	    for (Iterator i = actionListeners.iterator(); i.hasNext();) {
		ActionListener l = (ActionListener)i.next();
		l.actionPerformed(event);
	    }
	}
    };

    /**
     * Creates a box that places components along the specified axis.
     *
     * @param axis either <code>BoxLayout.X_AXIS</code> or
     *     <code>BoxLayout.Y_AXIS</code> for a horizontal or
     *     vertical box respectively.
     */
    public RadioButtonBox(int axis) {
	setLayout(new BoxLayout(this, axis));
    }

    /**
     * Creates a RadioButtonBox that arranges components from left to right.
     */
    public static RadioButtonBox createHorizontalBox() {
	return new RadioButtonBox(BoxLayout.X_AXIS);
    }

    /**
     * Creates a RadioButtonBox that arranges components from top to bottom.
     */
    public static RadioButtonBox createVerticalBox() {
	return new RadioButtonBox(BoxLayout.Y_AXIS);
    }

    /**
     * Returns the action command specified by the most recently
     * selected JRadioButton in the box.
     */
    public String getSelection() {
	return selection;
    }

    /**
     * Selects the corresponding button.
     */
    public void setSelection(String text) {
	for (Enumeration e = group.getElements(); e.hasMoreElements();) {
	    JRadioButton button = (JRadioButton)e.nextElement();
	    if (button.getText().equals(text)) {
		// N.B. setSelected(true) does not cause an action event
		// so we use doClick() instead.
		button.doClick();
		return;
	    }
	}
	throw new RuntimeException("No button with text " 
				   + Strings.quote(text));
    }

    /**
     * Called whenever a component is added to this RadioButtonBox.
     * This method ensures that any JRadioButton that is added
     * is given an ActionListener that will record the button's
     * action-command as this box's current selection when the
     * button is selected and will pass the ActionEvent generated
     * by the button to this RadioButtonBox's own ActionListeners.
     * The button is also added to a ButtonGroup.  If a button
     * isSelected() at the time when it is added, its action command
     * is recorded as this box's current selection.
     *
     * @see #addActionListener(ActionListener)
     * @see #listener
     * @see #group
     */
    protected void addImpl(Component comp, Object constraints, int index) {
	if (comp instanceof JRadioButton) {
	    JRadioButton button = (JRadioButton)comp;
	    group.add(button);
	    button.addActionListener(listener);
	    if (button.isSelected())
		selection = button.getActionCommand();
	}
	super.addImpl(comp, constraints, index);
    }

    /**
     * Adds an ActionListener to this RadioButtonBox.
     * ActionEvents from the JRadioButtons in this box
     * are passed to all of these listeners when they occur.
     */
    public void addActionListener(ActionListener listener) {
	actionListeners.add(listener);
    }

}
