/* Author: Jeff Dalton
 * Updated: Sun May 26 15:35:26 2002 by Jeff Dalton
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 */

package ix.iface.util;

import java.awt.Color;
import java.awt.Component;
import javax.swing.*;
import javax.swing.border.*;

import java.util.*;

import ix.icore.Priority;
import ix.util.*;

/**
 * A ComboBox that contains items that represent priorities.
 * "Normal" is initially selected.
 *
 * @see ix.icore.Priority
 */
public class PriorityComboBox extends JComboBox {

    public PriorityComboBox() {
	super();
	setRenderer(new PriorityCellRenderer());

	// The values are lowest-first, so we reverse them.
	List values = new ArrayList(Priority.values());
	Collections.reverse(values);
	for (Iterator i = values.iterator(); i.hasNext();) {
	    Priority p = (Priority)i.next();
	    addItem(p);
	}

	setSelectedItem(Priority.NORMAL);

    }

    public Priority getSelectedPriority() {
	return (Priority)getSelectedItem();
    }

    public void setPriority(Priority priority) {
	setSelectedItem(priority);
	setBackground(priority.getColor());
    }

    public static class PriorityCellRenderer extends DefaultListCellRenderer
	          implements ListCellRenderer {

	Map displayNames = new HashMap();
	Map selectedBorders = new HashMap();
	Map unselectedBorders = new HashMap();

	public PriorityCellRenderer() {
	    setOpaque(true);
	}

	public Component getListCellRendererComponent
	        (JList list,
		 Object value,
		 int index,
		 boolean isSelected,
		 boolean cellHasFocus) {
	    Priority priority = (Priority)value;
	    setText(Strings.capitalize(priority.toString()));
	    setBackground(priority.getColor());
	    setForeground(Color.black);
	    if (isSelected) {
		setBorder(getBorder(selectedBorders, priority, true));
	    }
	    else {
		setBorder(getBorder(unselectedBorders, priority, false));
	    }
	    return this;
	}

	String getDisplayName(Priority p) {
	    String name = (String)displayNames.get(p);
	    if (name == null) {
		name = Strings.capitalize(p.toString());
		displayNames.put(p, name);
	    }
	    return name;
	}

	Border getBorder(Map borders, Priority p, boolean isSelected) {
	    Border bord = (Border)borders.get(p);
	    if (bord == null) {
		bord = makeBorder(p, isSelected);
		borders.put(p, bord);
	    }
	    return bord;
	}

	Border makeBorder(Priority p, boolean isSelected) {
	    if (isSelected)
		return BorderFactory.createCompoundBorder
		    (// Outside
		     BorderFactory.createLineBorder
		         (Color.black, 1),
		     // Inside
		     BorderFactory.createMatteBorder
		         (1,4,1,5, p.getColor()));
		    
	    else
		return BorderFactory.createMatteBorder
		    (2,5,2,5, p.getColor());
	}

    }

}
