/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Sep 13 04:24:44 2006 by Jeff Dalton
 * Copyright: (c) 2001, 2003, 2005, AIAI, University of Edinburgh
 */

package ix.iface.util;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

import javax.swing.plaf.metal.MetalLookAndFeel;
import javax.swing.plaf.metal.MetalTheme;
import javax.swing.table.TableColumn;

import java.util.*;

import ix.util.*;

/**
 * Class for static utilities.
 */
public class IFUtil {

    private IFUtil() { }		// no instantiation

    /**
     * Makes any standard I-X changes to the look and feel.
     *
     * @see IXMetalTheme
     */
    public static void adjustLookAndFeel() {

	// Use the I-X Metal theme

	// /\/: Seems we have to do this before setting,
	// or even getting, the look and feel.

	Debug.noteln("Setting Metal look and feel theme.");
        Class ocean =
            Util.classForNameElseNull
                ("javax.swing.plaf.metal.OceanTheme");
        Debug.noteln("Ocean = " + ocean);
        MetalTheme theme = ocean != null
            ? (MetalTheme)
                Util.makeInstance
                 (Util.classForNameElseNull("ix.iface.util.IXOceanTheme"))
            : new IXMetalTheme();
        
        Debug.noteln("Using metal theme", theme);
	MetalLookAndFeel.setCurrentTheme(theme);

	// Explicitly set the look and feel to Metal

	Debug.noteln("Initial look and feel", UIManager.getLookAndFeel());
	String lafClassName =
	    UIManager.getCrossPlatformLookAndFeelClassName();
	Debug.noteln("Using look and feel class", lafClassName);
  	try {
 	    UIManager.setLookAndFeel(lafClassName);
 	}
 	catch (Exception e) {
 	    throw new RethrownException(e);
  	}

	// /\/: For some reason, it has no effect if we set the theme
	// after the Swing objects are created.  Perhaps we have to do:
	//
	// UIManager.setLookAndFeel("javax.swing.plaf.metal.MetalLookAndFeel");
	//
	// to get it to take effect.

    }

    /**
     * Returns the column that has the specified header.
     */
    TableColumn findColumn(JTable tab, String header) {
	for (Enumeration e = tab.getColumnModel().getColumns()
		 ; e.hasMoreElements() ; ) {
	    TableColumn col = (TableColumn)e.nextElement();
	    if (col.getHeaderValue().equals(header))
		return col;
	}
	throw new ConsistencyException("Can't find column", header);
    }

    /**
     * Creates a button with the specified action Listener
     * placed inside a {@link CatchingActionListener}.
     */
    public static JButton makeButton(String text, ActionListener lis) {
	JButton button = new JButton(text);
	button.addActionListener(CatchingActionListener.listener(lis));
	return button;
    }

    /**
     * Creates a button with the specified action Listener
     * placed inside a {@link CatchingActionListener}.
     */
    public static JButton makeButton(String text,
				     boolean enabled,
				     ActionListener lis) {
	JButton button = new JButton(text);
	button.addActionListener(CatchingActionListener.listener(lis));
	button.setEnabled(enabled);
	return button;
    }

    /**
     * Creates a menu-item with the specified action Listener
     * placed inside a {@link CatchingActionListener}.
     */
    public static JMenuItem makeMenuItem(String text, ActionListener lis) {
	JMenuItem item = new JMenuItem(text);
	item.addActionListener(CatchingActionListener.listener(lis));
	return item;
    }

    /**
     * Creates a menu-item, enabled or not, with the specified
     * action Listener placed inside a {@link CatchingActionListener}.
     */
    public static JMenuItem makeMenuItem(String text,
					 boolean enabled,
					 ActionListener lis) {
	JMenuItem item = new JMenuItem(text);
	item.addActionListener(CatchingActionListener.listener(lis));
	item.setEnabled(enabled);
	return item;
    }

    /**
     * Adds an item to a menu in a frame's menu bar.  If the bar
     * does not already contain a menu of the specified name, one
     * is created and added to the bar.  The frame must be a JFrame
     * and already have a JMenuBar.
     *
     * @param frame     the frame
     * @param menuName  the text of a menu in the frame's menu bar.
     * @param item      the menu item to add
     *
     * @see #ensureMenuBarMenu(JFrame frame, String menuName)
     */
    public static void addMenuBarItem(JFrame frame, 
				      String menuName,
				      JMenuItem item) {
	JMenu menu = ensureMenuBarMenu(frame, menuName);
	menu.add(item);
    }

    /**
     * Ensures that a frame's menu bar contains a menu of the
     * specified name, creating a menu and adding it if it is
     * not alreay present.  The menu is returned.  The frame
     * must be a JFrame and already have a JMenuBar.
     *
     * @param frame     the frame
     * @param menuName  the text of a menu in the frame's menu bar.
     */
    public static JMenu ensureMenuBarMenu(JFrame frame, String menuName) {
	JMenuBar bar = frame.getJMenuBar();
	Component[] components = bar.getComponents();
	int i = 0;
	for (; i < components.length; i++) {
	    if (components[i] instanceof JMenu) {
		JMenu menu = (JMenu)components[i];
		if (menu.getText().equals(menuName)) {
		    return menu;
		}
	    }
	    else
		// Stop looking if we see a non-JMenu, because it's
		// probably glue separating the main menu sequence
		// from special ones on the far right.
		break;
	}
	// Have to add a new menu to the bar.
	JMenu m = new JMenu(menuName);
	bar.add(m, i);
	return m;
    }

    /**
     * Returns the named menu from a menu bar if the menu exists.
     *
     * @param bar       the menu bar
     * @param menuName  the text of a menu in the bar
     * @param mustHave  whether to throw an exception or return null
     *                    if the specified menu does not exist.
     */
    public static JMenu getMenuBarMenu(JMenuBar bar,
                                       String menuName,
                                       boolean mustHave) {
	Component[] components = bar.getComponents();
	int i = 0;
	for (; i < components.length; i++) {
	    if (components[i] instanceof JMenu) {
		JMenu menu = (JMenu)components[i];
		if (menu.getText().equals(menuName)) {
		    return menu;
		}
	    }
	    else
		// Stop looking if we see a non-JMenu, because it's
		// probably glue separating the main menu sequence
		// from special ones on the far right.
                break;
	}
        if (mustHave)
            throw new IllegalStateException
                ("Could not find a menu bar menu named " +
                 Strings.quote(menuName));
        else
            return null;
    }

}
