/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Dec 14 19:01:44 2005 by Jeff Dalton
 * Copyright: (c) 2005, AIAI, University of Edinburgh
 */

package ix.iface.util;

import javax.swing.text.StyleConstants;
import javax.swing.text.Document;
import javax.swing.text.Element;

import javax.swing.text.html.HTML;
import javax.swing.text.html.HTMLDocument;

import java.util.*;

import ix.util.*;

/**
 * A utility object that can be used to visit the cells of an HTML table.
 * The walker assumes that the HTML document contains a single table
 * and that the table represents something that looks like the properties
 * of a list of objects.  The walker requires two collections: rowItems and
 * colItems.  The rowItems are the objects whose properties are being
 * shown, and the colItems are the properties.  The leftmost column
 * of the table is used to list the objects (rowItems), while the
 * other columns correspond to properties.  The top row is headers;
 * then there is a row for each object (each rowItem).
 *
 * <p>So at the upper left corner is a "th" that describes the objects
 * in some way.  Maybe it's just "Objects".  The rest of the top row
 * consists of "th" elements that describe the properites, one for
 * each property.  Then there is one row for each rowItem.  The
 * leftmost item on these rows is a "th" describing the object (rowItem)
 * that corresponds to that row; the rest of the row is "td"s
 * that contain descriptions of that object's property values.</p>
 *
 * <p>For example:
 * <table cellspacing="0" border="1" >
 *   <tr><th>Block</th> <th>Size</th>  <th>Colour</th><tr>
 *   <tr><td>A</td>     <td>large</td> <td>red</td>   </tr>
 *   <tr><td>B</td>     <td>small</td> <td>green</td> </tr>
 *   <tr><td>C</td>     <td>tiny</td>  <td>blue</td>  </tr>
 * </table>
 * </p>
 *
 * <p>A subclass will typically redefine the method
 * {@link #walkTD(Element td, Object rowItem, Object colItem)}.</p>
 */
public class HtmlTableWalker {

    protected Collection rowItems;
    protected Collection colItems;

    public HtmlTableWalker(Collection rowItems, Collection colItems) {
	this.rowItems = rowItems;
	this.colItems = colItems;
    }

    public void walkHTML(HTMLDocument doc) {
	// N.B. The objects and properties must match the HTMLDocument.
	Element html = doc.getDefaultRootElement();
	Element body = getChild(html, HTML.Tag.BODY);
	Element table = getChild(body, HTML.Tag.TABLE);
	walkTable(table);
    }

    public void walkTable(Element table) {
	Debug.expectSame(HTML.Tag.TABLE, getTag(table));
	// The 0th row is just headers, so we skip it.
	int rowIndex = 1;
	for (Iterator i = rowItems.iterator(); i.hasNext();) {
	    Object rowItem = i.next();
	    Element row = table.getElement(rowIndex);
	    walkRow(row, rowItem);
	    rowIndex++;
	}
	Debug.expect(rowIndex == table.getElementCount(), "too many rows");
    }

    public void walkRow(Element row, Object rowItem) {
	Debug.expectSame(HTML.Tag.TR, getTag(row));
	// The 0th cell is the TH for the rowItem itself.
	Debug.expectSame(HTML.Tag.TH, getTag(row.getElement(0)));
	int columnIndex = 1;
	for (Iterator i = colItems.iterator(); i.hasNext();) {
	    Object colItem = i.next();
	    Element td = row.getElement(columnIndex);
	    walkTD(td, rowItem, colItem);
	    columnIndex++;
	}
	Debug.expect(columnIndex == row.getElementCount(), "too many columns");
    }

    public void walkTD(Element td, Object rowItem, Object colItem) {
	Debug.expectSame(HTML.Tag.TD, getTag(td));
	;
    }

    protected HTML.Tag getTag(Element elt) {
	return (HTML.Tag)elt.getAttributes()
	                     .getAttribute(StyleConstants.NameAttribute);
    }

    protected Element getChild(Element elt, HTML.Tag tag) {
	HTMLDocument doc = (HTMLDocument)elt.getDocument();
	return doc.getElement(elt, StyleConstants.NameAttribute, tag);
    }

}
