/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Aug 28 17:53:34 2006 by Jeff Dalton
 * Copyright: (c) 2002, 2003, 2006, AIAI, University of Edinburgh
 */

package ix.iface.util;

import javax.swing.JOptionPane;
import java.awt.event.*;
import java.util.*;

import ix.util.*;

/**
 * An ActionListener that catches and reports exceptions.  It can
 * be used as a wrapper around a plain ActionListener or as a base
 * for subclasses that redefine {@link #innerActionPerformed(ActionEvent)}.
 *
 * <p>The static method {@link #listener(ActionListener)} is a
 * convenient way to wrap a listener that might be used more than
 * once.
 */
public class CatchingActionListener implements ActionListener {

    protected ActionListener innerListener;

    /**
     * Constructs a catching listener around the specified
     * inner listener.
     */
    public CatchingActionListener(ActionListener innerListener) {
	this.innerListener = innerListener;
    }

    /**
     * Constructs a catching listener without an inner listener.
     * This should be used only in subclasses that redefine the
     * {@link #innerActionPerformed(ActionEvent)} method or when
     * {@link #setInnerListener(ActionListener)} method will be
     * called.
     */
    protected CatchingActionListener() {
	this(null);
    }

    /**
     * Returns the inner action-listener that is wrapped in this
     * CatchingActionListener.
     */
    public ActionListener getInnerListener() {
	return innerListener;
    }

    /**
     * Sets the inner action-listener that is wrapped in this
     * CatchingActionListener.
     */
    public void setInnerListener(ActionListener innerListener) {
	this.innerListener = innerListener;
    }

    /**
     * Calls {@link #innerActionPerformed(ActionEvent)} on the same
     * event, but inside a <code>try-catch</code> statement that
     * catches any exception or error thrown.  The ActionEvent and the
     * exception or error are then passed to this object's handleException
     * method after a call to <code>Debug.noteException(e)</code>.
     *
     * <p>Note that the innerActionPerformed method define by this
     * calls calls the inner listener's actionPerformed method;
     * but that can be overridden in subclasses.
     *
     * @see ix.util.Debug#noteException(Throwable)
     */
    public void actionPerformed(ActionEvent event) {
	try {
	    innerActionPerformed(event);
	}
	catch (Throwable except) {
	    Debug.noteln("Caught exception during", event);
	    Debug.noteException(except);
	    handleException(event, except);
	}
    }

    /**
     * Called by {@link #actionPerformed(ActionEvent)} inside a
     * catch that will report any exceptions thrown.  This method
     * just calls the inner listener's actionPerformed method,
     * and it should be overridden in subclasses that do not use
     * an inner listener.
     */
    protected void innerActionPerformed(ActionEvent event) {
	Debug.expect(innerListener != null,
		     "No inner listener in " + this);
	innerListener.actionPerformed(event);
    }

    /**
     * Called if the innerActionPerformed method throws an exception
     * or error.  This method puts up an error message dialog that
     * displays the ActionEvent's action command and a description
     * of the exception or error.
     *
     * @see ix.util.Debug#foldException(Throwable)
     */
    public void handleException(ActionEvent event, Throwable except) {
	String command = event.getActionCommand();
	JOptionPane.showMessageDialog
            (Debug.probableMainFrame(),
             new Object[] {"Problem during command " + Strings.quote(command),
                           Debug.foldException(except)},
             "Error during " + command,
             JOptionPane.ERROR_MESSAGE);
    }

    /*
     * Factory.
     */

    static Map listenerCache = new WeakHashMap();

    /**
     * Returns the outer listener, if any, that was recorded for the
     * specified inner listener, or else constructs and records a
     * new CatchingActionListener.  The listeners are recorded in
     a WeakHashMap.
     */
    public static ActionListener listener(ActionListener inner) {
	ActionListener outer = (ActionListener)listenerCache.get(inner);
	if (outer == null) {
	    outer = new CatchingActionListener(inner);
	    listenerCache.put(inner, outer);
	}
	return outer;
    }

    /**
     * Records a specific outer listener for cases where it must
     * be an instance of class other than CatchingActionListener,
     * for instance a subclass.
     */
    public static void recordListener(ActionListener inner,
				      ActionListener outer) {
	listenerCache.put(inner, outer);
    }

}
