/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Aug 31 01:20:38 2004 by Jeff Dalton
 * Copyright: (c) 2001, 2004, AIAI, University of Edinburgh
 */

package ix.iface.domain;

import java.io.IOException;
import java.io.File;

import java.awt.Component;

import javax.swing.*;

import ix.icore.domain.Domain;
import ix.util.*;
import ix.util.xml.*;

/**
 * An object that writes a domain description in textual form.
 */
public abstract class DomainWriter {

    public abstract void writeDomain(Domain domain) throws IOException;

    /**
     * Factory method that returns an appropriate writer for the
     * indicated file, based on the file's type or extension.
     * If the file does not specify a parent directory, the result
     * of calling <code>DomainParser.getLibraryDirectory()</code> is used. <p>
     *
     * Here's how it might be used:
     * <pre>
     *   File domainName = ...;
     *   Domain domain = ...;
     *   ...
     *   try {
     *       DomainWriter.makeWriter(domainName).writeDomain(domain);
     *   }
     *   catch (IOException e) { ... }
     * </pre>
     */
    protected static DomainWriter makeWriter(File file) throws IOException {
	// Factory method based on file type.
	if (file.getParentFile() == null) {
	    File libDir = DomainParser.getLibraryDirectory();
	    Debug.noteln("Resolving " + file + " against " + libDir);
	    file = new File(libDir, file.getPath());
	}
	String name = file.getName();
	FileSyntaxManager fsm = XML.fileSyntaxManager();
	final FileSyntax syntax = fsm.getSyntax(file);
	Debug.noteln("Using file syntax", syntax);
	if (fsm.canWrite(syntax, Domain.class))
	    return new DomainWriterWrapper(file, syntax);
	else
	    throw new IOException("File " + file.getName()
				  + " does not have a known type for domains");
    }

    /**
     * Class used to wrap a FileSyntax as a DomainWriter.
     */
    protected static class DomainWriterWrapper extends DomainWriter {

	protected File domainFile;
	protected FileSyntax syntax;

	protected DomainWriterWrapper(File domainFile, FileSyntax syntax) {
	    this.domainFile = domainFile;
	    this.syntax = syntax;
	}

	public void writeDomain(Domain domain) {
	    try {
		syntax.writeObject(domain, domainFile);
	    }
	    catch (IOException e) {
		Debug.noteException(e);
		throw new RethrownException
		    (e, "Cannot write to " + domainFile +
		     ": " + Debug.describeException(e));
	    }
	}

    }

    /**
     * Writes a description of a domain to a file selected by the user,
     * conducting all necessary dialogs along the way.  It repeatedly
     * asks the user to select a file until either the domain description
     * has been successfully written or the user decices to cancel the
     * operation.  The user is informed of any exceptions thrown while
     * attempting to write, and "success" means that no exceptions were
     * thrown.  If the description has been written to a file, a
     * corresponding File (abstract pathname) is returned; otherwise,
     * the result is <code>null</code>. <p>
     *
     * The output syntax is a function of the file's type as
     * understood by <code>makeWriter(File)</code>.
     *
     * @see #makeWriter(File)
     *
     * @param parentComponent determines the Frame in which dialogs
     *        are displayed.
     * @param domain the Domain to save
     * @return a File if the domain has been written to a file;
     *         otherwise <code>null</code>
     */
    public static File saveDomain(Component parentComponent, Domain domain) {
	File libDir = DomainParser.getLibraryDirectory();
	// Loop until either the domain has successfully been written
	// to a file or else the user has decided to "Cancel" rather
	// that select a(nother) file.
	while (true) {
	    File domainFile = chooseFileToWrite(parentComponent, libDir);
	    if (domainFile == null)
		return null;		// user cancels
	    File result = saveDomain(parentComponent, domain, domainFile);
	    if (result != null)
		return result;		// success!
	}
    }

    /**
     * Writes a description of the domain to the specified file.
     * It is like {@link #saveDomain(Component, Domain)} but without
     * allowing the user to select the file.
     */
    public static File saveDomain(Component frame,
				  Domain domain,
				  File domainName) {
	// Tell the user if the domain looks empty, because some bugs
	// have caused it to be empty when it shouldn't be.
	if (domain.getRefinements() == null 
	    || domain.getRefinements().isEmpty()) {
	    switch(JOptionPane.showConfirmDialog(frame,
		       new String[] {
			   "The domain does not contain any refinements.",
			   "Are you sure you want to save it?"},
		       "Confirm",
		       JOptionPane.YES_NO_OPTION)) {
	    case JOptionPane.YES_OPTION:
		// Continue
		break;
	    case JOptionPane.NO_OPTION:
		// Return now and don't save.
		return null;
	    }
	}
	// See if file already exists.
	if (domainName.exists()) {
	    switch(JOptionPane.showConfirmDialog(frame,
		       "Overwrite " + Strings.quote(domainName.getPath()),
		       "Confirm",
		       JOptionPane.YES_NO_OPTION)) {
	    case JOptionPane.YES_OPTION:
		// Continue
		Util.renameToBackup(domainName);
		break;
	    case JOptionPane.NO_OPTION:
		// Return now and don't save.
		return null;
	    }
	}
	// Write a domain description if we can.
	Debug.noteln("Writing domain to", domainName);
	try {
	    DomainWriter.makeWriter(domainName).writeDomain(domain);
	    return domainName;
	}
	catch (Throwable t) {
	    Debug.noteException(t);
	    JOptionPane.showMessageDialog(frame,
                new Object[] {
                    "Problem saving " + Strings.quote(domainName.getPath()),
		    Debug.foldException(t)},
		"Trouble saving domain",
		JOptionPane.ERROR_MESSAGE);
	    return null;
	}
    }

    private static File chooseFileToWrite(Component frame, File directory) {
	JFileChooser chooser = new XMLSaver(null, Domain.class)
	                           .makeFileChooser();
	int option = chooser.showSaveDialog(frame);
	if (option == JFileChooser.APPROVE_OPTION)
	    return chooser.getSelectedFile();
	else
	    return null;
    }

}

// Issues:
// * We can't use the XMLSaver saveObject method because it returns
//   the object need to return the File that was selected.  That's why
//   this class has its own dialogue-conducting code and is the main
//   reason the class still Exists.
