/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Aug 31 00:29:19 2004 by Jeff Dalton
 * Copyright: (c) 2000 - 2004, AIAI, University of Edinburgh
 */

package ix.iface.domain;

import java.io.FileNotFoundException;
import java.io.File;
import java.net.URL;

import java.awt.Component;

import javax.swing.*;

import ix.icore.domain.Domain;
import ix.util.*;
import ix.util.xml.*;

/**
 * An object that parses a domain description to populate a Domain.
 */
public abstract class DomainParser {

    protected DomainParser() {
    }

    public abstract Domain readDomain();
    public abstract Domain readDomain(Domain dom);

    /**
     * Returns an abstract pathname (File) that refers to the default
     * library of domain descriptions.  This default can be set by
     * the "domain-library" command-line argument or by calling the
     * {@link Parameters#setParameter(String, String)} method.
     * If no such value has been specified, a File constructed
     * from the name "domain-library" is returned as a default.
     *
     * <p>For backwards compatibility, the parameter "domain-library"
     * will also be recognized if no "library-directory" is supplied.
     *
     * @see ix.util.Parameters
     */
    public static File getLibraryDirectory() {
	return 
	    new File(Parameters.getParameter
		     ("library-directory",
		      Parameters.getParameter("domain-library",
					      "domain-library")));
    }

    /**
     * Factory method that returns an appropriate parser for the
     * indicated file based on the file name's type or extension.
     *
     * Here's how it might be used:
     * <pre>
     *   File domainName = ...;
     *   Domain domain = ...;
     *   try {
     *       DomainParser.makeParser(domainName).readDomain(domain);
     *   }
     *   catch (FileNotFoundException e) { ... }
     * </pre>
     */
    protected static DomainParser makeParser(File domainFile)
	      throws FileNotFoundException {
	// Factory method.
	// Based on file type.
	if (!domainFile.exists())
	    throw new FileNotFoundException
		("Can't find " + domainFile);
	FileSyntaxManager fsm = XML.fileSyntaxManager();
	final FileSyntax parser = fsm.getSyntax(domainFile);
	Debug.noteln("Using file syntax", parser);
	if (fsm.canRead(parser, Domain.class))
	    return new DomainParserWrapper(domainFile, parser);
	else
	    throw new IllegalArgumentException
		("Do not know how to parse " + domainFile);
    }

    /**
     * Class used to wrap a FileSyntax as a DomainParser.
     */
    protected static class DomainParserWrapper extends DomainParser {

	protected File domainFile;
	protected FileSyntax parser;

	protected DomainParserWrapper(File domainFile, FileSyntax parser) {
	    this.domainFile = domainFile;
	    this.parser = parser;
	}

	public Domain readDomain() {
	    return readDomain(new Domain());
	}

	public Domain readDomain(Domain domain) {
	    Object temp;
	    try {
		temp = parser.readObject(domainFile.toURL());
	    }
	    catch (Exception e) {
		Debug.noteException(e);
		throw new RethrownException
		    (e, "Cannot load from " + domainFile +
		     ": " + Debug.describeException(e));
	    }
	    if (!(temp instanceof Domain))
		throw new ClassCastException
		    ("File " + domainFile +
		     " contains a " + XML.nameForClass(temp.getClass()) +
		     " not a domain.");
	    domain.takeFrom((Domain)temp);
	    return domain;
	}

    }

    /**
     * Reads a description of a domain from a file selected by the user,
     * conducting all necessary dialogs along the way.  It repeatedly
     * asks the user to select a file until either the domain description
     * has been successfully read or the user decides to cancel the
     * operation.  The user is informed of any exceptions thrown while
     * attempting to read, and "success" means that no exceptions were
     * thrown.  If the description has been read from a file, a
     * corresponding File (abstract pathname) is returned; otherwise,
     * the result is <code>null</code>. <p>
     *
     * The input syntax is a function of the file's type as
     * understood by <code>makeParser(File)</code>. <p>
     *
     * Note that this method modifies an existing Domain object
     * rather than creating a new one.  However, no modifications
     * occur until after the file has been completely processed
     * without an exception being thrown.
     *
     * @see #makeParser(File)
     *
     * @param parentComponent determines the Frame in which dialogs
     *        are displayed.
     * @param domain the Domain object to modify
     * @return a File if the domain description has been read successfully,
     *         otherwise <code>null</code>
     */
    public static File loadDomain(Component parentComponent, Domain domain) {
	File libDir = DomainParser.getLibraryDirectory();
	while (true) {
	    File domainFile = chooseDomain(parentComponent, libDir);
	    if (domainFile == null)
		return null;		// user cancels
	    File result = loadDomain(parentComponent, domain, domainFile);
	    if (result != null)
		return result;		// success!
	}
    }

    private static File loadDomain(Component frame,
				   Domain domain,
				   File domainName) {
	try {
	    Domain tempDomain = new Domain();
	    DomainParser.makeParser(domainName).readDomain(tempDomain);
	    if (tempDomain.getName() != null)
		Debug.noteln("Domain has name", tempDomain.getName());
	    domain.takeFrom(tempDomain);
	    return domainName;
	}
	catch (FileNotFoundException e) {
	    JOptionPane.showMessageDialog(frame,
		"Can't find file " + Util.quote(domainName.getPath()),
		"File not found",
		JOptionPane.ERROR_MESSAGE);
	    return null;
	}
	catch (Throwable t) {
	    Debug.noteException(t);
	    JOptionPane.showMessageDialog(frame,
		new Object[] {
		    "Exception while trying to parse "
			+ Util.quote(domainName.getPath()),
			Debug.foldException(t)},
		"Exception in Domain-Parser",
		JOptionPane.ERROR_MESSAGE);
	    return null;
	}
    }

    private static File chooseDomain(Component frame, File directory) {
	JFileChooser chooser = new XMLLoader(null, Domain.class)
	                           .makeFileChooser();
	int option = chooser.showOpenDialog(frame);
	if (option == JFileChooser.APPROVE_OPTION)
	    return chooser.getSelectedFile();
	else
	    return null;
    }

}

// Issues:
// * We can't use the XMLLoader loadObject method because it returns
//   the object that was read and we need to return the File that was
//   selected.  That's why this class has its own dialogue-conducting
//   code and indeed why this class still Exists.

