/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Nov 15 20:13:28 2004 by Jeff Dalton
 * Copyright: (c) 2001 - 2004, AIAI, University of Edinburgh
 */

package ix.icore.process;

import java.util.*;

import ix.icore.Variable;
import ix.icore.AbstractAnnotatedObject;
import ix.icore.ConstraintManager;
import ix.icore.process.event.*;
import ix.icore.plan.*;

import ix.icore.domain.Constraint;
import ix.icore.domain.Constrainer;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;
import ix.util.context.ContextAnnotatedObject;

// This class is mostly abstract, but it provides a few "uninteresting"
// utility methods.  /\/

// We still have to determine the right set of public methods.  /\/

public abstract class AbstractPMM extends ContextAnnotatedObject
       implements ProcessModelManager {

    private List processStatusListeners = new LinkedList();

    public AbstractPMM() {
    }

    // The plural add-constraint methods allow subclasses to have
    // only singular versions.

    public void addConstraints(PNode node, List constraints) {
	Debug.noteln("PMM adding constraints for node", node);
	for (Iterator ci = constraints.iterator(); ci.hasNext();) {
	    Constraint c = (Constraint)ci.next();
	    Debug.noteln("--- adding ", c);
	    addConstraint(node, c);
	}
    }

    public void addConstraints(List constraints) {
	Debug.noteln("PMM adding top-level constraints");
	for (Iterator ci = constraints.iterator(); ci.hasNext();) {
	    Constrainer c = (Constrainer)ci.next();
	    Debug.noteln("--- adding ", c);
	    addConstraint(c);
	}
    }

    public void registerConstraintManager
	           (Symbol type, Symbol relation, ConstraintManager cm) {
    }

    public ConstraintManager getConstraintManager(Symbol type, Symbol rel) {
	return null;
    }

    public ConstraintManager getConstraintManager(Constraint c) {
	return null;
    }

    /**
     * Returns a copy of the pattern in which {@link ItemVar}s have
     * been replaced by {@link Variable}s.
     */
    public LList putVariablesInPattern(LList pattern) {
	// When a subnode is introduced by expansion with a Refinement,
	// all ?-vars in its pattern are replaced by values or by
	// Variables when the Refinement is instantiated.  But when
	// a level-0 node is introduced, there is no refinement,
	// and this method is used to perform the replacement of
	// ?-vars by Variables using a global (one per model manager)
	// mapping.
	return
	    (LList)SimpleMatcher.emptyEnv
	        .instantiateTree(pattern, new MakeVarIfUnbound());
    }

    private class MakeVarIfUnbound implements Function1 {
	Map varEnv;
	MakeVarIfUnbound() { this.varEnv = getVarEnv(); }
	public Object funcall(Object name) {
	    Variable v = (Variable)varEnv.get(name);
	    if (v == null) {
		v = new Variable(name);
		varEnv.put(name, v); 		// is that ok?  /\/
	    }
	    return v;
	}
    }

    /*
     * Sentinels
     */

    protected interface Sentinel extends Runnable {
	boolean isReady();
    }

    protected abstract void addSentinel(Sentinel r);

    protected abstract void removeSentinel(Sentinel r);

    protected abstract List getSentinels();

    protected void runSentinels() {
	// /\/: Because the list of sentinels is an LLQueue in
	// Ip2ProcessManager, we can't (yet?) use the iterator's
	// remove() method.
	List removals = null;
	for (Iterator i = getSentinels().iterator(); i.hasNext();) {
	    Sentinel r = (Sentinel)i.next();
	    if (r.isReady()) {
		if (removals == null)
		    removals = new LinkedList();
		removals.add(r);
		r.run();
	    }
	}
	if (removals != null) {
	    getSentinels().removeAll(removals);
	}
    }

    /*
     * Listeners
     */

    public void addProcessStatusListener(ProcessStatusListener listener) {
	Debug.noteln("Adding process status listener", listener);
	processStatusListeners.add(listener);
    }

    // /\/: Never used and not clear what it's for.
    public void fireStatusUpdate() {
	ProcessStatusEvent event = new ProcessStatusEvent(this);
	for (Iterator i = processStatusListeners.iterator(); i.hasNext();) {
	    ProcessStatusListener listener = (ProcessStatusListener)i.next();
	    listener.statusUpdate(event);
	}
    }

    public void fireNewBindings(Map bindings) {
	ProcessStatusEvent event = new ProcessStatusEvent(this);
	for (Iterator i = processStatusListeners.iterator(); i.hasNext();) {
	    ProcessStatusListener listener = (ProcessStatusListener)i.next();
	    listener.newBindings(event, bindings);
	}
    }

    public void fireStateChange(Map delta) {
	ProcessStatusEvent event = new ProcessStatusEvent(this);
	for (Iterator i = processStatusListeners.iterator(); i.hasNext();) {
	    ProcessStatusListener listener = (ProcessStatusListener)i.next();
	    listener.stateChange(event, delta);
	}
    }

    public void fireStateDeletion(Map delta) {
	ProcessStatusEvent event = new ProcessStatusEvent(this);
	for (Iterator i = processStatusListeners.iterator(); i.hasNext();) {
	    ProcessStatusListener listener = (ProcessStatusListener)i.next();
	    listener.stateDeletion(event, delta);
	}
    }

}
