/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Jan 26 19:06:22 2005 by Jeff Dalton
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 */

package ix.icore.domain;

import java.io.Serializable;
import java.util.*;

import ix.icore.Variable;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.MatchEnv;

public class PatternAssignment implements Serializable {

    public static final Symbol S_TRUE = Symbol.intern("true");
    public static final Symbol S_FALSE = Symbol.intern("false");

    protected LList pattern;
    protected Object value;

    public PatternAssignment() {
    }

    public PatternAssignment(LList pattern, Object value) {
	this.pattern = pattern;
	this.setValue(value);
    }

    /** 
     * Construct a PatternAssignment in which the value is the
     * {@link ix.util.lisp.Symbol} named "true".
     */
    public PatternAssignment(LList pattern) {
	this(pattern, S_TRUE);
    }

    public PatternAssignment(Map.Entry e) {
	this((LList)e.getKey(), e.getValue());
    }

    public LList getPattern() {	return pattern; }
    public void setPattern(LList pattern) { this.pattern = pattern; }

    public Object getValue() { return value; }

    public void setValue(Object value) {
	this.value = value instanceof List && !(value instanceof LList)
	    ? LList.LListify((List)value)
	    : value;
    }

    public PatternAssignment instantiate(MatchEnv env, Function1 ifUnbound) {
	LList ipat = (LList)env.instantiateTree(pattern, ifUnbound);
	Object ival = env.instantiateTree(value, ifUnbound);
	return new PatternAssignment(ipat, ival);
    }

    public Set getVariables() {
	// N.B. should not be called until after instantiation. /\/
	Set vars = Variable.varsIn(pattern);
	vars.addAll(Variable.varsIn(value));
	return vars;
    }

    public int hashCode() {
	return 31 * pattern.hashCode() + value.hashCode();
    }

    public boolean equals(Object x) {
	PatternAssignment px;
	return this.getClass() == x.getClass()
	    && this.getPattern()
	           .equals((px = (PatternAssignment)x).getPattern())
	    && this.getValue()
	           .equals(px.getValue());
    }

    public String toString() {
	String p = Lisp.printToString(pattern),
	       v = Lisp.printToString(value);
	return "p=v[" + p + "=" + v + "]";
    }

    public static Map assignmentsToMap(Collection assigns) {
	Map m = new HashMap();
	for (Iterator i = assigns.iterator(); i.hasNext();) {
	    PatternAssignment pv = (PatternAssignment)i.next();
	    m.put(pv.getPattern(),
		  pv.getValue());
	}
	return m;
    }

    public static ListOfPatternAssignment mapToAssignments(Map m) {
	ListOfPatternAssignment assigns = new LinkedListOfPatternAssignment();
	for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    assigns.add(new PatternAssignment(e));
	}
	return assigns;
    }

}
