/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Aug 31 15:04:04 2006 by Jeff Dalton
 * Copyright: (c) 2001, 2002, 2004, 2006, AIAI, University of Edinburgh
 */

package ix.icore;

import java.util.*;

// import ix.icore.domain.Refinement;
import ix.icore.process.PNode;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.context.*;

public class Variable implements Comparable {

    protected Object name;		// must be Comparable

    private ContextValue __value = new ContextValue(null);

    private static LLQueue __allVariables = new LLQueue();

    private static ContextInt __count = new ContextInt(0);

    // Info and debugging fields
    protected int number = __count.incrBefore();
    protected PNode sourceNode = null;

    public Variable(Object name) {
	this.name = name;
	__allVariables.add(this);
    }

    public Object getName() { return name; }
    public void setName(Object name) { this.name = name; }

    public Object getValue() { return __value.get(); }

    public void setValue(Object v) {
	Debug.noteln("Setting value of " + this + " to " + v);
	Debug.expect(getValue() == null,
		     this + " given value " + v +
		     " when it already has value " + getValue());
	Debug.expect(!(v instanceof ItemVar));
	Debug.expect(!(v instanceof Variable));
	this.__value.set(v);
    }

    public PNode getSourceNode() { return sourceNode; }
    public void setSourceNode(PNode n) { this.sourceNode = n; }

    public boolean isBound() { // /\/: New, s.b. used in more places
	return getValue() != null;
    }

    public boolean isGlobal() {
	return sourceNode == null;
    }

    public static List getAllVariables() { 
	return __allVariables.contents(); // unmodifiable
    }

    public int compareTo(Object o) {
	Variable v = (Variable)o;
	int nameLT = ((Comparable)name).compareTo(v.name);
	return (nameLT == 0)
	    ? number - v.number
	    : nameLT;
    }

    public String displayString() {
// 	return "[" + number + ":" + name
// 	    + (getValue() == null ? "" : "=" + Lisp.printToString(getValue()))
// 	    + "]";
        if (getValue() != null)
            return Lisp.printToString(getValue());
        else
            return name + "." + number;
    }

    public String toString() {
	return displayString();
    }

    /* Static utilities */

    // /\/: Don't need all the variations.

    // /\/: Use TreeSets instead of HashSets.

    public static Set varsIn(Object tree) {
	if (tree instanceof Variable) {
	    HashSet result = new HashSet();
	    result.add(tree);
	    return result;
	}
	else if (tree instanceof LList)
	    return varsIn((LList)tree);
	else
	    return new HashSet();
    }

    public static Set varsIn(LList tree) {
	final Set vars = new HashSet();
	tree.walkTree(new Function1() {
	    public Object funcall(Object item) {
		if (item instanceof Variable)
		    vars.add(item);
		return null;
	    }
	});
	return vars;
    }

    public static Set varsAnywhereIn(Object obj) {
	final Set vars = new HashSet();
	new ObjectWalker() {
	    public void walk(Object o) {
		if (o instanceof Variable)
		    vars.add(o);
		else
		    super.walk(o);
	    }
	}.walk(obj);
	return vars;
    }

    public static Set unboundVarsAnywhereIn(Object obj) {
	final Set vars = new HashSet();
	new ObjectWalker() {
	    public void walk(Object o) {
		if (o instanceof Variable && ((Variable)o).getValue() == null)
		    vars.add(o);
		else
		    super.walk(o);
	    }
	}.walk(obj);
	return vars;
    }

    public static Set unboundVarsIn(Collection c) {
	Set vars = new HashSet();
	for (Iterator i = c.iterator(); i.hasNext();) {
	    Variable v = (Variable)i.next();
	    if (v.getValue() == null)
		vars.add(v);
	}
	return vars;
    }

    public static boolean isFullyBound(Object tree) {
	if (tree instanceof Variable)
	    return ((Variable)tree).getValue() != null;
	else if (tree instanceof Cons) {
	    Cons c = (Cons)tree;
	    return isFullyBound(c.car()) && isFullyBound(c.cdr());
	}
	else
	    return true;
    }

    /** 
     * Returns a copy of the tree in which every Variable is replaced
     * by the ItemVar that is its name.
     */
    public static Object revertVars(Object tree) {
	if (tree instanceof Variable) {
	    Variable var = (Variable)tree;
	    return var.getName();
	}
	else if (tree instanceof Cons) {
	    Cons t = (Cons)tree;
	    return new Cons(revertVars(t.car()), (LList)revertVars(t.cdr()));
	}
	else
	    return tree;
    }

    /**
     * Returns a copy of the object in which every Variable is replaced 
     * by the ItemVar that is its name.
     */
    public static Object revertAllVars(Object obj) {
	return new DeepCopier() {
	    public Object copy(Object obj) {
		if (obj instanceof Variable) {
		    Variable var = (Variable)obj;
		    return var.getName();
		}
		else
		    return super.copy(obj);
	    }
	}.copy(obj);
    }

    /**
     * Returns a copy of the tree in which every Variable is replaced 
     * by its value, if it has a value, or else by the ItemVar that is
     * its name.
     */
    public static Object removeVars(Object tree) {
	if (tree instanceof Variable) {
	    Variable var = (Variable)tree;
	    Object val = var.getValue();
	    return val == null ? var.getName() : val;
	}
	else if (tree instanceof Cons) {
	    Cons t = (Cons)tree;
	    return new Cons(removeVars(t.car()), (LList)removeVars(t.cdr()));
	}
	else
	    return tree;
    }

    /**
     * Returns a copy of the object in which every Variable is replaced 
     * by its value, if it has a value, or else by the ItemVar that is
     * its name.
     */
    public static Object removeAllVars(Object obj) {
	return new DeepCopier() {
	    public Object copy(Object obj) {
		if (obj instanceof Variable) {
		    Variable var = (Variable)obj;
		    Object val = var.getValue();
		    return val == null ? var.getName() : val;
		}
		else
		    return super.copy(obj);
	    }
	}.copy(obj);
    }

}
