/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Jan 17 02:54:19 2004 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.util.rdf;

import java.io.*;
import java.lang.reflect.*;
import java.util.*;

import com.hp.hpl.jena.rdf.model.*;
import com.hp.hpl.jena.vocabulary.*;
import com.hp.hpl.jena.ontology.*;
import com.hp.hpl.jena.util.iterator.*;

import ix.util.*;
import ix.util.xml.*;
import ix.util.reflect.*;
import ix.util.lisp.*;

import ix.test.xml.Mapping;	// /\/
import ix.test.xml.MapEntry;	// /\/

/**
 * Produces an OWL Full ontology for the representation used
 * by the standard {@link RdfTranslator}.  The ontology stays
 * close to OWL DL.
 */
public class OwlSyntax extends XMLSyntax {

    String base = Rdf.baseURI;
    String hash = base + "#";

    Vocabulary xsd = new Vocabulary("http://www.w3.org/2001/XMLSchema");
    Morphology morph = new Morphology();

    Class rootClass;
    OntModel om;
    RDFWriter writer;

    private DatatypeProperty _hasValueP; // use hasValueProp() method
    
    public OwlSyntax(Class rootClass) {
	super();		//\/ also gives us an XMLTranslator
	this.rootClass = rootClass;
	this.classSyntax = Rdf.classSyntax();
    }

    public OntModel getOntModel() {
	Debug.expect(om == null, "model already built in", this);

	// OWL Full, in-memory, no reasoner.
	om = ModelFactory.createOntologyModel(OntModelSpec.OWL_MEM, null);
	om.setNsPrefix("ix", hash);

	Ontology ont = om.createOntology(base);
	ont.addProperty(om.createAnnotationProperty(DC.creator.getURI()),
			"I-X");

	writer = makeWriter();

	buildOntModel();
	return om;
    }

    RDFWriter makeWriter() {
	// Having our own writer lets us do any special setup we require.
	RDFWriter w = om.getWriter("RDF/XML-ABBREV");
	w.setProperty("tab", "1"); //\/ for XML.printXMLWithWhitespace.
	w.setProperty("prettyTypes",
	    new Resource[] {
                OWL.Ontology,
                OWL.ObjectProperty,
                OWL.DatatypeProperty,
                OWL.FunctionalProperty,
		OWL.Class,
		OWL.AllDifferent
	    });
	return w;
    }

    void write(OutputStream out) {
	writer.write(om, out, base);
    }

    void buildOntModel() {

	List relevantClasses = relevantClasses(rootClass);
	inheritance = new InheritanceTree(relevantClasses);

	// Object class
	ontClass(Object.class);

	// Define Map
	// typedListOntClass("map", MapEntry.class);
	// addClass(MapEntry.class);
	addClass(MapEntry.class);
	addClass(Mapping.class);

	// Add information from each class relevant to the root.
	for (Iterator i = relevantClasses.iterator(); i.hasNext();) {
	    Class c = (Class)i.next();
	    addClass(c);
	}

    }

    OntClass ontClass(Class javaClass) {
	return ontClass(getClassDescr(javaClass));
    }

    OntClass ontClass(ClassDescr cd) {
	// /\/: Not clear whether we ought to call "get" first
	// or whether it's always safe to call "create".  Might
	// a "create" lose a facet or make "get" ambiguous?
	String uri = hash + cd.getExternalName();
	OntClass c = om.getOntClass(uri);
	if (c == null)
	    c = om.createClass(uri);
	return c;
    }

    List ontClassList(List classes) {
	return (List)Collect.map(classes, new Function1() {
	    public Object funcall(Object c) {
		return ontClass((Class)c);
	    }
	});
    }

    void addClass(Class javaClass) {
	ClassDescr cd = getClassDescr(javaClass);
	if (cd.isStruct()) {
	    if (cd.isAbstract())
		addAbstractStructClass(cd);
	    else
		addStructClass(cd);
	}
	else if (cd.isEnumeration())
	    addEnumerationClass(cd);
	else if (cd.isPrimitive())
	    addPrimitiveClass(cd);
//  	else if (cd.isXML())
//  	    addLiteralDocumentClass(cd);
	else {
	    OntClass c = ontClass(cd);
	    c.addComment("Definition not yet available.", "en");
	}
	// Give every class its name as a label.
	ontClass(cd).addLabel(cd.getExternalName(), null);
    }

    /*
     * Structs
     */

    void addAbstractStructClass(ClassDescr cd) {
	OntClass c = ontClass(cd);
	c.addComment("Abstract", null);
	List javaSubclasses = inheritance.getSubclasses(cd.theClass);
	// Do fun things with the subclasses, if any exist.
	if (javaSubclasses != null) {
	    // Make the class be an owl:equivalentClass to
	    // the union of its subclasses.  /\/: It's not clear
	    // that we should do this rather than allow further
	    // subclasses disjoint from the existing ones.
	    List ontSubclasses = ontClassList(javaSubclasses);
	    RDFList ontSubs = om.createList(ontSubclasses.iterator());
	    UnionClass union = om.createUnionClass(null, ontSubs);
	    c.addProperty(OWL.equivalentClass, union);
	    // Make the subclasses owl:disjointWith each other.
	    Collect.walkTriangle(ontSubclasses, new Proc() {
		public void call(Object arg1, Object arg2) {
		    OntClass c_i = (OntClass)arg1;
		    OntClass c_j = (OntClass)arg2;
		    c_i.addProperty(OWL.disjointWith, c_j);
		}
	    });
	}
    }

    void addStructClass(ClassDescr cd) {
	OntClass c = ontClass(cd);
	List fields = cd.getFieldDescrs();
	for (Iterator fi = fields.iterator(); fi.hasNext();) {
	    FieldDescr fd = (FieldDescr)fi.next();
	    addField(c, fd);
	}
	// Superclass, if any.
	Class sup = inheritance.getSuperclass(cd.theClass);
	if (sup != null)
	    c.addSuperClass(ontClass(sup));
    }

    void addField(OntClass c, FieldDescr fd) {
	ClassDescr valueDescr = fd.getTypeDescr();
	String extName = fd.getExternalName();
	addField(c, extName, valueDescr);
    }

    void addField(OntClass c, String extFieldName, ClassDescr valueDescr) {
	if (valueDescr.isTypedList()) {
	    // A list of instances of some element class
	    String fieldName = morph.singular(extFieldName);
	    String propName = hash + "has-" + fieldName;
	    ObjectProperty p = om.createObjectProperty(propName);
	    ClassDescr eltDescr = valueDescr.getEltType();
	    // All values of the property must be instances of
	    // the element class.
	    AllValuesFromRestriction avfr =
		om.createAllValuesFromRestriction
		       (null, p, fieldClass(eltDescr));
	    c.addSuperClass(avfr);
	    // No cardinality restriction
	}
	else if (valueDescr.isMap()) {
	    // The field-value is a Map.
	    // Pretend it's a Mapping.
	    addField(c, extFieldName, getClassDescr(Mapping.class));
	}
	else {
	    // An ordinary, object-valued field
	    String fieldName = extFieldName;
	    String propName = hash + "has-" + fieldName;
	    ObjectProperty p = om.createObjectProperty(propName);
	    // All values must be instances of the field-value class.
	    AllValuesFromRestriction avfr =
		om.createAllValuesFromRestriction
		       (null, p, fieldClass(valueDescr));
	    c.addSuperClass(avfr);
	    // At most one value.
	    MaxCardinalityRestriction mcr =
		om.createMaxCardinalityRestriction(null, p, 1);
	    c.addSuperClass(mcr);
	}
    }

    Resource fieldClass(ClassDescr valueCD) {
	if (valueCD.isList()) {
	    ClassDescr eltCD = valueCD.getEltType();
	    if (eltCD == null)
		return RDF.List;
	    else
		// This won't work for lists of lists? /\/
		return ontClass(eltCD);
	}
	else {
	    return ontClass(valueCD);
	}
    }

    /*
     * Typed lists
     */

    OntClass typedListOntClass(Class javaEltClass) {
	ClassDescr ecd = getClassDescr(javaEltClass);
	String name = "list-of-" + ecd.getExternalName();
	return typedListOntClass(name, javaEltClass);
    }

    OntClass typedListOntClass(String ontClassName, Class javaEltClass) {
	String uri = hash + ontClassName;
	OntClass c = om.getOntClass(uri);
	if (c != null)
	    return c;
	c = om.createClass(uri);
	c.addSuperClass(RDF.List);
	OntClass eltClass = ontClass(javaEltClass);
	c.addSuperClass
	    (om.createAllValuesFromRestriction(null, RDF.first, eltClass));
	c.addSuperClass
	    (om.createAllValuesFromRestriction(null, RDF.rest, c));
	return c;
    }

    /*
     * Enumerations
     */

    void addEnumerationClass(final ClassDescr cd) {
	final EnumeratedClass enumClass =
	    ontClass(cd).convertToEnumeratedClass
	        (om.createList()); // no members yet
	Iterator vi = getEnumerationValues(cd.theClass).iterator();
	Map1 m = new Map1() {
	    public Object map1(Object v) {
		Debug.expect(v instanceof EnumeratedValue);
		String name = v.toString();
		String cbase = base + cd.getExternalName();
		return om.createIndividual(cbase + "#" + v, enumClass);
	    }
	};
	Map1Iterator mi = new Map1Iterator(m, vi);
	RDFList members = om.createList(mi);
	enumClass.setOneOf(members);
	om.createAllDifferent
	    (om.createList(members.iterator())); // separate RDFList
    }

    /*
     * Primitive types
     */

    void addPrimitiveClass(ClassDescr cd) {
	OntClass c = ontClass(cd);
	String xsdType = classSyntax.xmlSchemaDatatype(cd.theClass);
	AllValuesFromRestriction res =
	    om.createAllValuesFromRestriction
	        (null, hasValueProp(), xsd.resource(xsdType));
	c.addSuperClass(res);
    }

    DatatypeProperty hasValueProp() { // construct when needed
	if (_hasValueP == null)
	    _hasValueP =
		om.createDatatypeProperty(hash + "has-litvalue",
					  true); // functional
	return _hasValueP;
    }

    /**
     * Outputs an OWL ontology for I-X domains or for the class specified
     * by the "root" parameter.  
     *
     * @see ix.util.Parameters#getParameter(String)
     */
    public static void main(String[] argv) {
	Debug.off();
	Parameters.processCommandLineArguments(argv);

	Class root = Rdf.classSyntax().
	    classForExternalName(Parameters.getParameter("root", "domain"));
	OwlSyntax syntax = new OwlSyntax(root);
	OntModel m = syntax.getOntModel();

	// syntax.write(System.out);
	OutputStream out = new ByteArrayOutputStream();
	syntax.write(out);
	XML.printXMLWithWhitespace(out.toString(), 2);

    }

}

// Issues:
//
// * What should the default root class be?  Probably Object, but
//   all the other syntax classes still use Plan, for historical
//   reasons.
//
// * Using "#" in a class URI turns out to be what makes the class
//   definitions appear in RDF/XML with rdf:ID instead of rdf:about.
//
// * The properties that correspond to fields have to remain quite
//   generic at the top level, at least if we want to keep close
//   to OWL DL, because they might have very different value types
//   in different classes.  So we can't make them datatype properties,
//   even of that would suit some classes, beause they might have
//   to be object properties in other classes (and those two property
//   types have to be disjoint in OWL DL).  But that also means we
//   can't use literals directly as field values; the values always
//   have to be instances of classes so that we can use object properties.
//
// * We use separate copies of the members RDFList when calling
//   enumClass.setOneOf and om.createAllDifferent to avoid seeing
//   output containing constructs such as
//     <owl:AllDifferent>
//       <owl:distinctMembers rdf:nodeID="A0"/>
//     </owl:AllDifferent>
