/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Jan 17 02:53:17 2004 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.util.rdf;

import java.util.*;

import com.hp.hpl.jena.rdf.model.*;
import com.hp.hpl.jena.vocabulary.*;
import com.hp.hpl.jena.datatypes.xsd.XSDDatatype;

import ix.util.*;
import ix.util.xml.*;
import ix.util.reflect.*;

import ix.test.xml.Mapping;	// /\/
import ix.test.xml.MapEntry;	// /\/

/**
 * Converts an object to a Jena RDF model.
 */
class ObjectModel {

    Vocabulary ix = new Vocabulary(Rdf.baseURI);

    ClassSyntax classSyntax = Rdf.classSyntax();

    RdfTranslator translator;
    Object root;
    Model model;

    ObjectModel(RdfTranslator translator, Object root) {
	this.translator = translator;
	this.root = root;
    }

    Model getModel() {
	Debug.expect(model == null, "model already built in", this);
	model = makeModel();
	objectToResource(root);
	return model;
    }

    Model makeModel() {
	Model m = ModelFactory.createDefaultModel();
	m.setNsPrefix("ix", ix.getHashURI());
	return m;
    }

    Resource objectToResource(Object obj) {
	if (obj == null)
	    return nullToResource();
	else if (obj instanceof String)
	    return stringToResource((String)obj);
	else {
	    ClassDescr cd = classSyntax.getClassDescr(obj.getClass());
	    if (cd.isPrimitive())
		return primitiveToResource(obj);
	    else if (cd.isList())
		return listToResource((List)obj);
//  	    else if (cd.isSet())
//  		return setToResource((Set)obj);
	    else if (cd.isMap())
		return mapToResource((Map)obj);
	    else if (cd.isStruct())
		return structToResource(obj);
//  	    else if (cd.isXML())
//  		return embeddedXMLToResource(obj);
	    else
		throw new IllegalArgumentException("Unknown type " + cd);
	}
    }

    Resource nullToResource() {
	Resource r = model.createResource();
	r.addProperty(RDF.type, ix.resource("null"));
	return r;
    }

    Resource stringToResource(String str) {
	return primitiveToResource(str);
    }

    Resource primitiveToResource(Object obj) {
	Class c = obj.getClass();
	ClassDescr cd = classSyntax.getClassDescr(c);
	String cname = cd.getExternalName();
	Resource r;
	if (obj instanceof EnumeratedValue) {
	    String cbase = ix.getBaseURI() + cd.getExternalName();
	    r = model.createResource(cbase + "#" + obj);
	}
	else {
	    r = model.createResource();
	    r.addProperty(ix.property("litvalue"),
			  primitiveToLiteral(obj));
	}
	r.addProperty(RDF.type, ix.resource(cname));
	return r;
    }

    Literal primitiveToLiteral(Object obj) {
	// /\/: We could let Jena perform the toString conversion
	// for classes that correspond to XML Schema datatypes.
	Class c = obj.getClass();
	Stringer s = classSyntax.getStringer(c);
	String text = s != null ? s.toString(obj) : obj.toString();
	String xsdType = classSyntax.xmlSchemaDatatype(c);
	XSDDatatype dtype = new XSDDatatype(xsdType);
	// N.B. The Jena javadoc for the Model class says the 
	// createTypedLiteral call will: "Build a typed literal
        // from its lexical form. The lexical form will be parsed
	// now and the value stored. If the form is not legal
	// this will throw an exception." /\/
	return model.createTypedLiteral(text, dtype);
    }

    Resource listToResource(List obj) {
	List resources = new ArrayList(obj.size());
	for (Iterator i = obj.iterator(); i.hasNext();) {
	    resources.add(objectToResource(i.next()));
	}
	RDFList list = model.createList(resources.iterator());
	Debug.expect(list.isValid(), "Invalid list",
		     list.getValidityErrorMessage());
	return list;
//  	com.hp.hpl.jena.rdf.model.Seq s = model.createSeq();
//  	for (Iterator i = obj.iterator(); i.hasNext();) {
//  	    s.add(objectToResource(i.next()));
//  	}
//  	return s;
    }

    Resource mapToResource(Map map) {
	// /\/: Used to return mapToRDFList(map);
	return objectToResource(new Mapping(map));
    }

    RDFList mapToRDFList(Map map) { 			// unused /\/
	return mapToRDFList(map.entrySet().iterator());
    }

    private RDFList mapToRDFList(Iterator entries) { 	// unused /\/
	if (entries.hasNext()) {
	    Map.Entry e = (Map.Entry)entries.next();
	    Resource car = objectToResource(new MapEntry(e));
	    RDFList cdr = (RDFList)mapToRDFList(entries);
	    RDFList cons = cdr.cons(car);
	    cons.addProperty(RDF.type, ix.resource("map"));
	    return cons;
	}
	else
	    return model.createList();
    }

    Resource structToResource(Object obj) {
	Class c = obj.getClass();
	ClassDescr cd = classSyntax.getClassDescr(c);
	String cname = cd.getExternalName();
	List fields = cd.getFieldDescrs();
	Resource r = model.createResource();
	r.addProperty(RDF.type, ix.resource(cname));
	for (Iterator fi = fields.iterator(); fi.hasNext();) {
	    FieldDescr fd = (FieldDescr)fi.next();
	    visitField(obj, fd, r);
	}
	return r;
    }

    void visitField(Object obj, FieldDescr fd, Resource r) {
	Object value = getFieldValue(obj, fd);
	if (value == null) {
	    return;		// omit null-valued fields
	}
	ClassDescr valueDescr = fd.getTypeDescr();
	String fieldName = fd.getExternalName();
	/*	
        if (valueDescr.isPrimitive() && !(value instanceof String)) {
	    // /\/ Not clear Strings shouldn't be included.
	    r.addProperty(ix.property(fieldName),
			  model.createLiteral(primitiveToLiteral(value)));
	}
	else */ if (valueDescr.isTypedList()) {
	    Property fieldProp = ix.singular(fieldName);
	    for (Iterator i = ((List)value).iterator(); i.hasNext();) {
		r.addProperty(fieldProp,
			      objectToResource(i.next()));
	    }
	}
	else {
	    r.addProperty(ix.property(fieldName),
			  objectToResource(value));
	}
    }

    Object getFieldValue(Object obj, FieldDescr fd) {
	try {
	    return fd.getValue(obj);
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new XMLException // s.b. different class /\/
		("Cannot get value of field " + fd.getName()
		 + " of class " + obj.getClass().getName(),
		 e);
	}
    }

}
