/*
 * Created on Dec 27, 2003
 *
 */
// package org.mindswap.owls.io.impl;

package ix.util.owls;
import org.mindswap.owls.io.impl.*;

import ix.util.*;                                  // for debugging output /\/
import com.hp.hpl.jena.rdf.model.ResourceFactory;  // for fixing a bug /\/

import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.mindswap.owl.Util;
import org.mindswap.owls.OWLSFactory;
import org.mindswap.owls.grounding.AtomicGrounding;
import org.mindswap.owls.grounding.Grounding;
import org.mindswap.owls.grounding.MessageMap;
import org.mindswap.owls.grounding.MessageMapList;
import org.mindswap.owls.grounding.UPnPAtomicGrounding;
import org.mindswap.owls.grounding.WSDLAtomicGrounding;
import org.mindswap.owls.process.AtomicProcess;
import org.mindswap.owls.process.Choice;
import org.mindswap.owls.process.CompositeProcess;
import org.mindswap.owls.process.Condition;
import org.mindswap.owls.process.ConditionList;
import org.mindswap.owls.process.ControlConstruct;
import org.mindswap.owls.process.DataFlow;
import org.mindswap.owls.process.DataFlowElement;
import org.mindswap.owls.process.Effect;
import org.mindswap.owls.process.EffectList;
import org.mindswap.owls.process.Parameter;
import org.mindswap.owls.process.Process;
import org.mindswap.owls.process.ProcessComponent;
import org.mindswap.owls.process.ProcessComponentList;
import org.mindswap.owls.process.ProcessModel;
import org.mindswap.owls.process.Sequence;
import org.mindswap.owls.process.Unordered;
import org.mindswap.owls.process.ValueMap;
import org.mindswap.owls.profile.Profile;
import org.mindswap.owls.service.Service;
import org.mindswap.owls.vocabulary.jena.FLAServiceOnt;
import org.mindswap.owls.vocabulary.jena.OWLS_1_0;

import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.rdf.model.StmtIterator;
import com.hp.hpl.jena.vocabulary.RDF;
import com.hp.hpl.jena.vocabulary.RDFS;

/**
 * @author Evren Sirin
 *
 */
public class ServiceReader extends OWLSReaderImpl {
	public static boolean DEBUG = false;
	
	/**
	 * 
	 */
	public ServiceReader() {
		version = "1.0";
	}
	
	protected List createService(URI fileURI, Model model) {
		StmtIterator si = model.listStatements(null, RDF.type, OWLS_1_0.Service.Service);
		List list = new ArrayList();
		
		while(si.hasNext()) {
			Resource serviceInfo = si.nextStatement().getSubject();		
			try {
				startService(serviceInfo.toString());				
				Resource profileInfo = serviceInfo.getProperty(OWLS_1_0.Service.presents).getResource();
				Resource processModelInfo = serviceInfo.getProperty(OWLS_1_0.Service.describedBy).getResource();
				// Resource groundingInfo = serviceInfo.getProperty(OWLS_1_0.Service.supports).getResource();
				
				Service service = OWLSFactory.createService(serviceInfo);
				service.setFileURI(fileURI);
				service.setOWLSVersion(version);
				
				createProcessModel(service, processModelInfo);
				createProfile(service, profileInfo);		
				// createGrounding(service, groundingInfo);
				
				if(!errorOccurred)
					list.add(service);
				
				finishService(serviceInfo.toString());				
			} catch (Exception e) {
				error("Invalid service description " + serviceInfo);
				e.printStackTrace();
			}
		}
		
		return list;		
	}

	private void createProcessModel(Service service, Resource processModelInfo) {	
		Resource processInfo = processModelInfo.getProperty(OWLS_1_0.Process.hasProcess).getResource();
		
		ProcessModel processModel = OWLSFactory.createProcessModel(processModelInfo);
		Process process = createProcess(service, processInfo);
		
		processModel.setProcess(process);
		processModel.setService(service);	
	}

	private boolean isProcess(Resource processInfo) {
		return processInfo.hasProperty(RDF.type, OWLS_1_0.Process.AtomicProcess) ||
			   processInfo.hasProperty(RDF.type, OWLS_1_0.Process.CompositeProcess) ||
			   processInfo.hasProperty(RDF.type, OWLS_1_0.Process.SimpleProcess);
	}
		
	private Process createProcess(Service service, Resource processInfo) {	
		if(!isProcess(processInfo))
			throw new RuntimeException("The process is neither atomic nor composite: " + processInfo.getURI());
			
		Process process = null;	
		if(processInfo.hasProperty(RDF.type, OWLS_1_0.Process.AtomicProcess))
			process = createAtomicProcess(service, processInfo);
		if(processInfo.hasProperty(RDF.type, OWLS_1_0.Process.CompositeProcess))
			process = createCompositeProcess(service, processInfo);
	
		if(process == null)
			return null;
		
		createProcessParams(process, true, processInfo);		
		createProcessParams(process, false, processInfo);
		
		createConditions(process.getPreconditions(), processInfo.listProperties(OWLS_1_0.Profile.hasPrecondition));
		createEffects(process.getEffects(), processInfo.listProperties(OWLS_1_0.Profile.hasEffect));			
		
		createDefaultValues(process.getDefaultValues(), processInfo);
		createDataFlow(process, processInfo);	
				
		process.setService(service);
		
		return process;
	}
	
	private AtomicProcess createAtomicProcess(Service service, Resource processInfo) {	
		AtomicProcess process =	OWLSFactory.createAtomicProcess(processInfo);
		
		return process;
	}

	private CompositeProcess createCompositeProcess(Service service, Resource processInfo) {
		Statement stmt = processInfo.getProperty(OWLS_1_0.Process.composedOf);
		
		if(stmt == null) {
			error("Cannot find the components for composite process " + 
					"\n      process " + processInfo);	
			return null;
		}
		
		Resource composedInfo = stmt.getResource();
		
		ControlConstruct controlConstruct = createControlConstruct(service, composedInfo);
		
		CompositeProcess process = OWLSFactory.createCompositeProcess(processInfo);
		process.setComposedOf(controlConstruct);
		
		return process;   
	}	
	
	private ProcessComponent createProcessComponent(Service service, Resource processComponentInfo) {
		ProcessComponent processComponent = null;
		
		if(isProcess(processComponentInfo))
			processComponent = createProcess(service, processComponentInfo);
		else
			processComponent = createControlConstruct(service, processComponentInfo);

		return processComponent;
	}	
		
	private ControlConstruct createControlConstruct(Service service, Resource controlConstructInfo) {
		Resource ccType = controlConstructInfo.getProperty(RDF.type).getResource();
		Resource componentsInfo = controlConstructInfo.getProperty(OWLS_1_0.Process.components).getResource();
		
		ControlConstruct cc = null;
		if(ccType.equals(OWLS_1_0.Process.Sequence))
			cc = createSequence(service, componentsInfo);
		else if(ccType.equals(OWLS_1_0.Process.Choice))
			cc = createChoice(service, componentsInfo);
		else if(ccType.equals(OWLS_1_0.Process.Unordered))
			cc = createUnordered(service, componentsInfo);
		else
			error("Don't know how to read the control construct " + ccType);
		
		return cc;
	}	
	
	private Sequence createSequence(Service service, Resource sequenceInfo) {
		Sequence sequence = OWLSFactory.createSequence(sequenceInfo);

		createComponents(service, sequence, sequenceInfo);		
		
		return sequence;
	}	
	
	private Unordered createUnordered(Service service, Resource unorderedInfo) {
		Unordered unordered = OWLSFactory.createUnordered(unorderedInfo);

		createComponents(service,unordered, unorderedInfo);		
		
		return unordered;
	}		

	private Choice createChoice(Service service, Resource choiceInfo) {
		Choice choice = OWLSFactory.createChoice(choiceInfo);

		createComponents(service, choice, choiceInfo);		
		
		return choice;
	}
		
	private void createComponents(Service service, ControlConstruct cc, Resource ccInfo) {		
		List list = Util.createList(ccInfo);
		for(int i = 0; i < list.size(); i++) {
			Resource processComponentInfo = (Resource) list.get(i);
			ProcessComponent processComponent = createProcessComponent(service, processComponentInfo);
			
			if(processComponent == null)
				error("Invalid ProcessComponent description " + processComponentInfo);
			else
				cc.getComponents().add(processComponent);
		}
	}	

	
	private void createDefaultValues(ValueMap defaultValues, Resource processComponentInfo) {
	}

        // /\/: Fix bug in data-flow processing: it was looking for "theProperty" instead of "theParameter".
        private Property OWLS_1_0_Process_theParameter =
	    ResourceFactory.createProperty("http://www.daml.org/services/owl-s/1.0/Process.owl#theParameter");
        // /\/: and for "atClass" instead of "atProcess".
        private Property OWLS_1_0_Process_atProcess =
	    ResourceFactory.createProperty("http://www.daml.org/services/owl-s/1.0/Process.owl#atProcess");

	private void createDataFlow(Process process, Resource processComponentInfo) {
		try {	
			DataFlow dataFlow = process.getDataFlow();
			StmtIterator i = processComponentInfo.listProperties(OWLS_1_0.Process.sameValues);
			while(i.hasNext()) {
				Resource sameValuesList = i.nextStatement().getResource();
				Resource value1 = sameValuesList.getProperty(RDF.first).getResource();
				Resource value2 = sameValuesList.getProperty(RDF.rest).getResource().getProperty(RDF.first).getResource();
	
				Statement stmt;
				
				Resource processName1, paramName1, processName2, paramName2;
				
				// /\/ wrong: stmt = value1.getProperty(OWLS_1_0.Process.theProperty);
				stmt = value1.getProperty(OWLS_1_0_Process_theParameter);
				if(stmt.getObject() instanceof Resource)
					paramName1 = stmt.getResource();
				else
					paramName1 = Util.toResource(stmt.getObject().toString());
	
				// /\/ wrong: stmt = value1.getProperty(OWLS_1_0.Process.atClass);
				stmt = value1.getProperty(OWLS_1_0_Process_atProcess);
				if(stmt.getObject() instanceof Resource)
					processName1 = stmt.getResource();
				else
					processName1 = Util.toResource(stmt.getObject().toString());
				
				// /\/ wrong: stmt = value2.getProperty(OWLS_1_0.Process.theProperty);
				stmt = value2.getProperty(OWLS_1_0_Process_theParameter);
				if(stmt.getObject() instanceof Resource)
					paramName2 = stmt.getResource();
				else
					paramName2 = Util.toResource(stmt.getObject().toString());
	
				// /\/ wrong: stmt = value2.getProperty(OWLS_1_0.Process.atClass);
				stmt = value2.getProperty(OWLS_1_0_Process_atProcess);
				if(stmt.getObject() instanceof Resource)
					processName2 = stmt.getResource();
				else
					processName2 = Util.toResource(stmt.getObject().toString());
							
				Process p1 = findProcess(process, Util.toURI(processName1));
				Process p2 = findProcess(process, Util.toURI(processName2));
				Parameter param1 = null, param2 = null;
				
				if(p1 == null) {
					error("Cannot find the process data flow refers to " + 
							"\n      process " + processName1 +
							"\n    parameter " + paramName1 +
							"\n data flow in " + process.getURI());
				}
				else {
					param1 = p1.getParameter(Util.toURI(paramName1));
					if(param1 == null) {
						error("Cannot find the parameter data flow refers to " + 
								"\n      process " + processName1 +
								"\n    parameter " + paramName1 +
								"\n data flow in " + process.getURI());					
					}
				}
	
				if(p2 == null) {
					error("Cannot find the process data flow refers to " + 
							"\n      process " + processName2 +
							"\n    parameter " + paramName2 +
							"\n data flow in " + process.getURI());
				}
				else {
					param2 = p2.getParameter(Util.toURI(paramName2));
					if(param2 == null) {
						error("Cannot find the parameter data flow refers to " + 
								"\n      process " + processName2 +
								"\n    parameter " + paramName2 +
								"\n data flow in " + process.getURI());					
					}
				}
				
				if(param1 == null || param2 == null)
					continue;
				
				DataFlowElement dfe = OWLSFactory.createDataFlowElement();
				dfe.add(param1);
				dfe.add(param2);
				
				dataFlow.add(dfe);
			}
		} catch(Exception e) {
		        Debug.noteException(e);
			error("Invalid data flow specification");
		}
	}	

	private void createProcessParams(Process process, boolean isInput, Resource processInfo) {	
		Property prop = isInput ? OWLS_1_0.Process.hasInput : OWLS_1_0.Process.hasOutput;		
		StmtIterator i = processInfo.listProperties(prop);		
		while(i.hasNext()) {
			Resource p = i.nextStatement().getResource();

			Parameter param = null;
			if(isInput) {
				param = OWLSFactory.createInput(p);
				process.getInputs().add(param);
			}
			else {
				param = OWLSFactory.createOutput(p);
				process.getOutputs().add(param);
			}
				
			param.setProcess(process);	
			
			if(p.hasProperty(OWLS_1_0.Process.parameterType))
				param.setType(OWLSFactory.createOWLResource(p.getProperty(OWLS_1_0.Process.parameterType).getResource()));
			else
				error("Cannot find the type for the process parameter " + 
						"\n      process " + process +
						"\n    parameter " + param);
			
			
//			TODO default values
//			StmtIterator j = processInfo.listProperties(RDFS.subClassOf);
//			while(j.hasNext()) {
//				Resource r = j.nextStatement().getResource();
//				if(r.hasProperty(OWL.onProperty, p) && r.hasProperty(OWL.hasValue)) {
//					Object value = null;
//					RDFNode node = r.getProperty(OWL.hasValue).getObject();
//					if(node instanceof Literal)
//						value = ((Literal) node).getValue();
//					else 
//						value = ((Resource) node).getURI();
//					process.getDefaultValues().setValue(param, value);
//					break;
//				}
//			}
			
								
			if(DEBUG) {
				System.out.println("  Process   " + process.getURI() + "\n" + 
								   (isInput ? "  Input     ":"  Output    " ) +
								   param.getLabel() + "\n" + 
								   "  Type      " + param.getType() + "\n");
			}			
		}
	}	
	
	private void createProfile(Service service, Resource profileInfo) {
		try {
			Profile profile = OWLSFactory.createProfile(profileInfo);
			Process process = service.getProcess();
			
			createProfileParams(profile, process, true, profileInfo);
			createProfileParams(profile, process, false, profileInfo);
			
			createConditions(profile.getPreconditions(), profileInfo.listProperties(OWLS_1_0.Profile.hasPrecondition));
			createEffects(profile.getEffects(), profileInfo.listProperties(OWLS_1_0.Profile.hasEffect));			
			
			profile.setType(Util.toURI(profileInfo.getProperty(RDF.type).getResource()));		
			copyPropertyValues(profileInfo, OWLS_1_0.Profile.serviceName, RDFS.label);
			
			profile.setService(service);
		} catch (RuntimeException e) {
			error("Invalid profile description");
			e.printStackTrace();
		}
	}
	
	private void createProfileParams(Profile profile, Process process, boolean isInput, Resource profileInfo) {
		Property prop = isInput ? OWLS_1_0.Profile.hasInput : OWLS_1_0.Profile.hasOutput;
	    	
		StmtIterator i = profileInfo.getModel().listStatements(profileInfo, prop, (Resource)null);
		while(i.hasNext()) {
			Resource p = (Resource) i.nextStatement().getObject();			
			String paramURI   = p.getURI();			
			Parameter refersTo = process.getParameter(URI.create(paramURI)); 
						
			if(refersTo == null) {
				error("The parameter defined in profile does not exist in the process " + 
					"\n    parameter " + paramURI +
					"\n   in profile " + profile.getURI() +
					"\n      process " + process.getURI());
			}	
			else {
				if(isInput) 
					profile.getInputs().add(refersTo);
				else
					profile.getOutputs().add(refersTo);
				
				if(DEBUG) {
					System.out.println("  Profile   " + profile.getURI() + "\n" + 
							(isInput ? "  Input     ":"  Output    " ) +
							paramURI + "\n" + 
							"  Refers to " + refersTo + "\n");
				}				
			}
		}	
	}
	
	private void createConditions(ConditionList conds, StmtIterator i) {
		while(i.hasNext()) {
			Resource p = (Resource) i.nextStatement().getObject();			
						
			conds.add(createCondition(p));
		}				
	}
		
	private void createEffects(EffectList effects, StmtIterator i) {
		while(i.hasNext()) {
			Resource p = (Resource) i.nextStatement().getObject();			
			
			effects.add(createEffect(p));
		}				
	}	
	
	private Condition createCondition(Resource condition) {
		Condition cond = OWLSFactory.createCondition(condition);
		
		return cond;
	}
	
	private Effect createEffect(Resource effect) {
		Effect eff = OWLSFactory.createEffect(effect);
		
		return eff;
	}
	
	private void createGrounding(Service service, Resource groundingInfo) {
		Grounding grounding = OWLSFactory.createGrounding(groundingInfo);

		StmtIterator i = groundingInfo.listProperties(OWLS_1_0.Grounding.hasAtomicProcessGrounding); 
		while(i.hasNext()) {
			Resource apGroundingInfo = i.nextStatement().getResource();
			
			AtomicGrounding apGrounding = createAPGrounding(service, apGroundingInfo);
			if(apGrounding != null)
				grounding.addGrounding(apGrounding);
			else
				error("Invalid AtomicProcess grounding " + apGroundingInfo);
		}
		i = groundingInfo.listProperties(FLAServiceOnt.hasUPnPAtomicProcessGrounding); 
		while(i.hasNext()) {
			Resource apGroundingInfo = i.nextStatement().getResource();
			
			AtomicGrounding apGrounding = createAPGrounding(service, apGroundingInfo);
			if(apGrounding != null)
				grounding.addGrounding(apGrounding);
			else
				error("Invalid AtomicProcess grounding " + apGroundingInfo);
		}			
		
		grounding.setService(service);	
	}	
		
	private AtomicGrounding createAPGrounding(Service service, Resource groundingInfo) {
		try {
			if(groundingInfo.hasProperty(RDF.type, OWLS_1_0.Grounding.WsdlAtomicProcessGrounding))		
				return createWSDLGrounding(service, groundingInfo);

			if(groundingInfo.hasProperty(RDF.type, FLAServiceOnt.UPnPAtomicProcessGrounding))
				return createUPnPGrounding(service, groundingInfo);
		} catch (RuntimeException e) {
			e.printStackTrace();
		}

		return null;	
	}	

	private AtomicGrounding createWSDLGrounding(Service service, Resource groundingInfo) {
		URI processURI = Util.toURI(groundingInfo.getProperty(OWLS_1_0.Grounding.damlsProcess).getResource());
		Resource operationInfo = groundingInfo.getProperty(OWLS_1_0.Grounding.wsdlOperation).getResource();
		String wsdlLoc = groundingInfo.getProperty(OWLS_1_0.Grounding.wsdlDocument).getObject().toString();
		String opName = operationInfo.getProperty(OWLS_1_0.Grounding.operation).getObject().toString();		
		String portType = operationInfo.getProperty(OWLS_1_0.Grounding.portType).getObject().toString();	
		
		
		
  		AtomicProcess process = (AtomicProcess) findProcess(service.getProcess(), processURI);
  		if(process == null) {
  			error("The process specified in the grounding cannot be found " + 
  					"\n    grounding " + groundingInfo +
					"\n      process " + processURI);
  			return null;
  		}
  		
		WSDLAtomicGrounding g = OWLSFactory.createWSDLAtomicGrounding(groundingInfo); 
		g.setProcess(process);
		g.setWSDL(wsdlLoc);
		g.setOperation(opName);
		g.setPortType(portType);

		if(groundingInfo.hasProperty(OWLS_1_0.Grounding.wsdlInputMessage))
			g.setInputMessage(groundingInfo.getProperty(OWLS_1_0.Grounding.wsdlInputMessage).getObject().toString());		

		if(groundingInfo.hasProperty(OWLS_1_0.Grounding.wsdlOutputMessage))
			g.setOutputMessage(groundingInfo.getProperty(OWLS_1_0.Grounding.wsdlOutputMessage).getObject().toString());
		
		createMessageMapList(g, groundingInfo, true);
		createMessageMapList(g, groundingInfo, false);

		if(DEBUG) {
			System.out.println("  Process   " + process.getURI() + "\n" + 
							   "  WSDL file " + wsdlLoc + "\n" + 
							   "  Operation " + opName + "\n");
		}
		
		return g;
	}
	
	private AtomicGrounding createUPnPGrounding(Service service, Resource groundingInfo) {
		URI processURI = Util.toURI(groundingInfo.getProperty(OWLS_1_0.Grounding.damlsProcess).getResource());
		String upnpDevice = groundingInfo.getProperty(FLAServiceOnt.upnpDeviceURL).getObject().toString();
		String upnpService = groundingInfo.getProperty(FLAServiceOnt.upnpServiceID).getObject().toString();
		String upnpAction = groundingInfo.getProperty(FLAServiceOnt.upnpCommand).getObject().toString();
		
		AtomicProcess process = (AtomicProcess) findProcess(service.getProcess(), processURI);
		
		UPnPAtomicGrounding g = OWLSFactory.createUPnPAtomicGrounding(groundingInfo); 
		g.setProcess(process);
		g.setUPnPDescription(upnpDevice);
		g.setUPnPService(upnpService);
		g.setUPnPAction(upnpAction);
		
		createMessageMapList(g, groundingInfo, true);
		createMessageMapList(g, groundingInfo, false);

		if(DEBUG) {
			System.out.println("  Process " + process.getURI() + "\n" + 
							   "  Device  " + upnpDevice + "\n" + 
							   "  Service " + upnpService + "\n" +
							   "  Action  " + upnpAction + "\n");
		}
		
		return g;
	}
		
	private void createMessageMapList(AtomicGrounding g, Resource groundingInfo, boolean isInput) {
		Process process = g.getProcess();	
		Property messageParts = null;
		Property messagePart = null;
		
		if(g instanceof UPnPAtomicGrounding) {
			messageParts = isInput ? 
				FLAServiceOnt.UPnPInputMapping: 
				FLAServiceOnt.UPnPOutputMapping;
			messagePart = FLAServiceOnt.upnpParameter;
										
		}
		else if(g instanceof WSDLAtomicGrounding) {
			messageParts = isInput ? 
				OWLS_1_0.Grounding.wsdlInputMessageParts: 
				OWLS_1_0.Grounding.wsdlOutputMessageParts;
			messagePart = OWLS_1_0.Grounding.wsdlMessagePart;
		}
		
		if(!groundingInfo.hasProperty(messageParts))
			return;
			
		Resource messageMapInfo = (Resource) groundingInfo.getProperty(messageParts).getObject();
		List messageMapList = Util.createList(messageMapInfo);

		MessageMapList mapList = isInput ? g.getInputMap() : g.getOutputMap();
		for(int i = 0; i < messageMapList.size(); i++) {
			Resource messageMap = (Resource) messageMapList.get(i);

			MessageMap map = OWLSFactory.createMessageMap(messageMap);

			URI owlsParameterInfo = Util.toURI(messageMap.getProperty(OWLS_1_0.Grounding.damlsParameter).getResource());
			Parameter owlsParameter = isInput ?
				process.getInputs().getParameter(owlsParameterInfo):
				process.getOutputs().getParameter(owlsParameterInfo);
			map.setOWLSParameter(owlsParameter);
			
			String wsdlMessagePartInfo = messageMap.getProperty(messagePart).getObject().toString();
			map.setGroundingParameter(wsdlMessagePartInfo);

			String transformation = null;
			if(messageMap.hasProperty(OWLS_1_0.Grounding.xsltTranformation)) {
				transformation = messageMap.getProperty(OWLS_1_0.Grounding.xsltTranformation).getString();
				map.setTransformation(transformation);
			}
			
			if(owlsParameter == null) {
				error("Cannot find the target of message map for " + 
					"\n   wsdl parameter " + wsdlMessagePartInfo +
					"\n       in process " + process.getURI() +
					"\n        mapped to " + owlsParameterInfo);
			}				

			if(DEBUG) {
				System.out.println("  Process   " + process.getURI() + "\n" + 
								   "  Param     " + owlsParameterInfo + "\n" + 
								   "  Grounding " + wsdlMessagePartInfo + "\n" + 
								   "  Transform " + transformation + "\n");
			}
			
			mapList.add(map);			
		}
	}	

	private Process findProcess(ProcessComponent process, URI processURI) {
		if(process == null)
			return null;
		else if(process.getURI() != null && process.getURI().equals(processURI)){
			if(process instanceof Process) return (Process) process;
		}
		else if(process instanceof CompositeProcess) {
			return findProcess(((CompositeProcess) process).getComposedOf(), processURI);
		}
		else if(process instanceof ControlConstruct) {	
			ProcessComponentList list = ((ControlConstruct)process).getComponents();
			
			for(int i = 0; i < list.size(); i++) {
				ProcessComponent pc = list.processComponentAt(i);
				Process p = findProcess(pc, processURI);
				if(p != null) return p;
			}
		}
			
		return null;	
	}
	
	private void copyPropertyValues(Resource subj, Property srcProp, Property targetProp) {
		List values = new ArrayList();
		StmtIterator si = subj.listProperties(srcProp);
		while(si.hasNext())
			values.add(si.nextStatement().getObject());
		Iterator i = values.iterator(); 
		while(i.hasNext())
			subj.addProperty(targetProp, (RDFNode) i.next());
			
	}
}
