/****************************************************************************
 * A graphical editor/viewer panel for activity expansion structures
 *
 * @author Jussi Stader
 * @version 4.0+
 * Updated: Mon Nov  6 11:11:45 2006
 * Copyright: (c) 2001, AIAI, University dinburgh
 *
 *****************************************************************************
 */
package ix.iview.igraph;

import java.awt.*;
import java.util.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import lt.monarch.graph.*;
import lt.monarch.graph.view.*;
import lt.monarch.graph.view.layouts.*;
import lt.monarch.graph.view.looks.*;
import lt.monarch.graph.view.utils.*;
import lt.monarch.graph.plugins.*;
import lt.monarch.graph.swing.*;

import ix.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.iface.util.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.domain.*;
import ix.iview.*;
import ix.iview.util.*;
import ix.iview.domain.*;


/****************************************************************************
 * A graphical editor/viewer panel for activity expansion structures
 *
 */
public class GExpansionEditor extends AConstructEditorPanel
implements ConstructEditing, ActionListener, UndoEditing {

  private String editorName = "Expansion Editor";
  //private JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
  //private final JEditorPane notesEditor = new JEditorPane ();
  //protected final JPanel contentPanel = new JPanel (new BorderLayout ());
  private final ExpansionModel model = new ExpansionModel();
  protected ActionEditorPanel parent;
  protected NodeViewBase currentNode = null;
  private JPopupMenu nodePopup;
  private JMenu constructMenu;
  private JMenu expansionsMenu;
  private JTextField nameField;
  private JTextField patternField;
  private final GraphView graphView = new GraphView();
  private final MouseEventDistributor eventForwarder = 
                  new MouseEventDistributor();
  private final GraphPanel graphPanel = new GraphPanel(graphView);
  private final NodeSelectionManager selection = 
                  new NodeSelectionManager(model);


  private final CatchingActionListener catchingListener = 
    new CatchingActionListener(this);


  public GExpansionEditor() {
    parent = null;
    Debug.noteln("ERROR: ******** Do not use constructor GExpansionEditor()" + 
		 " - give it a parent!");
  }

  public GExpansionEditor(ActionEditorPanel parent) {
    
    //Debug.noteln("GEE: Making graph editor");
    setLayout(new BorderLayout());
    model.setPanel(parent);
    this.parent = parent;


    graphView.setModel(model);

    graphView.addPlugin(new SelectedNodeEmphasizer(selection));
    graphView.addPlugin(new NodeSelector(selection));

    LinkDragger linkDragger = new LinkDragger(model.getLinkStrategy());
    linkDragger.setInputModifierMask(0);
    graphView.addPlugin(linkDragger);

    graphView.addPlugin(new NodeBoxSelector(selection));
    graphView.addPlugin(new NodeDragger(selection));

    //final MouseEventDistributor eventForwarder = new MouseEventDistributor();
    graphView.addPlugin(eventForwarder);
    graphView.setNodeViewFactory(new ExpansionNodeViewFactory());

    graphPanel.setBackground(new Color (240,240,240));
    graphPanel.setPreferredSize(new Dimension(500,500));
    //this.setPreferredSize(new Dimension(500,500));
    this.add(BorderLayout.CENTER, new JScrollPane(graphPanel));

    nodePopup = makePopup(catchingListener);
    populateExpansionsMenu();

    makeToolbar();
    //splitPane.setDividerLocation(0.8);    
  }


  JToolBar toolBar = new JToolBar();
  ImageIcon newGNodeIcon = Util.resourceImageIcon("ide-add-node.gif");
  ImageIcon deleteGNodesIcon = Util.resourceImageIcon("ide-delete-nodes.gif");
  ImageIcon layoutIcon = Util.resourceImageIcon("ide-layout.gif");
  ImageIcon layoutLeftIcon = Util.resourceImageIcon("ide-layout-left.gif");
  public IXToolItem newGNodeItem =
    new IXToolItem(toolBar, this, "newGNode", "Add Node", newGNodeIcon, 
		   "Add a new node");
  public IXToolItem deleteGNodesItem =
    new IXToolItem(toolBar, this, "deleteGNodes", "Delete", deleteGNodesIcon, 
		   "Delete selected node(s)");
  public IXToolItem layoutItem =
    new IXToolItem(toolBar, this, "layout", "Layout", layoutIcon,
		   "Layout graph");
  public IXToolItem layoutLeftItem =
    new IXToolItem(toolBar, this, "layoutLeft", "Layout Left", layoutLeftIcon,
		   "Layout graph left-aligned");


  final Point newNodeLocation = new Point(100, 100);

  private void makeToolbar() {

    toolBar.setAlignmentX(TOP_ALIGNMENT);
    
    newGNodeItem.show();
    deleteGNodesItem.show();
    toolBar.addSeparator();
    layoutItem.show();
    layoutLeftItem.show();
    toolBar.addSeparator();

    Box bitBox = new Box(BoxLayout.Y_AXIS);
    nameField = new JTextField(10);
    patternField = new JTextField(10);
    MouseListener ml = new MouseAdapter() {};
    ThingEditorPanel nameBit = new ThingEditorPanel(ml, "Name  ", nameField);
    //toolBar.add(nameBit);
    bitBox.add(nameBit);
    nameBit = new ThingEditorPanel(ml, "Pattern", patternField);
    //toolBar.add(nameBit);
    bitBox.add(nameBit);
    toolBar.add(bitBox);

    nameField.addFocusListener(new FocusAdapter() {
	public void focusLost(FocusEvent fEvent) {
	  //if (!fEvent.isTemporary())  //only listen if focus within window
	    noteName();
	};
      });
    patternField.addFocusListener(new FocusAdapter() {
	public void focusLost(FocusEvent fEvent) {
	  //if (!fEvent.isTemporary())  //only listen if focus within window
	    notePattern();
	};
      });

    this.add(BorderLayout.NORTH, toolBar);
  }


  private JPopupMenu makePopup(CatchingActionListener catchingListener) {
    JPopupMenu menu = new JPopupMenu();
    JMenuItem menuItem = new JMenuItem("Load graph");
    menuItem.addActionListener(catchingListener);
    menu.add(menuItem);
    menu.addSeparator();
    menuItem = new JMenuItem("Properties");
    menuItem.addActionListener(catchingListener);
    menu.add(menuItem);
    menu.addSeparator ();
    expansionsMenu = new JMenu("Expansions");
    menu.add(expansionsMenu);
    menuItem = new JMenuItem("New Expansion");
    menuItem.addActionListener(catchingListener);
    menu.add(menuItem);
    return menu;
  }
  
  public void populateExpansionsMenu() {
    ArrayList exps = new ArrayList();
    if (currentNode != null) {
      NodeSpec node = (NodeSpec)currentNode.getModelNode();
      exps = parent.getAllExpansions(node);
    }
    UIUtil.populateMenu(this, expansionsMenu, "Expansions", exps);
  }


  public void setAction(UIRefinement uir) {
    uiConstruct = uir;
    loadFromObject();
  }

  public void layoutFromGraph(IGraph graph) {
    //Debug.noteln("GEE: layout from graph");
    Iterator mNodes = model.getNodes();
    if ((graph.view == null) || (mNodes == null) || (!mNodes.hasNext())) { 
      //fresh canvas point model?
      //Debug.noteln("GEE: laying out from canvas");
      graph.layoutFromCanvasPoints(graphView);
    }
    else while (mNodes.hasNext()) {
      Object mNode = mNodes.next();
      //Debug.noteln(" doing node " + mNode.toString());
      Point position = graph.getPosition(mNode);
      if (position != null) {
	//Debug.noteln(" Setting position " + position.toString());
	NodeView vNode = (NodeView)graphView.nodeRegistry.getViewFor(mNode);
	vNode.setPosition(position);
      }
      else Debug.noteln("GEE: node without position");
    }
  }

  private void shiftLeft(GraphView graphView) {
    if (model == null) return;
    for (Iterator mNodes = model.getNodes(); mNodes.hasNext(); ) {
      Object mNode = mNodes.next();
      //Debug.noteln(" doing node " + mNode);
      NodeView vNode = (NodeView)graphView.nodeRegistry.getViewFor(mNode);
      if (vNode != null) {
	Rectangle rect = vNode.getBounds();
	if (rect != null) {
	  rect.setLocation(20, (int)rect.getY());
	  //Debug.noteln(" Setting position", rect);
	  Point point = new Point();
	  point.setLocation(rect.getCenterX(), rect.getCenterY());
	  vNode.setPosition(point);
	  //vNode.invalidate();
	}
	else Debug.noteln("GEE: node without bounds", mNode);
      }
      else Debug.noteln("GEE: node without view", mNode);
    }
  }

  public void expandNode() {
    if (currentNode == null) return;  //this should not happen
    ExpansionModel.NodeComponent node = 
      (ExpansionModel.NodeComponent)currentNode.getModelNode();
    model.expandNode(node);
  }

  public void saveExpansionGraph(UIRefinement uir) {
    if ((uir == null) || uir.isUndefined() || uir.isEmpty()) return;
    String name = uir.getName();
    //Debug.noteln("GEE: Saving expansion graph " + name);
    IGraph graph = uir.getGraph();
    if (graph == null) {
      graph = new IGraph(name, graphView);
      uir.setGraph(graph);
    }
    else graph.refreshGraph(graphView);
  }

  //-----------------------UndoEditing things-----------------------
  // reading/updating the model will have the desired effect

  /** Sets the given field to the given value in the editor. */
  public void undoSetValue(String field, Object value) {
    Debug.noteln("Undo set: field "+field+", value", value);
    if (field == null) return;
    else if (field.equals("name")) nameField.setText(value.toString());
    else if (field.equals("pattern")) {
      String pat = "";
      if (value != null) pat = UIUtil.listToDisplay((LList)value);
      patternField.setText(pat);
    }
    else if (field.equals("graph")) {
      Debug.noteln(" Graph value class", value.getClass());
    }
    //getModel().setValue(field, value);
  }

  /** Gets the given field to the given value in the editor. */
  public Object undoGetValue(String field) {
    Debug.noteln("Undo get: field", field);
    //return getModel().getValue(field);
    return null;
  }


   //-----------------------ConstructEditing things-----------------------

  
  /** Sets the current UI construct. */
  public void setUIConstruct(UIObject construct) {
    setUIConstructOnly(construct);
  }
  /** Sets the current UI construct. */
  public void setUIConstructOnly(UIObject construct) {
    UIObject old = getUIConstruct();
    //if (construct.equals(uiConstruct)) return;
    setAction((UIRefinement)construct);
    parent.fireConstructChanged(old, construct, this);
  }

  /** Save the data from the panel into the given object. */
  public void saveToObject(EditableObject object) {
    UIRefinement uir = (UIRefinement)object;
    saveToGivenObject(uir);
  }

    private void noteName() {
	UIRefinement uir = (UIRefinement)getUIConstruct();
	uir.setName(nameField.getText().trim());
	nameField.setText(uir.getName()); //in case of name clash
    }
    private void notePattern() {
	UIRefinement uir = (UIRefinement)getUIConstruct();
	uir.setPattern(Lisp.elementsFromString(patternField.getText().trim()));
    }

  public void saveToGivenObject(UIRefinement uir) {
    //if (uir != null) Debug.noteln("GEE: saving object", uir.getName());
    //else Debug.noteln("GEE: saving null object");
    noteName();
    notePattern();
    model.updateAction(uir);
    this.saveExpansionGraph(uir);
  }
  
  /** load the data for the panel from the current object. */
  public void loadFromObject() {
    UIRefinement uir = (UIRefinement)getUIConstruct();
    nameField.setText("");
    patternField.setText("");
    this.model.setAction(uir);
    if (uir != null) {
      nameField.setText(uir.getName());
      patternField.setText(UIUtil.listToDisplay(uir.getPattern()));
      IGraph graph = uir.getGraph();
      if (graph != null) layoutFromGraph(graph);
      else autoAutoLayout(graphView);
    }    
  }

  public void displayFieldData(String fieldName) {
    if (fieldName == null) return;
    else {
      UIRefinement uir = (UIRefinement)getUIConstruct();
      model.setAction(uir);
      IGraph graph = (IGraph)parent.getUIDomain().getNamedGraph(uir.getName());
      if (graph != null) layoutFromGraph(graph);
      else autoAutoLayout(graphView);
    }
    //Cannot do those below because deleting links is too much faff
    //else if (fieldName.equals("orderings")) model.loadLinks();
    //else if (fieldName.equals("nodes")) model.loadNodes();
  }

  public void autoAutoLayout(GraphView graphView) {
    autoLayout(graphView);
  }
  public void userAutoLayout(GraphView graphView) {
    autoLayout(graphView);
  }
  public void userAutoLayoutLeft(GraphView graphView) {
    autoLayoutLeft(graphView);
  }
  public void autoLayout(GraphView graphView) {
    //graphView.setLayouter(new FlowGraphLayouter());
    graphView.layoutGraph();
    updateURIAutoGraph(graphView);
  }
  public void autoLayoutLeft(GraphView graphView) {
    graphView.layoutGraph();
    shiftLeft(graphView);
    updateURIAutoGraph(graphView);
  }

  private void updateURIAutoGraph(GraphView graphView) {
    UIRefinement uir = (UIRefinement)getUIConstruct();
    if (uir != null) {
      IGraph tempGraph = new IGraph(uir.getName(), graphView);
      tempGraph.refreshGraph(graphView);
      uir.setAutoPoints(tempGraph.getGraphCanvasPoints());
    }
  }



  //-----------------------Actions etc-----------------------------------------
  /**
   * Wakes up the ActionListener with a user action.
   * This is called when a KeyStroke happens in which the ActionListener
   * registered its interest.
   */
  public void actionPerformed(ActionEvent ae){
    String command = ae.getActionCommand();
    Debug.noteln("GEE got action command", command);
    if (command.equals("New Expansion")) expandNode();
    else if (command.equals("Expansions")) {
      //get the construct name from the item text and show it
      JMenuItem item = (JMenuItem)ae.getSource();
      String name = item.getText().trim();
      parent.setUIConstruct((UIObject)parent.getNamedConstruct(name)); 
    }
    else if (command.equals("newGNode")) {
      //Debug.noteln("Got add node");
      Object node = model.addNewNode();
      if (node != null) {
	NodeView view = (NodeView)graphView.nodeRegistry.getViewFor(node);
	view.setPosition(newNodeLocation);
	newNodeLocation.x += 20;
	newNodeLocation.y += 20;
      }
    }
    else if (command.equals("deleteGNodes")) {
      //Debug.noteln("Got delete");
      Iterator nodes = selection.getSelectedNodes();
      while (nodes.hasNext()) model.deleteNode((NodeSpec)nodes.next());
    }
    else if (command.equals("layout")) {
      //Debug.noteln("Got layout");
      userAutoLayout(graphView);
    }
    else if (command.equals("layoutLeft")) {
      //Debug.noteln("Got layout left");
      userAutoLayoutLeft(graphView);
    }
    //else if (command.equals("")) 
    else UIUtil.notImplemented(parent,command);
  }

  private class ExpansionNodeViewFactory implements NodeViewFactory {
    public NodeView createViewFor(Object o) {
      ExpansionModel.NodeComponent c = (ExpansionModel.NodeComponent)o;
      CompositeRectangularNode node = new 
	CompositeRectangularNode(graphView, graphPanel, c) {
	protected RectangularPin createPinFor(Object pin, String label) {
	  RectangularPin pinView = super.createPinFor(pin, label);
	  pinView.setBackground(Color.green);
	  return pinView;
	}
      };
      
      node.setOrientation(node.ORIENTATION_HORIZONTAL);
      node.setBackground(Color.white);
      node.setLabel(c.id + ": " + c.label);
      /*
      node.setLabel(c.label);
      Name id = c.id;
      if (id != null) node.setBottom(c.id.toString());
      */
      // create pin views for the node
      for (int i=0; i<c.inputPins.length; i++)
	node.addInputPin(c.inputPins[i], c.inputPins[i].label);
      for (int i=0; i<c.outputPins.length; i++)
	node.addOutputPin(c.outputPins[i], c.outputPins[i].label);
      // Right click on the node pops up a menu 
      final NodeViewBase targetNode = node;
      GExpansionEditor.this.eventForwarder.addMouseListener(node, 
							    new MouseAdapter(){
	public void mousePressed (MouseEvent e) {
	  if (!SwingUtilities.isRightMouseButton(e)) return;
	  GExpansionEditor.this.currentNode = targetNode;
	  //populateConstructMenu();//***should be done via popupMenuListener
	  populateExpansionsMenu();//***should be done via popupMenuListener
	  nodePopup.show(graphPanel, e.getX(), e.getY());
	}
      });
      
      return node;
    }
  }
  
}
