/****************************************************************************
 * A data model for expansion graphs (nodes and links)
 *
 * @author Jussi Stader
 * @version 4.0+
 * Updated: Wed Aug 31 11:23:54 2005
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
// ExpansionModel.java

package ix.iview.igraph;


import java.util.*;
//import java.awt.*;
import javax.swing.*;
import java.io.*;
import lt.monarch.graph.*;
import lt.monarch.graph.plugins.*;

import ix.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.domain.*;
import ix.iview.*;
import ix.iview.util.*;
import ix.iview.domain.*;

/**
 * graph model for the electrical demo
 */
public class ExpansionModel extends AbstractGraphModel {
  private LList components;
  public UIRefinement currentAction;
  private int count = 0;
  public ActionEditorPanel editor;

  static class Pin {
    String label;
    NodeComponent component;
    ArrayList connections;
   
    Pin (NodeComponent n, String label) {
      this.component = n;
      this.label = label;
      connections = new ArrayList();
    } 

    boolean isInputPin () {
      return Arrays.asList(component.inputPins).contains (this);
    }

    boolean isOutputPin () {
      return Arrays.asList(component.outputPins).contains (this);
    }

    Iterator getConnections () {
      return connections.iterator();
    }
  }


  static class NodeComponent extends UINodeSpec {
    String label = UIUtil.listToDisplay(getPattern());
    Name id = this.getId();
    Pin[] inputPins = new Pin[] {new Pin(this, "B")};
    Pin[] outputPins = new Pin[] {new Pin(this, "E")};

    public NodeComponent(UIDomain uiDomain, Name id) {
      this(uiDomain, id, Lisp.NIL);
    }
    public NodeComponent(UIDomain uiDomain, Name id, LList pattern) {
      super(uiDomain, id, pattern);
      //Debug.noteln("EM: made node", this.show());
    }
    public NodeComponent(UIDomain uiDomain, Name id, String pattern) {
      super(uiDomain, id, Lisp.elementsFromString(pattern));
    }
    public NodeComponent(UIDomain uiDomain, String name, LList pattern) {
      super(uiDomain, name, pattern);
    }

    public String show() {
      String show = "";
      show = show + getId();
      show = show + ": " + getPattern();
      return show;
    }
  }

  public ExpansionModel(UIRefinement uir) {
    super();
    setAction(uir);
  }

  public ExpansionModel() {
    components = Lisp.NIL;
  }


  public void setPanel(ActionEditorPanel p) {
    editor = p;
  }

  public void setAction(UIRefinement uir) {
    //Debug.noteln("EM: setting action to", uir);
    currentAction = uir;
    deleteAllNodeViews();
    count = 0;
    if (uir == null) return;

    //turn the action's sub-nodes into components
    //Debug.noteln("EModel - setAction: turning nodes into components");
    List nodes = currentAction.getNodes();
    loadNodes(nodes);
    //make a link for each ordering constraint
    List orderings = currentAction.getOrderings();
    loadLinks(orderings);
    updateCounter();
  }

  /** Updates the current action from the graph */
  public void updateAction() {
    updateAction(currentAction);
  }
  /** Updates the given action from the graph */
  public void updateAction(UIRefinement uir) {
    updateNodesAndLinks(uir);
  }

  private void loadNodes(List nodes) {
    for (Iterator i = nodes.iterator(); i.hasNext(); )
      try { addNode((UINodeSpec)i.next());}
      catch (Exception e) {
	Debug.noteException(e);
      }
  }
  private void loadLinks(List orderings) {
    for (Iterator i = orderings.iterator(); i.hasNext(); )
      try { loadLink(i.next()); }
      catch (RuntimeException nodeEndProb) {
	Debug.noteException(nodeEndProb);
      }
  }

  /* Updates the given refinement's node information from the graph */
  private void updateNodesAndLinks(UIRefinement uir) {
    try {
      //uir.clearNodes(); 
      //uir.clearOrderings(); 
      ArrayList nodes = new ArrayList();
      ArrayList links = new ArrayList();
      NodeComponent node;
      Pin[] pins;
      Iterator it = getNodes();
      while (it.hasNext()) {
	node = (NodeComponent)it.next();
	//Debug.noteln("EM: Updating node ", node.getId().toString());
	nodes.add((UINodeSpec)node);
	pins = node.inputPins;
	for (int i=0; i<pins.length; i++) addLinks(links, pins[i]);
	pins = node.outputPins;
	for (int i=0; i<pins.length; i++) addLinks(links, pins[i]);
      }
      if (!uir.sameNodes(uir.getNodes(), nodes)) uir.setNodes(nodes);
      if (!IVUtil.sameSet(uir.getOrderings(), links)) uir.setOrderings(links);
    }
    catch (Exception e) {
      Debug.noteException(e);
    }
  }

  private void addLinks(List links, Pin fromPin) {
    Iterator toPins = getLinksFrom(fromPin);
    Name fromId = fromPin.component.getId();
    //Debug.noteln(" EM: Adding links of node", fromId.toString());
    Name toId;
    End tpFrom;
    End tpTo;
    if (fromPin.isInputPin()) tpFrom = End.BEGIN; else tpFrom = End.END;
    while (toPins.hasNext()) {
      Pin toPin = (Pin)toPins.next();
      if (toPin.isInputPin()) tpTo = End.BEGIN; else tpTo = End.END;
      toId = toPin.component.getId();
      //Debug.noteln(" toId is", toId);
      //if (null != uir.findNode(toId)) { //ignore old-style links
      if (null != findNode(toId)) { //ignore old links
	//Debug.note("  EM: adding link from ");
	//Debug.noteln(fromId.toString() + " to", toId.toString());
	Ordering ordering = 
	  new Ordering(new NodeEndRef(tpFrom, fromId),
		       new NodeEndRef(tpTo, toId));
	links.add(ordering);
      }
    }
    //seems to reverse links list, so put it back.
    Collections.reverse(links);
  }

  private void updateNodesAndLinksOld(UIRefinement uir) {
    try {
      uir.clearNodes(); 
      uir.clearOrderings(); 
      NodeComponent node;
      Pin[] pins;
      Iterator it = getNodes();
      while (it.hasNext()) {
	node = (NodeComponent)it.next();
	//Debug.noteln("EM: Updating node ", node.getId().toString());
	uir.addNode((UINodeSpec)node);
	pins = node.inputPins;
	for (int i=0; i<pins.length; i++) addLinksOld(uir, pins[i]);
	pins = node.outputPins;
	for (int i=0; i<pins.length; i++) addLinksOld(uir, pins[i]);
      }
    }
    catch (Exception e) {
      Debug.noteException(e);
    }
  }

  private void addLinksOld(UIRefinement uir, Pin fromPin) {
    Iterator toPins = getLinksFrom(fromPin);
    Name fromId = fromPin.component.getId();
    //Debug.noteln(" EM: Adding links of node", fromId.toString());
    Name toId;
    End tpFrom;
    End tpTo;
    if (fromPin.isInputPin()) tpFrom = End.BEGIN; else tpFrom = End.END;
    while (toPins.hasNext()) {
      Pin toPin = (Pin)toPins.next();
      if (toPin.isInputPin()) tpTo = End.BEGIN; else tpTo = End.END;
      toId = toPin.component.getId();
      //Debug.noteln(" toId is", toId);
      //if (null != uir.findNode(toId)) { //ignore old-style links
      if (null != findNode(toId)) { //ignore old links
	//Debug.note("  EM: adding link from ");
	//Debug.noteln(fromId.toString() + " to", toId.toString());
	Ordering ordering = 
	  new Ordering(new NodeEndRef(tpFrom, fromId),
		       new NodeEndRef(tpTo, toId));
	uir.addOrdering(ordering);
      }
    }
    //seems to reverse links list, so put it back.
    Collections.reverse(uir.getOrderings());
  }

  public void expandNode(NodeComponent node) {
    String expansionName = 
      JOptionPane.showInputDialog("Please enter the name of the expansion.");
    if (expansionName == null) return;
    UIDomain uiDomain = editor.getUIDomain();
    Object sInDomain = uiDomain.getNamedRefinement(expansionName);
    if (sInDomain == null) {
      //the name is not in the domain so make it
      UIRefinement uir = new UIRefinement(editor.getUIDomain());
      uir.setName(expansionName);
      uir.setPattern(node.getPattern());
      uiDomain.addConstruct(uir);
      editor.setUIConstruct((UIObject)uir);
    }
    else {
      String message = "There already is an expansion with this name." 
	+ " Please enter a new name.";
      JOptionPane.showMessageDialog(editor, message);
      expandNode(node);
    }
  }
  private void updateCounter() {
    Integer newId = new Integer(currentAction.getHighestNodeId().toString());
    count = newId.intValue();    
    /*
    Iterator nodes = this.getNodes();
    Name largest = Name.valueOf("0");
    int c;
    while (nodes.hasNext()) {
      Name id = ((NodeComponent)nodes.next()).getId();
      c = largest.compareTo(id.toString());
      if (c < 0) largest = id;
    }
    count = Integer(largest.toString()).intValue();
    */
  }


  protected void loadLink(Object o) throws RuntimeException {
    Class orderClass = o.getClass();
    Ordering ordering;
    if (o instanceof Ordering) ordering = (Ordering)o;
    else {
      Debug.noteln("EM ERROR: cannot make orderings from class ", 
		   o.getClass());
      //(new Throwable()).printStackTrace();
      return;
    }
    NodeEndRef nodeEnd = ordering.getFrom();
    NodeComponent fromN = findNode(nodeEnd);
    End fromT = nodeEnd.getEnd();
    nodeEnd = ordering.getTo();
    NodeComponent toN = findNode(nodeEnd);
    End toT = nodeEnd.getEnd();
    if (fromN == null) 
      throw new RuntimeException("Cannot recognise from-node in " + 
				 ordering.toString());
    if (toN == null) 
      throw new RuntimeException("Cannot recognise to-node in " + 
				 ordering.toString());
    if ((fromN == null) || (toN == null)) {
      return;
    }
    Pin fromPin;
    Pin toPin;
    if (fromT == End.BEGIN)
      fromPin = fromN.inputPins[0];
    else fromPin = fromN.outputPins[0];
    if (toT == End.END)
      toPin = toN.outputPins[0];
    else toPin = toN.inputPins[0];
    addLink(fromPin, toPin);
  }

  /**
   * Finds the NodeComponent referenced by the given object. 
   * 
   */ 
  protected NodeComponent findNode(Object o) {
    Iterator nodes = this.getNodes();
    while (nodes.hasNext()) {
      Object n = nodes.next();
      NodeComponent node = (NodeComponent)n;
      if (currentAction.isRef(node, o)) return node;
    }
    return null;    
  }


  public Iterator getNodes() {
    return components.iterator(); 
  }

  public Iterator getLinksFrom (Object p)
  {
    if (!(p instanceof Pin))
      return Collections.EMPTY_LIST.iterator ();

    Pin pin = (Pin)p;
    return pin.getConnections ();
  }

  protected void fireLinkRemoved(Object p1, Object p2) {
    super.fireLinkRemoved(p1, p2);
  }

  protected void fireLinkAdded(Object p1, Object p2) {
    super.fireLinkAdded(p1, p2);
  }

  public void addLink(Object o1, Object o2) {
    Pin originPin = (Pin)o1;
    Pin targetPin = (Pin)o2;
    //Debug.noteln("EM: Adding link from " + originPin.component.getId() 
    //	 + " to", targetPin.component.getId());
    originPin.connections.add(targetPin);
    fireLinkAdded(o1, o2);
  }

  public LinkDragStrategy getLinkStrategy () {
    return new AbstractLinkDragStrategy() {
      
      public boolean isValidOriginPin(Object pin) {
        return true; //((Pin)pin).isOutputPin();
      }

      public boolean isValidTargetPin(Object origin, Object pin) {
        Pin originPin = (Pin)origin;
        Pin targetPin = (Pin)pin;
        //return originPin.component != targetPin.component &&
        //       targetPin.isInputPin();
	return originPin.component != targetPin.component;
      }

      public void addLink(Object o1, Object o2) {
	ExpansionModel.this.addLink(o1,o2);
      }

      public void removeLink(Object o1, Object o2) {
        Pin originPin = (Pin)o1;
        Pin targetPin = (Pin)o2;
        originPin.connections.remove (targetPin);
        targetPin.connections.remove (originPin);
        fireLinkRemoved (o1, o2);
      }
    };
  }

  /** Called when a node is added by the user. */
  public Object addNewNode() {
    String pattern = 
      JOptionPane.showInputDialog("Please enter the sub-node's pattern.");
    if (pattern == null) return null;
    count++; 
    NodeComponent node = 
      new NodeComponent(editor.getUIDomain(), Name.valueOf("" + count), 
			pattern);
    currentAction.addNode(node);
    components = new Cons(node, components);
    fireNodeAdded(node);
    return node;
  }
  public Object addNode(UINodeSpec node) {
    LList pattern = node.getPattern();
    Name id = node.getId();
    //Debug.noteln("em Making new node component with id", id);
    NodeComponent nc = new NodeComponent(editor.getUIDomain(), id, pattern);
    components = new Cons(nc, components);
    fireNodeAdded((Object)nc);
    return (Object)nc;
  }
  //just delete the node from the view
  public void deleteNode(NodeSpec node) {
    currentAction.deleteNode(node);
    try { deleteNodeView((NodeComponent)node); }
    catch (Exception e) { Debug.noteln(Debug.foldException(e)); }
  }
  public void deleteNodeView(NodeComponent node) {
    components = components.without(node);
    /*
    for (int i=0; i<node.inputPins.length; i++) 
      deleteLinksViews(node.inputPins[i]);
    for (int i=0; i<node.inputPins.length; i++) 
      deleteLinksViews(node.outputPins[i]);
      */
    fireNodeRemoved(node);
  }


  public void deleteAllNodeViews () {
    if (components.isEmpty()) return;
    for (Iterator i = getNodes(); i.hasNext(); ) {
      NodeComponent node = (NodeComponent)i.next();
      fireNodeRemoved(node);
    }
    components = Lisp.NIL;
    /*
    else {
      deleteNodeView((NodeComponent)components.car());
      deleteAllNodeViews();
    }
    */
  }



}
