/****************************************************************************
 * A class that helps to edit type specifications for activity
 * relatable objects.
 *
 * File: UIObjectClass.java
 * @author: Jussi Stader (jussi@wiay.aiai.ed.ac.uk)
 * @version: 4.2+
 * Updated: Mon Oct  2 10:49:52 2006
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 *
 ****************************************************************************
 */

package ix.iview.domain;

import java.util.*;
import javax.swing.tree.*;
import javax.swing.JOptionPane;
import ix.util.*; //Debug, EnumeratedValue
import ix.util.lisp.*;//Symbol, ListOfSymbol
import ix.icore.domain.*;
import ix.icore.*;
import ix.iview.*; 
import ix.iview.util.*; 
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.ui.event.*;
import ix.iface.ui.tree.*;


/** 
 * A class that helps to edit type specifications for activity
 * relatable objects.  Note that these are all types rather than the
 * objects themselves. Objects can be instantiated during execution,
 * but not here.
 */
public class UIObjectClass extends AbstractAnnotatedObject
  implements Named, EditableObject, UIObject, IXNode, TreeNode
{
  // object model. Can be anything. Has type, properties (with types),
  // constraints

  private Symbol oldName;
  public Symbol name;
  /** A list of symbol names of parent UIObjectClasses */
  public ListOfSymbol superClassNames = new LinkedListOfSymbol(); 
  /** A list of  symbol names of child UIObjectClasses */
  public ListOfSymbol subClassNames = new LinkedListOfSymbol(); 
  /** 
   * A list of symbols that are property names. May turn into a list of
   * property types 
   */
  public ListOfObjectProperty objectProperties = 
    new LinkedListOfObjectProperty();
  public Annotations annotations;

  protected UIDomain uiDomain;
  protected ObjectClass baseObject; //used to note draft
  protected String baseReference = ""; /** name of the original */

  //noting Symbol-ObjectProperty, especially for lookups
  private HashMap propertyMap = new HashMap();

  public UIObjectClass(UIDomain uiDomain, String name) {
    this(uiDomain, Symbol.intern(name));
  }
  public UIObjectClass(UIDomain uiDomain, Symbol name) {
    super();
    this.name = name;
    this.uiDomain = uiDomain;
    //Debug.noteln("UIOC***************made new UIObjectClass*", print());
    //(new Throwable()).printStackTrace();
  }
  public UIObjectClass(UIDomain uiDomain, ObjectClass original) {
    super();
    //Debug.noteln("UIOC***************making new UIObjectClass**");
    //(new Throwable()).printStackTrace();
    name = Symbol.intern(original.getName());
    this.uiDomain = uiDomain;
    try {
      baseObject = (ObjectClass)original.clone();
    }
    catch (CloneNotSupportedException e) {
      baseObject = (ObjectClass)makeBaseObject();
    }
    //if (!isEmpty(original)) { //***put back in for efficiency!
      baseReference = original.getName();
      loadFromDomain();
      //}
    uiDomain.addedObject(this, original);
  }


  public UIObjectClass cloneThis() {
    UIObjectClass copy;
    try {
      copy = (UIObjectClass)this.clone();
      copy.baseObject = null;
      copy.baseReference = "";
    }
    catch (CloneNotSupportedException e) {
      copy = new UIObjectClass(uiDomain, getName());
      copy.setSuperClassNames(new  LinkedListOfSymbol(superClassNames));
      copy.setSubClassNames(new  LinkedListOfSymbol(subClassNames));
      copy.setObjectProperties(new LinkedListOfObjectProperty(objectProperties));
      copy.setAnnotations(getAnnotations());
    }
    return copy;
  }


  

  public String getName() { 
    if (name == null) return "";
    else return name.toString(); 
  }
  public void setName(String newName) {
    //Debug.noteln("UIOC: setName", newName);
    if ( Symbol.intern(newName) != name) 
      uiDomain.changeName(this, newName);
  }
  public void setLegalName(String newName) {
    if ( Symbol.intern(newName) == name) return; //no change

    oldName = name;
    Symbol newValue = Symbol.intern(newName);
    name = newValue;
    fireDataChange("name", oldName, newValue);
    //tidyParents(oldValue, newValue);
    //tidyChildren(oldValue, newValue);
  }

  public List getObjectProperties() {
    return objectProperties;
  }
  public List getSuperClassNames() {
    return superClassNames;
  }
  public List getSubClassNames() {
    return subClassNames;
  }

  public Annotations getAnnotations() { return annotations; }
  public void setAnnotations(Annotations annotations) {
    this.annotations = annotations;
  }

  public boolean isProperty(String prop) {
    Object property = propertyMap.get(Symbol.intern(prop));
    return (property != null);
  }
  public boolean isProperty(ObjectProperty prop) {
    HashSet own = getAllProperties();
    if (own == null) return false;
    else return own.contains(prop);
  }
  public boolean isOwnProperty(ObjectProperty prop) {
    HashSet own = getOwnProperties();
    if (own == null) return false;
    else return own.contains(prop);
  }
  public HashSet getOwnProperties() {
    return new HashSet(objectProperties);
  }
  public HashSet getAllProperties() {
    HashSet properties = new HashSet(getOwnProperties());
    HashSet props = getAncestorProperties();
    if (props != null) properties.addAll(props);
    return properties;
  }
  public HashSet getAncestorProperties() {
    List parents = getParents();
    if ((parents == null) || parents.isEmpty()) return null;
    else {
      HashSet properties = new HashSet();
      for (Iterator i = parents.iterator(); i.hasNext(); ) {
	Object next = i.next();
	UIObjectClass parent = null;
	if (next instanceof ObjectClass) {
	  //get the UI version
	  Debug.noteln("UIOC: got ObjectClass for name", next);
	  Debug.noteln(" ancestor properties stop here");
	  Collection ps = parent.getObjectProperties();
	}
	else if (next instanceof UIObjectClass) 
	  parent = (UIObjectClass)next;
	if (parent != null) { //seems to happen - should not!
	  HashSet pps = parent.getAllProperties();
	  if (pps != null) properties.addAll(pps);
	}
      }
      return properties;
    }
  }


  /** Edits the ARO type. Implements EditableObjct */
  public boolean setValue(String field, Object value) {
    try {
      //Debug.noteln("UIOC: value class for " + field, value.getClass());
      //Debug.noteln(" value is ", value);
      if (field == null) return false;
      else if (field.equals("name")) 
	setName((String) value);
      else if (field.equals("superClassNames")) 
	setSuperClassNames((List)value);
      else if (field.equals("objectProperties")) 
	setObjectProperties((List)value);
      else return false;
      return true;
    }
    catch (Exception e) {
      Debug.noteException(e);
      return false;
    }
    
  }
  /** Determines whether the two given values of the given field differ.
   * implements EditableObjct
   */
  public boolean sameValue(String field, Object value, Object otherValue) {
    if (field == null) return false;
    if (value == null) return UIUtil.isEmptyThing(otherValue);
    if (otherValue == null) return false;
    if (field.equals("name")) IVUtil.sameSymbol(value, otherValue);
    if (field.equals("name") || field.equals("comments") || 
	field.equals("annotations"))
      return value.equals(otherValue);
    if (field.equals("superClassNames") || field.equals("objectProperties")) 
      return IVUtil.sameSet((Collection)value, (Collection)otherValue);
    return false; //safe exit
  }

  /**
   * If the specifications are empty, we don't care about the parents.
   */
  public boolean isEmpty() {
    return (isUndefined()
	    && (objectProperties == null || objectProperties.isEmpty())
	    && (subClassNames == null || subClassNames.isEmpty())
	    && (annotations == null || annotations.isEmpty())); 
  }
  public boolean isUndefined() {
    return ((getName() == "") || 
	    (getName().equals("undefined")) || 
	    (getName().equals("Undefined")));
  }
  /**
   * If the original specifications are empty
   */
  public boolean isEmpty(ObjectClass oc) {
    return ((oc == null) ||
	    ((oc.getName() == "") || 
	     (oc.getName().equals("undefined")) || 
	     (oc.getName().equals("Undefined")))
	    && (oc.getObjectProperties() == null || 
		oc.getObjectProperties().isEmpty())
	    && (oc.getSuperClassNames() == null || 
		oc.getSuperClassNames().isEmpty())
	    && (oc.getAnnotations() == null || 
		oc.getAnnotations().isEmpty())); 
  }


  /**
   * Makes sure that super classes know about this one!
   */
  public void updateSuperClasses() {
    String name = getName();
    //Debug.noteln("UIOC: Updating super classes for", name);
    List supers = getSuperClassNames();
    //Debug.noteln("  super classes are", UIUtil.show(supers));
    boolean news = false;
    if ((supers != null) && (supers.size() != 0)) news = true; //newParents
    //else Debug.noteln("UIOC: no new parents for", name);
    List allUIOs = uiDomain.getAllObjectClasses();
    //remove old references 
    for (Iterator i = allUIOs.iterator(); i.hasNext(); ) {
      Object o = i.next();
      if (o instanceof UIObjectClass) {
	UIObjectClass uio = (UIObjectClass)o;
	if (!this.equals(uio)) {  //not self
	  Symbol pName = Symbol.intern(uio.getName());
	  if (news && !supers.contains(pName)) {
	    //Debug.noteln("  Removing from parent", o);
	    uio.removeChild(name);
	  }
	  if (news && supers.contains(pName)) {
	    //Debug.noteln("  Adding to parent", o);
	    uio.addChild(name);
	  }
	}
      }
      //else Debug.noteln("UIOC: Cannot add child to IX", o.toString());
    }
    //Debug.noteln("UIOC: Updated super classes for", name);
  }
  public void setSuperClassNames(List newValue) {
    Collection oldValue = superClassNames;
    Collection oldParents = Collect.difference(oldValue, newValue);
    Collection newParents = Collect.difference((Collection)newValue, oldValue);
    String name = getName();

    //remove this child from all old parents that are not in new
    for (Iterator i = oldParents.iterator(); i.hasNext();) {
      String sup = i.next().toString();
      Object oc = uiDomain.getNamedObjectClass(sup);
      if ((oc != null) && (oc instanceof UIObjectClass))
	((UIObjectClass)oc).removeChild(name);
      else Debug.noteln("UIOC: temporary missing super class", sup);
    }
    if (newValue == null) superClassNames = new LinkedListOfSymbol();
    else {
      superClassNames = new LinkedListOfSymbol(newValue); 
      for (Iterator i = newParents.iterator(); i.hasNext();) {
	String sup = i.next().toString();
	Object oc = uiDomain.getNamedObjectClass(sup);
	if ((oc != null) && (oc instanceof UIObjectClass))
	  ((UIObjectClass)oc).addChild(name);
	else Debug.noteln("UIOC: temporary missing super class", sup);
      }
    }
    fireDataChange("superClassNames", oldValue, newValue);
  }
  private void addParent(Symbol parent) {
    if ((parent != null) && (!superClassNames.contains(parent))) {
      List oldNames = new LinkedListOfSymbol(superClassNames);
      superClassNames.add(parent);
      fireDataChange("superClassNames", oldNames, superClassNames);
      updateSuperClasses();
    }
  }
  private void addParent(String parent) {
    if ((parent != null) && !parent.equals(""));
      addParent(Symbol.intern(parent));
  }
  public void addParent(IXNode parent) {
    if (parent != null) 
      addParent(Symbol.intern(((Named)parent).getName()));
  }
  /**
   * Adds a list of UIObjectClass to super classes.
   */
  public void addParents(List parents) {
    //make a list of symbols, add them all at once, then fire
    LinkedListOfSymbol newNames = collectClassNames(parents);
    List oldNames = new LinkedListOfSymbol(getSuperClassNames());
    newNames.removeAll(oldNames); //don't add things that are already there
    oldNames.addAll(newNames);
    setSuperClassNames(oldNames);
  }

  public void removeParent(Symbol parent) {
    if (superClassNames.contains(parent)) {
      List oldNames = new LinkedListOfSymbol(superClassNames);
      superClassNames.remove(parent);
      fireDataChange("superClassNames", oldNames, superClassNames);
      updateSuperClasses();
    }
  }

  public void removeParent(String parent) {
    removeParent(Symbol.intern(parent));
  }
  public void removeParent(IXNode parent) {
    removeParent(Symbol.intern(((Named)parent).getName()));
  }

  /**
   *
   */
  protected void tidyParents(Symbol oldN, Symbol newN) {
    List parentNames = getSuperClassNames();
    if (parentNames == null) return;
    for (Iterator i = parentNames.iterator(); i.hasNext(); ) {
      //find parent class, get child names, replace the symbols
      Object parentName = i.next();
      Object parent = uiDomain.getNamedObjectClass(parentName.toString());
      if (parent != null) {
        List children = ((UIObjectClass)parent).getSubClassNames();
	if (children != null) {
          int index = children.indexOf(oldN);
	  if (index != -1) children.set(index, newN);
	}  
      }
    }
  }
  /**
   *
   */
  protected void tidyChildren(Symbol oldN, Symbol newN) {
    List childNames = getSubClassNames();
    if (childNames == null) return;
    for (Iterator i = childNames.iterator(); i.hasNext(); ) {
      //find child class, get parent names, replace the symbols
      Object childName = i.next();
      Object child = uiDomain.getNamedObjectClass(childName.toString());
      if (child != null) {
        List parents = ((UIObjectClass)child).getSuperClassNames();
	if (parents != null) {
          int index = parents.indexOf(oldN);
	  if (index != -1) parents.set(index, newN);
	}  
      }
    }
  }

  public static LinkedListOfSymbol collectClassNames(List objects) {
    LinkedListOfSymbol names = new LinkedListOfSymbol();
    for (Iterator i = objects.iterator(); i.hasNext(); ) {
      UIObjectClass object = (UIObjectClass)i.next();
      if (object != null) names.add(Symbol.intern(object.getName()));
    }
    return names;
  }

  public void setSubClassNames(List newValue) {
    List oldValue = subClassNames;
    if (newValue == null) subClassNames = new LinkedListOfSymbol();
    else subClassNames = new LinkedListOfSymbol(newValue);
    subClassNames.remove(Symbol.intern(""));
    fireDataChange("subClassNames", oldValue, newValue);
  }
  public void addChild(Symbol child) {
    if (!subClassNames.contains(child) && 
	(child != null) && (child != Symbol.intern(""))) {
      List oldNames = new LinkedListOfSymbol(subClassNames);
      subClassNames.add(child);
      fireDataChange("subClassNames", oldNames, subClassNames);
    }
  }
  public void addChild(String child) {
    if ((child != null) && !child.equals(""))
	addChild(Symbol.intern(child));
  }
  public void addChild(IXNode child) {
    addChild(Symbol.intern(((Named)child).getName()));
  }
  /**
   * Adds a list of UIObjectClass to sub classes.
   */
  public void addChildren(List children) {
    //make a list of symbols, add them all at once, then fire
    LinkedListOfSymbol newNames = collectClassNames(children);
    List oldNames = new LinkedListOfSymbol(getSubClassNames());
    newNames.removeAll(oldNames); //don't add things that are already there
    oldNames.addAll(newNames);
    setSubClassNames(oldNames);
  }
  public void removeChild(Symbol child) {
    if (subClassNames.contains(child)) {
      List oldNames = new LinkedListOfSymbol(subClassNames);
      subClassNames.remove(child);
      fireDataChange("subClassNames", oldNames, subClassNames);
    }
  }
  public void removeChild(String child) {
    removeChild(Symbol.intern(child));
  }
  public void removeChild(IXNode child) {
    removeChild(Symbol.intern(((Named)child).getName()));
  }

  public void setObjectProperties(List newValue) {
    //Debug.noteln("UIOC: setting object properties for", this);
    List oldValue = objectProperties;
    propertyMap.clear();
    if (newValue == null) objectProperties = new LinkedListOfObjectProperty();
    else {
      objectProperties = new LinkedListOfObjectProperty(newValue);
      for (Iterator i = newValue.iterator(); i.hasNext(); ) {
	ObjectProperty prop = (ObjectProperty)i.next();
	propertyMap.put(prop.getName(), prop);
      }
    }
    fireDataChange("objectProperties", oldValue, newValue);
  }
  public void addProperty(Symbol property, ObjectProperty.Syntax type) {
    if (property == null) return;
    if (propertyMap.containsKey(property))
      handlePropertyOverwrite(property, type, 
			      (ObjectProperty)propertyMap.get(property));    
    else {
      ObjectProperty prop = new ObjectProperty();
      prop.setName(property);
      if (type != null) prop.setSyntax(type);
      addToProps(prop);
    }
  }
  private void addToProps(ObjectProperty prop) {
    List oldValue = new LinkedListOfObjectProperty(getObjectProperties());
    objectProperties.add(prop);
    propertyMap.put(prop.getName(), prop);
    fireDataChange("objectProperties", oldValue, objectProperties);
  }
  public void addProperty(String property, ObjectProperty.Syntax type) {
    addProperty(Symbol.intern(property), type);
  }
  public void addProperty(String property, String type) {
    addProperty(Symbol.intern(property), ObjectProperty.Syntax.valueOf(type));
  }
  public void addProperty(Symbol property) {
    addProperty(property, null);
  }
  public void addProperty(String property) {
    addProperty(Symbol.intern(property));
  }
  public void addProperty(ObjectProperty property) {
    //Debug.noteln("UIOC: adding property");
    addProperty(property.getName(), property.getSyntax());
  }

  public void replaceProperty(ObjectProperty oldProp, ObjectProperty newProp) {
    List oldValue = new LinkedListOfObjectProperty(getObjectProperties());
    int index = objectProperties.indexOf(oldProp);
    objectProperties.remove(index);
    objectProperties.add(index, newProp);
    propertyMap.remove(oldProp.getName());
    propertyMap.put(newProp.getName(), newProp);
    fireDataChange("objectProperties", oldValue, objectProperties);
    
  }

  public void removeProperty(Object property) {
    if (property instanceof ObjectProperty) {
      List oldValue = new LinkedListOfObjectProperty(getObjectProperties());
      objectProperties.remove(property);
      propertyMap.remove(((ObjectProperty)property).getName());
      fireDataChange("objectProperties", oldValue, objectProperties);
    }
    else UIUtil.notImplemented(null,"UIObjectClass: Removing properties");    
  }

  private boolean isLoading() {
    return getUIDomain().isLoading();
  }
      
  private void handlePropertyOverwrite(Symbol newName, 
				       ObjectProperty.Syntax type,
				       ObjectProperty oldProp) {
    //a property with the name already exists. Ask overwrite

    String messageStart = "There already is a property " + newName.toString();

    ObjectProperty.Syntax oldType = oldProp.getSyntax();
    if (IVUtil.sameObject(type, oldType)) {//trying to add same prop again
      if (!isLoading()) {
	String[] message = {messageStart + " [" + type.toString() + "]", 
			    "Nothing done."};
	JOptionPane.showMessageDialog(null, message);
      }
    }
    else { //different prop with same name
      int y = 0; //replace is default when loading
      if (!isLoading()) {
	Object[] options = {"Replace", "Change Name", "Cancel"};
	String[] message = {messageStart + " Do you want to",
			    "Replace the existing property?",
			    "Change the name of the new property?",
			    "Cancel, i.e. ignore the new property" 
			    + " and keep the exitsting one?"};
	y = JOptionPane.showOptionDialog(null, message, "WARNING: Name clash",
					 JOptionPane.DEFAULT_OPTION, 
					 JOptionPane.WARNING_MESSAGE,
					 null, options, options[0]);
      }

      if (y == 0) { //replace
	ObjectProperty newProp = new ObjectProperty();
	newProp.setName(newName);
	newProp.setSyntax(type);
	replaceProperty(oldProp, newProp);
      }
      else if (y == 1) { //get new name and start again
	String m = "Please enter the new name of the property";
	String answer = JOptionPane.showInputDialog(m);
	if (answer != null) addProperty(answer, type);	  
      }
      //do nothing on cancel
    }
  }   

  //-----------------------UIObject things------------------------------------
  public Domain getDomain() {
    return uiDomain.getDomain();
  }
  public UIDomain getUIDomain() {
    return uiDomain;
  }

  public void moveToUIDomain(UIDomain uiDomain) {
    if (this.uiDomain == null) this.uiDomain = uiDomain;
    else { //**this should be more difficult - what does it mean?
      this.uiDomain = uiDomain;
      uiDomain.addedObject(this, getBaseObject());
    }
  }


  public Object getBaseReference() {
    return baseReference;
  }
  public void setBaseReference(Object ref) {
    baseReference = (String) ref;
  }


  public IXObject getBaseObject() {
    return baseObject;
  }
  public IXObject makeBaseObject() {
    ObjectClass oc = new ObjectClass();
    oc.setName(name.toString());
    return oc;
  }
  public void setBaseObject(IXObject object) {
    baseObject = (ObjectClass)object;
    baseReference = baseObject.getName();
  }
  public Class getBaseClass() {
    return ObjectClass.class;
  }

  public boolean hasOriginal() {
    if ((getBaseReference() == null) || (getBaseReference().equals("")))
      return false;
    else return true;
  }


  /**
   * Saves this object into a suitable baseObject. Does not add the base object
   * to the domain.
   */
  public void saveToDomain() {
    if (baseObject == null) baseObject = (ObjectClass)makeBaseObject();
    //Debug.noteln("UIR-saveToDomain: ", baseObject.toString());
    saveToDomain(baseObject);
    oldName = name;
    //Debug.noteln(" nodes now:", UIUtil.listToDisplay(baseObject.getNodes()));
  }
  public void saveToDomain(Object domainObject) {
    ObjectClass oc = (ObjectClass) domainObject;
    oc.setName(getName()); //name is set when making base object
    oc.setSuperClassNames(superClassNames);
    oc.setObjectProperties(objectProperties);
    oc.setAnnotations(getAnnotations());
  }

  public void addToDomain() { addToDomain(getDomain()); }
  public void addToDomain(Domain domain) {
    //If there is no base-object, make a new one.
    if (baseObject == null) baseObject = (ObjectClass)makeBaseObject();

    //if there already is such an object, compare and complain if not equal

    
    //copy this into the base object (to be safe)
    saveToDomain();

    String name = baseObject.getName();
    ObjectClass oc = domain.getNamedObjectClass(name);
    if (oc != null) {
      if (baseObject.equals(oc)) return; //nothing to be done
      else {
	Debug.noteln("UIOC: Trying to add object class that is already there");
	saveToDomain(oc);
	return;
      }
    }

    // then pass the base object to the domain.
    domain.addObjectClass(baseObject);
  }

  public void updateInDomain(Domain domain) throws DomainReferenceException {
    //if there is no base object, complain (add instead?)
    String ref = (String)getBaseReference();
    ObjectClass oc = getReferredObject(domain, ref, getName());
    if (oc == null) {
      Debug.noteln("UIOC -Update object class: cannot find base object", ref);
      Debug.noteln(" Domain is", domain);
      return;
    }
    //copy this into base object (to be safe) and pass it with its id to the
    //domain
    //Debug.noteln("UIOC: Update refinement base is ", ref);
    if (isEmpty()) domain.deleteNamedObjectClass(oc.getName());
    else {
      saveToDomain();
      domain.replaceObjectClass(oc, baseObject);
    }
  }

  /**
   * Loads this object from its baseObject. 
   * ***Think about undoing this!!
   */
  public void loadFromDomain() {
    if (baseObject == null) {
      baseObject = (ObjectClass)makeBaseObject();
    }
    name = Symbol.intern(baseObject.getName());
    setSuperClassNames(baseObject.getSuperClassNames());
    updateSuperClasses();
    setObjectProperties(baseObject.getObjectProperties());
    setAnnotations(baseObject.getAnnotations());
    fireDataChange("all", null, null);
  }
  public void loadFromDomain(Object domainObject) {
    baseObject = (ObjectClass)domainObject;
    loadFromDomain();
  }

  public void removeFromDomain() throws DomainReferenceException { 
    removeFromDomain(getDomain()); 
  }
  public void removeFromDomain(Domain domain) throws DomainReferenceException {
    //if there is no base object id, try to find the id in the domain
    //Complain if it cannot be found
    String ref = (String)getBaseReference();
    //Debug.noteln("UIOC: removing refinement " + ref + " or", getName());
    ObjectClass ot = getReferredObject(domain, ref, getName());
    if (ot == null) {
      Debug.noteln("Remove ObjectClass: Cannot find base object called ",ref);
      return;
    }
    //give the object to the domain and ask for removal
    //Debug.noteln("...deleting");
    domain.deleteObjectClass(ot);
  }

  private ObjectClass getReferredObject(Domain domain, 
					String reference, String name) 
    throws DomainReferenceException
  {
    String sRef;
    if ((reference == null) || (reference.equals(""))) 
      sRef = name;
    else sRef = reference;
    ObjectClass found = domain.getNamedObjectClass(sRef);
    if (found == null) {
      List all = domain.getObjectClasses();
      if (((all == null) || all.isEmpty())
	  && IXTrees.ROOTNAME.equals(name))
	return null;
	  
      throw new DomainReferenceException("Cannot find object Class " + sRef +
					 " in domain " + domain.toString() +
					 " with classes " + 
					 IVUtil.namedListToDisplay(all));
    }
    else 
      return found;
  }


  public void loadFromOriginal() {
    if (hasOriginal()) {
      Object ref = getBaseReference();
      //Debug.noteln("UIOC: base ref is", ref);
      ObjectClass original = getDomain().getNamedObjectClass((String)ref);
      try {
	setBaseObject((ObjectClass)original.clone());
	loadFromDomain();
      }
      catch (Exception e) { //CloneNotSupported or null pointer
	Debug.noteException(e);
	Debug.noteln("Continuing with new object as the original");
	setBaseObject(makeBaseObject());
	loadFromDomain();
      }
    }
    else {
      setBaseObject(makeBaseObject());
      loadFromDomain();
    }
  }



  /**
   * Get the UIDomain to note the change of this object
   */
  public void noteChange() {
    //Debug.noteln("UIOC: noting change for", this);
    if (isEmpty() && !hasOriginal()) 
      uiDomain.removeConstruct(this); //empty with no original. Get rid of it!
    else if ((oldName != null) && nameHasChanged(oldName.toString())) {
      //Debug.noteln("UIOC: got name change");
      tidyParents(oldName, name);
      tidyChildren(oldName, name);
      oldName = name;
      saveToDomain();
      uiDomain.updateConstruct(this);
    }
    else if (hasChanged()) {
      //saving, so update the base object
      saveToDomain();

      //Debug.noteln("UIOC: noting change");
      uiDomain.updateConstruct(this);
    }
  }

  public boolean hasChanged() {
    return hasChanged((ObjectClass)getBaseObject());
  }
  public boolean hasChanged(ObjectClass original) {
    List changes = collectChanges(original);
    return ((changes != null) && !changes.isEmpty());
  }
  public boolean hasChangedFromOriginal() {
    //has no original so it is new (changed!)
    if (!this.hasOriginal()) return true;

    //has original, so compare the two
    ObjectClass original = getDomain().getNamedObjectClass(baseReference);
    return hasChanged(original);
  }

  /**
   * Collects the differences between this object and the base one.
   * @return a list of strings describing changes
   */
  public List collectChanges() {
    return collectChanges((ObjectClass)getBaseObject(), "draft");
  }
  /**
   * Collects the differences between this object and the base one.
   * @return a list of strings describing changes
   */
  public List collectChangesFromOriginal() {
    ObjectClass original = 
      getDomain().getNamedObjectClass(baseReference);
    return collectChanges(original, "published");
  }
  /**
   * Collects the differences between this object and the given one.
   * @return a list of strings describing changes
   */
  public List collectChanges(ObjectClass original) {
      return collectChanges(original, "old");
  }
  public List collectChanges(ObjectClass original, String which) {
    List changes = new ArrayList();
    if (isEmpty(original) && isEmpty()) return changes;
    //dummy created when "editing" (undefined)
    //this does not count as a change
    if (isEmpty(original) 
	&& ((getName() == null) ||
	    (getName().equals("Undefined") || getName().equals("undefined")))
	&& (objectProperties == null || objectProperties.isEmpty()))
      return changes;
    if (original == null) {
	if (isEmpty()) return changes;
	else {
	    changes.add("The " + which + 
			" ObjectClass is empty, this panel is not:");
	    changes.add("  " + print());
	    return changes;
	}
    }
    else {
	if (nameHasChanged(original.getName())) 
	    changes.add("Name has changed from " + original.getName() +
		" to " + getName());
	if (!IVUtil.sameSet(superClassNames,original.getSuperClassNames())) 
	    changes.add("Parent set has changed from " + 
			UIUtil.show(original.getSuperClassNames())
			+ " to " + UIUtil.show(superClassNames));
	if (!IVUtil.sameSet(objectProperties, original.getObjectProperties()))
	    changes.add("Properties have changed from " + 
		UIUtil.show(original.getObjectProperties()) +
		" to " + UIUtil.show(objectProperties));
	if (!IVUtil.sameStableMap(getAnnotations(),original.getAnnotations())) 
	    changes.add("Annotations have changed from " + 
		original.getAnnotations().entrySet() + " to " + 
			getAnnotations().entrySet());
	//Debug.noteln("Changes are:", changes);
    }
   return changes;				   
  }

  public boolean nameHasChanged(Object nameThing) {
    if (nameThing == null) 
      return ((getName() == null) || getName().equals(""));
    String original;
    try { original = (String) nameThing; }
    catch (Exception e) {
      try { original = nameThing.toString(); }
      catch (Exception e2) { original = ""; }
    }
    if ((((getName() == null) || getName().equals("")) &&
	 ((original == null) || (original.equals("")))) ||
	((getName() != null) && getName().equals(original)))
      return false;
    else return true;
  }




  public String toString() {
    return getName();
  }
  public String superString() {
    return super.toString();
  }
  public String print() {
    return "ObjectClass: " + getName() + "\n" +
      " SuperClassNames: " + UIUtil.show(superClassNames) + "\n" +
      " SubClassNames: " + UIUtil.show(subClassNames) + "\n" +
      " ObjectProperties: " + UIUtil.show(objectProperties);
  }



  //-----------------------TreeNode things-------------------------------------

  private static final int NOFIND = -1;
  public Enumeration children() { 
    //nodes = this.getNodes();
    //Debug.noteln("UIOC: children are", UIUtil.show(subClassNames) + ".");
    //Debug.noteln(" length", subClassNames.size());
    subClassNames.remove(null);
    subClassNames.remove("");
    return new Enumeration() {
	/*
	  int i = 0;
	  public boolean hasMoreElements() { return i < nodes.size(); }
	  public Object nextElement() { return nodes.get(i++); }
	*/
	
	int i = subClassNames.size()-1;
	public boolean hasMoreElements() { return i >= 0; }
	public Object nextElement() { 
	  //Debug.noteln(" getting child i", i);
	  //Debug.noteln(" of",  subClassNames.size());
	  String name = subClassNames.get(i).toString();
	  i = i-1;
	  //Debug.noteln(" child is", name);
	  Object el = null;
	  if (name != "") el = uiDomain.getNamedObjectClass(name);
	  //Debug.noteln(" UIOC: next child object", el);
	  //Debug.noteln("  i now", i);
	  return el;
	}
      };
  }
  public List getChildren() { 
    List children = new ArrayList();
    for (Iterator i = subClassNames.iterator(); i.hasNext(); ) 
      children.add(uiDomain.getNamedObjectClass(i.next().toString()));
    return children;
  }
  public boolean getAllowsChildren() { return true; }
  public TreeNode getChildAt(int index) { 
    Object sub = getSubClassNames().get(index);
    return (TreeNode)uiDomain.getNamedObjectClass(sub.toString());
  }
  public int getChildCount() { 
    if (getSubClassNames() == null) return 0;
    else return getSubClassNames().size(); 
  }
  public int getIndex(TreeNode node) {
    if (node instanceof UIObjectClass) {
      List subs = getSubClassNames();
      if (subs == null) return NOFIND;
      else return subs.indexOf(Symbol.intern(((Named)node).getName()));
    }
    return NOFIND;//dummy
  }
  public TreeNode getParent() { //return just the first parent
    List parents = getSuperClassNames();
    if ((parents == null) || (parents.size() == 0))
      return null;
    else {
      //try to find a suitable parent to return
      for (Iterator i = parents.iterator(); i.hasNext();) {
	Object parent = i.next();
	if (parent instanceof TreeNode)
	  return (TreeNode)parent;
      }
      return null; 
    } 
  }
  public List getParents() { //return all parents
      ArrayList parents = new ArrayList();
      List parentNames = getSuperClassNames();
      for (Iterator i = parentNames.iterator(); i.hasNext(); ) {
	parents.add(uiDomain.getNamedObjectClass(i.next().toString()));
      }
      return parents;
  }
  public boolean isLeaf() { return (getChildCount() == 0);}


  public boolean checkLegalParent(UIObjectClass otherNode) {
    return ((otherNode != null) && 
	    !isNodeAncestor(otherNode) &&
	    !otherNode.isNodeAncestor(this));
  }
  public String reportLegalParent(UIObjectClass otherNode) {
    String message = "";
    if (otherNode == null) 
      message = "null is not a legal parent for " + this.toString();
    else if (otherNode.isParentOf(this)) return message; 
    else if (isNodeAncestor(otherNode))
      message = otherNode.toString() + " is not a legal parent of " 
	+ this.toString() + " because it is already an ancestor";
    else if (otherNode.isNodeAncestor(this))
      message = otherNode.toString() + " is not a legal parent of " 
	+ this.toString() + " because it is a descendant";
    return message;
  }
  public String reportLegalChild(UIObjectClass otherNode) {
    String message = "";
    if (otherNode == null) 
      message = "null is not a legal child for " + this.toString();
    else if (isParentOf(otherNode)) return message; 
    else if (isNodeAncestor(otherNode))
      message = otherNode.toString() + " is not a legal child of " 
	+ this.toString() + " because it is an ancestor";
    else if (otherNode.isNodeAncestor(this))
      message = otherNode.toString() + " is not a legal child of " 
	+ this.toString() + " because it is already a descendant";
    return message;
  }

  public boolean isParentOf(UIObjectClass otherNode) {
    if (otherNode == null) return false;
    List otherParents = otherNode.getParents();
    return otherParents.contains(this);
  }

  /**
   * Returns true if the node is this node, or if it is a parent of
   * this node, or if it is an ancestor of one of the parents.
   */
  public boolean isNodeAncestor(TreeNode otherNode) {
    if (this.equals(otherNode)) {
      return true;
    }
    List parents = getParents();
    if (parents.contains(otherNode)) {
      return true;
    }
    else {
      for (Iterator i = parents.iterator(); i.hasNext(); ) {
	if (((UIObjectClass)i.next()).isNodeAncestor(otherNode)) return true;
      }
      return false;
    }
  }

  //-----------------------Listener things------------------------------------


  HashSet dataChangeListeners = new HashSet();

  public void addDataChangeListener(DataChangeListener l) {
    dataChangeListeners.add(l);
  }
  public void removeDataChangeListener(DataChangeListener l) {
    dataChangeListeners.remove(l);
  }

  private void fireDataChange(String field, Object oldValue, Object newValue) {
    //Debug.noteln("UIOC: changing value in " + field + " from", oldValue);
    //Debug.noteln(" to", newValue);
    if (sameValue(field, oldValue, newValue)) {
      //Debug.noteln("UIOC: same field value");
      return; //no change
    }
    //Debug.noteln("UIOC: firing change for", dataChangeListeners);
    for (Iterator i = dataChangeListeners.iterator(); i.hasNext(); ) {
      DataChangeListener l = (DataChangeListener)i.next();
      l.dataChanged(this, field, oldValue, newValue);
    }
  }


}
