/****************************************************************************
 * A construct that mirrors a domain object and adds UI things
 *
 * @author Jussi Stader
 * @version 4.1
 * Updated: Thu Mar  2 11:45:17 2006
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview.domain;

import java.util.*;

import ix.icore.domain.event.*;
import ix.icore.domain.*;
import ix.icore.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.iface.ui.*;
import ix.iview.event.*;

/**
 * A construct that mirrors a domain object and adds UI things.
 * A UIObject has a base object which is its corresponding object in the
 * domain. The contents of this object may change - it is a clone of what
 * is in the domain.
 * The UIObject also has a reference that can be used to identify its
 * corresponding object in the domain (e.g. for a refinement that should
 * be its name). Once this is set, it should not be changed. It will be needed 
 * if the domain has to work out what the original is (in case of remove or
 * update operations)
 */

public interface UIObject extends Named, UIEditableObject {

  /** Gets the base domain. */
  public Domain getDomain();
  /** Gets the ui domain of which this object is part. */
  public UIDomain getUIDomain();
  /** Determines whether the object has an original in the base domain. */
  public boolean hasOriginal();
  /** Determines whether the object has any information in it. */
  public boolean isEmpty();

  /** Determines whether the object has changed from its original. */
  public boolean hasChangedFromOriginal();
  /**
   * Determines whether the object has changed since it was last saved to draft
   */
  public boolean hasChanged();
  /**
   * Collects the object's changes since it was last saved to draft.
   * @return list of strings describing the changes, {} if none.
   */
  public List collectChanges();
  /**
   * Collects the object's changes since it was last published.
   * @return list of strings describing the changes, {} if none.
   */
  public List collectChangesFromOriginal();
  /**
   * Sets the given field to the given value. Should default to 
   * UIUtil.setObjectFieldValue(Object object, String field, Class fieldClass,
   * Object value).
   */
  public boolean setValue(String field, Object value);
  /**
   * Determines whether the two given values of the given field differ
   */
  public boolean sameValue(String field, Object value, Object otherValue);
  /**
   * Determines whether the object's identifying fields have changed
   */
  public boolean nameHasChanged(Object nameThing);
  /**
   * Gets the Id that can be used to identify the base object in the domain. 
   */
  public Object getBaseReference();
  /**
   * Sets the Id that can be used to identify the base object in the domain. 
   */
  public void setBaseReference(Object id);

  /** Gets the object that corresponds to this one in the domain. */
  public IXObject getBaseObject();
  /**
   * Makes an empty object of the right class that can represent this UIObject
   * in the domain.
   */
  public IXObject makeBaseObject();
  /** Sets the object that will represent this one in the domain. */
  public void setBaseObject(IXObject object);
  /** Gets the class of objects that can represent this one in the domain. */
  public Class getBaseClass();


  //-------------------Saving and loading-----------------------------------
  /**
   * Saves the contents of this object into its corresponding domain object.
   * This can be done without worries about the original because if the
   * UIObject has a base object, it is always a clone of the original.
   */
  public void saveToDomain();
  /**
   * Saves the contents of this object into the given object.
   */
  public void saveToDomain(Object domainObject);
  /**
   * Loads the contents of this object from its corresponding domain object.
   */
  public void loadFromDomain();
  /**
   * Loads the contents of this object from the given object.
   * The given object should be of the right class (same as the result of 
   * getBaseClass()).
   */
  public void loadFromDomain(Object domainObject);

  /**
   * Loads the contents of this object from its original domain object.
   */
  public void loadFromOriginal();

  /** Get the UIDomain to note the change of this object. */
  public void noteChange();


  //-------------------Changing original domain-----------------------------
  /**
   * Adds the contents of this UIObject to the domain (part of publish).
   */
  public void addToDomain(Domain domain);
  /**
   * Updates the base object in the domain with the contents of this object.
   */
  public void updateInDomain(Domain domain);
  /**
   * Removes the object's corresponding base object from the domain.
   */
  public void removeFromDomain(Domain domain);

  /** Move the object from its current UIDomain to the given one. */
  public void moveToUIDomain(UIDomain uiDomain);



  //public String getName();
  public void setName(String name);

  public String print();

}


// Issues:
// * 
// * 
// * 
