/****************************************************************************
 * A simple text-input editor for property specifications, e.g. for
 * object classes.
 *
 * @author Jussi Stader
 * @version 3.1
 * Updated: Mon Oct  2 10:51:29 2006
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import javax.swing.*;       
import javax.swing.text.JTextComponent;       
import java.util.*;
import java.awt.Color;
import java.awt.Frame;
import java.awt.Component;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
import ix.*;
import ix.util.*;
import ix.util.Debug;
import ix.util.lisp.Symbol;
import ix.util.lisp.Lisp;
import ix.icore.domain.*;
import ix.icore.*;
import ix.iview.util.*;
import ix.iview.domain.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;

/****************************************************************************
 * A simple text-input editor for entering property names with a list
 * for the syntax type. Used for ObjectClass property lists. It is
 * like a JOptions dialogue, but it has two entry parts (string and
 * list) and an "Add More" button.
 *
 * The editor can be created with no arguments or with a given ObjectClass
 * which will be set as the current ObjectClass for which the properties
 * are to be defined. Subsequently, the ObjectClass in the editor can be set
 * using the setObjectClass method.
 *****************************************************************************
 */
public class PropertyEditor extends JConstraintEditor 
{
  
  public String message;
  private JTextComponent nameField;
  protected JList syntaxList;

  /**
   * Creates an editor for specifying time point relations.
   * The editor has components for selecting two time points of two
   * sub-activities, and two buttons (Ok, Cancel)
   */
  public PropertyEditor(Frame owner) {
    this(owner, "Please enter the property specification.");
  }
    
  /**
   * Creates an editor for specifying time point relations.
   * The editor has components for selecting two time points of two
   * sub-activities, and two buttons (Ok, Cancel)
   */
  public PropertyEditor(Frame owner, String message) {
    super(owner, "Property Editor", message);
    this.setSize(305, 220);
  }
    
  protected Component makeBits() {return makeRels();}
  
  /**
   * Makes all items on the relations panel - just a text field and a list.
   */
  protected Component makeRels() {
    nameField = new JTextField();
    syntaxList = new JList(new Vector(ObjectProperty.Syntax.values()));
    ListEditorPanel syntPanel = new ListEditorPanel(this,"Syntax",syntaxList);
    ThingEditorPanel namePanel = new ThingEditorPanel(this,"Name",nameField);

    syntaxList.setAutoscrolls(true);
    syntaxList.setVisibleRowCount(6);
    syntaxList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    syntaxList.setSelectedIndex(1);


    Box box = new Box(BoxLayout.Y_AXIS);
    box.add(namePanel);
    box.add(syntPanel);

    return box;
  }

  public void setText(String text) {
    nameField.setText(text);
  }

  protected JComponent getFocusComponent() {
    return nameField;
  }

  public void start() {
    setVisible(true);
    initFromObject();
  }


  protected List getPreviousList(Object constraint) {
    List props = ((UIObjectClass)currentObject).getObjectProperties();
    //Debug.noteln("PropEd: previous list", props);
    return new LinkedListOfObjectProperty(props);
  }  
  protected String getField(Object constraint) { return "objectProperties";}

  
  /**
   * Nothing to do to initialise the editor from an object class
   */
  protected void initFromObject() {
    setText("");
    nameField.requestFocus();
  }
  /**
   * when editing, put text in and select syntax
   */
  protected void setFromObject(Object original) {
    if (original instanceof ObjectProperty) {
      setText(((ObjectProperty)original).getName().toString());
      syntaxList.clearSelection();
      syntaxList.setSelectedValue(((ObjectProperty)original).getSyntax(), 
				  true);
      nameField.requestFocus();
    }
  }


  /**
   * Reads the selected relation and let interested parties know.
   * Checks that selections have been made.
   */
  protected Object collectConstraint() {
    //Debug.noteln("PropEd: collecting prop for object", currentObject);
    String name = nameField.getText().trim();
    if ((name == null) || name.equals("")) return null;
    else {
      //Debug.noteln("PropEd: Got prop:", name);
      ObjectProperty prop = new ObjectProperty();
      if (isEditing()) { //"clone" original so class notices change
	ObjectProperty oProp = (ObjectProperty)original;
	prop.setValueClassName(oProp.getValueClassName());
	//name and syntax from current selections!
      }
      prop.setName(Symbol.intern(name));
      Object syn = syntaxList.getSelectedValue();
      //Debug.noteln(" selected syntax is", syn);
      //Debug.noteln(" class", syn.getClass());
      ObjectProperty.Syntax syntax = null;
      if (syn == null) syntax = prop.getSyntax();
      else syntax = (ObjectProperty.Syntax)syn;
      if (syntax == null) syntax = ObjectProperty.Syntax.DEFAULT;
	
      prop.setSyntax(syntax);
      //Debug.noteln("PropEd: collected", prop);
      return prop;
    }
  }

  /**
   * make UIO update properly so that the event is noted elsewhere (panels..)
   * Overwrites JConstraintEditor to allow for handling overwrite
   */
  protected Object noteNewValue(Object constraint) {
    //Debug.noteln("PropEd: - noting value", constraint);
    if (isEditing())     //as usual: remove original, add new
      return super.noteNewValue(constraint);
    else {
      //Debug.noteln("PropEd: new constraint - noting value");
      ((UIObjectClass)currentObject).addProperty((ObjectProperty)constraint);
      return constraint;
    }
  }

}

