/****************************************************************************
 * An editor for specifying (world-state) constraints on objects in
 * refinements.
 *
 * @author Jussi Stader
 * @version 3.1
 * Updated: Mon Jul 31 11:25:26 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import javax.swing.*;       
import javax.swing.event.*;       
import javax.swing.table.*;       
import java.util.*;
import java.awt.Color;
import java.awt.Frame;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;

import ix.*;
import ix.util.*;
import ix.util.Debug;
import ix.util.lisp.*;
import ix.icore.domain.*;
import ix.iview.util.*;
import ix.iview.domain.*;
import ix.iview.tree.*;
import ix.iview.table.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.ui.table.*;
import ix.iface.domain.*;
import ix.ip2.Ip2ModelManager;

/****************************************************************************
 * An editor for specifying (world-state) constraints on objects in
 * refinements.
 *
 * The editor is implemented as a frame using a JTextField, a JCheckBox, 
 * and a JTable.  <p>
 *
 * The editor can be created with no arguments or with a given refinement
 * which will be set as the current refinement whose object constraints
 * are to be specified. Subsequently, the refinement in the editor can be set
 * using the setRefinement method.
 *****************************************************************************
 */
public class ObjectConstraintEditor extends ConstraintEditor 
  implements ListSelectionListener, MouseListener, PropertyConstraintsListener
{

  //relation items

  //  protected ThingEditorPanel varNameEd;
  protected ListEditorPanel varNameEd;
  protected ListEditorPanel classEd;
  protected ListEditorPanel propConstraintEd;
  protected ToggleEditorPanel propModePan;

  private PropertyTableModel tableModel;
  private JComboBox vasBox;

  protected UIDomain uiDomain;
  protected UIObjectClass objectClass; //the variable's object class
  private List oldConstraints;
  protected Constraint typeConstraint; //the variable's object class constraint
  protected ItemVar variable; //the current variable

  //ObjectProperty, ConstraintSet map for a variable of the class
  protected HashMap propertyConstraints = new HashMap(); 
  //variable - propertyConstraints (ItemVar - HashMap map)
  protected HashMap varConstraints = new HashMap(); 
  
  private int mode = ALL;
  private ObjectClassTreeNode root;


  
  /**
   * Creates an editor for specifying object constraints.
   * The editor has components for specifying a variable, its type, and 
   * a set of constraints on the variable.
   * The editor will always be  brought up within the context of a
   * refinement; This method is preparation only.
   * To work with the editor:
   * "- select or enter variable name (starting with ?)"
   * "- select or enter object class of the variable,",
   * "- select or enter property to be constrained,"
   * "- enter constraint details (value, cond/eff, begin/end,"
   * "- select anoter property or finish with the variable"
   * "- select anoter variable or finish with the editor"
   */
  public ObjectConstraintEditor(Frame owner, UIDomain domain){
    super(owner, "Object Constraint Editor", 
	  "Please specify object constraints");
    //Debug.noteln("OCE: got domain", domain.print());
    setDomain(domain);
  }
    
  /**
   * Creates an editor for specifying object constraints within the given
   * refinement.
   * The editor will come up with the current variables and constraints
   * in place. Otherwise as above.
   * @param refinement the refinement whose objects are to be consrained. 
   */
  public ObjectConstraintEditor(Frame owner, UIRefinement refinement) {
    this(owner, refinement.getUIDomain());
    setObject(refinement);
  }
  
    //---------------------- Building Window -------------------------------
  /**
   * Makes all items on the relations panel. Items are:
   * - Prompt (label)
   * - Variable name (text field)
   * - Type (combo box)
   * - property constraints (table): 
   *    - property, syntax, own-flag, value, nodeEnd, cond/effect
   * - property diplay mode (own vs ancestors)
   */
  protected Component makeRels() {
    //make a panel that can take all the bits
    JPanel panel = new JPanel(new GridBagLayout());

    makeVarPanel();
    makeClassPanel();
    makePropModePanel(); //mode toggles (own/ancestor properties)
    makePropPanel();
    
    //make the activity list panel
    //actList = new JList();    
    //actEd = makeActPanel(new JList()); if swapping, change gridbag constr
    //actEd = makeActPanel(new JComboBox()); //attach to self


    
    //add it all to the panel
    GridBagConstraints c;
    if (label == null) label = new JLabel();
    c = new GridBagConstraints(0, 0, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.NORTHWEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(label, c);
    c = new GridBagConstraints(0, 1, //x,y
			       1, 1, //width height
			       1.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.HORIZONTAL,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(varNameEd, c);
    
    c = new GridBagConstraints(1, 1, //x,y
			       1, 1, //width height
			       1.0,0.0,  //weight x,y
			       GridBagConstraints.EAST, //anchor
			       GridBagConstraints.HORIZONTAL,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(classEd, c); //class

    
    c = new GridBagConstraints(0, 2, //x,y
			       2, 2, //width height
			       1.0,1.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.BOTH,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(propConstraintEd, c); //class

    
    c = new GridBagConstraints(0, 4, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(propModePan, c); //class

    
    /*
    c = new GridBagConstraints(1, 5, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       //GridBagConstraints.BOTH,  //fill
			       GridBagConstraints.HORIZONTAL,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(actEd, c);//attach to self
    */

    return panel;
  }

  /*
   * Make a sub-panel with the given activity list/box to be filled later
   * @param list the JList or JComboBox that will hold the activities
   * Uses a NodeNameRenderer to display the list items
   * @return the new panel
   */
  private ListEditorPanel makeActPanel(JComponent list){
    ListEditorPanel actEd = new ListEditorPanel(this,"Attach to node",list);
    if (list instanceof JList) {
	((JList)list).setAutoscrolls(true);
	((JList)list).setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    }
    actEd.setRenderer(new NodeNameRenderer(false));
    return actEd;
  }
  /*
   * Make a sub-panel for specifying the variable name (with help)
   * @return the new panel
   */
  private IXEditorPanel makeVarPanel() {
    Set vars = getVariables();
    JComboBox varBox;
    if (vars == null)
      varBox = new JComboBox();
    else {
      varBox = new JComboBox(vars.toArray());
    }
    //Debug.noteln("got variables in", varBox.getModel().getSize());
    varBox.setEditable(true);
    varNameEd = new ListEditorPanel(this,"Variable (e.g. ?x)", varBox);
    //list.setBackground(Color.white);
    varBox.setSelectedItem(" ");
    PopupMenuListener pml = 
      new PopupMenuListener() {
	private final ObjectConstraintEditor oce = ObjectConstraintEditor.this;
	public void popupMenuCanceled(PopupMenuEvent e) {}
	public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {}
	public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
	  oce.varNameEd.setData(oce.getVariables());
	  //oce.varNameEd.setSelectedObject(oce.variable);
	}
      };
    varBox.addPopupMenuListener(pml);
    return varNameEd;
  }
  /*
   * Make a sub-panel for specifying the variable name (with help)
   * @return the new panel
   */
  private IXEditorPanel makeTextVarPanel() {
    JTextField specField = new JTextField();
    specField.setText(" ");//else var-offering does not happen
    specField.addFocusListener(new FocusAdapter() {
	public void focusLost(FocusEvent fEvent) {
	  //if (!fEvent.isTemporary()) //only listen if focus within window
	  //Debug.noteln("OCE: focus lost to", fEvent.getOppositeComponent());
	  updateVariable();
	};
      });
    IVUtil.enableVars(this, specField);
    ThingEditorPanel varNameEd = 
      new ThingEditorPanel(this,"Variable name",specField);
    //list.setBackground(Color.white);
    specField.setText("");//else var-offering does not happen
    return varNameEd;
  }
  /*
   * Make a sub-panel for specifying the object class
   * @return the new panel
   */
  private IXEditorPanel makeClassPanel() {
    List classes = getObjectClasses();
    //Debug.noteln("OCE: got classes", UIUtil.show(classes));
    //JList box;
    JComboBox classBox;
    if (classes == null)
      classBox = new JComboBox();
    else
      classBox = new JComboBox(classes.toArray());
    //classBox.setEditable(true);
    //classBox.setSelectedItem("Thing");
    //Debug.noteln("got class data in", classBox.getModel().getSize());
    classEd = new ListEditorPanel(this,"Type (Object Class)", classBox);
    //list.setBackground(Color.white);
    updateClassEdSelection(); //selects root if possible and sets objectClass
    return classEd;
  }

  /*
   * Make a sub-panel for specifying the object class
   * @return the new panel
   */
  private IXEditorPanel makePropPanel() {
    //List properties = getObjectProperties(objectClass);
    //propConstraintEd = 
    //new ListEditorPanel(this,"Property Constraints", new JList());

    tableModel = new PropertyTableModel(); //not markable
    IXTable table = new IXTable(tableModel);
    tableModel.adjustTable(table);

    propConstraintEd = new ListEditorPanel(this,"Property Constraints", table);
    tableModel.setObjectClass(objectClass);
    tableModel.addPropertyConstraintsListener(this);
    return propConstraintEd;
  }

  private ToggleEditorPanel makePropModePanel() {
    JToggleButton ownBut = new JCheckBox("Own", true);
    JToggleButton ancestBut = new JCheckBox("Ancestors", true);
    ownBut.setActionCommand("propMode");
    ancestBut.setActionCommand("propMode");
    JToggleButton[] buts = {ownBut, ancestBut};
    //IXButPanel propModePan = new IXButPanel(this, BoxLayout.Y_AXIS, buts);
    propModePan = new ToggleEditorPanel(this, "Show Properties", buts);
    return propModePan;
  }

  protected JComponent getFocusComponent() {
    return varNameEd.getItemComponent();
  }


  private Set getVariables() {
    if (currentObject == null) return null;
    return ((UIRefinement)currentObject).getVariablesUsed();
  }

    //---------------------- Managing Data -------------------------------

  boolean isAdjusting = false;

  /**
   * Set up the editor to specify more of the same (add more button)
   * This is disabled for the editor - all constraints are done here anyway
   */
  public void moreInitFromObject() throws NullPointerException {
  }

  /**
   * Sets the refinement whose objects are to be constrained.
   * ********revise this
   * @throws NullPointerException if *******
   */
  protected void initFromObject() throws NullPointerException {
    isAdjusting = true;
    super.moreBut.setEnabled(false);

    setPropertyMode(ALL);
    
    UIRefinement refinement = (UIRefinement) currentObject;
    if (refinement == null) return;

    oldConstraints = refinement.getConstraints();
    Debug.noteln("OCE noted old constraints");

    List children = new LinkedList(refinement.getNodes());
    children.add(0, refinement);
    // Debug.noteln("Children are", children);
    makeActList(children.toArray());
    variable = null;
    //Debug.noteln("OCE: done children");
    updateClasses();
    //actList.clearSelection();
    //actList.setSelectedIndex(0);
    //children.remove(0);
    //statement.requestFocus();
    isAdjusting = false;
    focus();
  }

  /**
   * Set the editor for editing a constraint (not putting in a new one)
   */
  protected void setFromObject(Object original) {
      //actList.clearSelection();
    if (original == null) return;
    if (!(original instanceof Constraint)) {
      String message = "Condition/Effect editor can only deal with constraints"
	+ " not objects of class " + original.getClass().getName();
      JOptionPane.showMessageDialog(this,message);
      return;
    }

    Constraint constraint = (Constraint)original;
    if (constraint.getType() != Refinement.S_WORLD_STATE) {
      String message = "Condition/Effect editor cannot deal with constraints"
	+ " Of type " + constraint.getType().toString();
      JOptionPane.showMessageDialog(this,message);
      return;
    }
    propConstraintEd.setSelectedObject(constraint);
    //actEd.setSelectedIndex(0);
    //actList.clearSelection();
    //actList.setSelectedIndex(0); //self only
    
    //statement.setText(UIUtil.listToDisplay(constraint.getParameters()));
    //statement.requestFocus();
  }

  /**
   * Set the object (UIRefinement) whose objects are to be constrained
   */
  public void setObject(Object o) {
    clear();
    if (o instanceof UIObject) 
      setDomain(((UIObject)o).getUIDomain());
    super.setObject(o);
  }

  /* Strictly private
   * shorthand only - sorts out domain listening stuff
   */
  private void setDomain(UIDomain domain) { 
    if (uiDomain != null) {
      if (!uiDomain.equals(domain)) {
	  ;//uiDomain.removeDomainListener(this);
      }
    }
    uiDomain = domain;
    if (uiDomain != null)
	;//uiDomain.addDomainListener(this);
  }

  public void clear() { //remember *not* to clear domain
    variable = null;
    varNameEd.removeData();
    updateVariable();
  }

  /**
   * Get the object classes from the domain and put them into the classEditor 
   */   
  private void updateClasses() {
    //Debug.noteln("OCE: update classes");
    List data = getObjectClasses();
    //Debug.noteln(" classes are", UIUtil.show(data));
    data.add("new Class");
    classEd.setData(data);
    updateClassEdSelection();
  }

  //selects root if possible and sets objectClass to selection.
  //Called during init and when classes are reloaded.
  private void updateClassEdSelection() { //make the right selection
    if ((root != null) && (root.getUserObject() != null))
      classEd.setSelectedObject(root.getUserObject());
    else classEd.setSelectedIndex(0);
    Object o = classEd.getSelectedObject();
    if (o instanceof UIObjectClass) objectClass = (UIObjectClass)o;
    else {
      objectClass = null;
      if (o != null) 
	Debug.noteln("OCE: object class of wrong type:", o.getClass());
      else Debug.noteln("OCE: object class is", o);
    }
    noteType();
    //do following in case variable is set
    if (tableModel != null) tableModel.setObjectClass(objectClass);
  }
  /**
   * Make the table take account of the currently selected UIObjectClass
   * **********move into table? 
   */   
  private void updateProperties() {
    Object currentClass = classEd.getSelectedObject();
    //Debug.noteln("OCE: updating props for", currentClass);
    if (currentClass instanceof UIObjectClass) {
      //Debug.noteln("OCE: Selected object class", currentClass);
      tableModel.setObjectClass((UIObjectClass)currentClass);
    }
    else {
      Debug.noteln("Selected object class of type", currentClass.getClass());
    }
    //tableModel.setObjectClass(currentClass);
  }


  protected void updateVariable() {
    Object varOb = varNameEd.getSelectedObject();
    //find and set current variable (may be null)
    if (varOb == null) variable = null;
    else if (varOb instanceof String) {
      String varName = varOb.toString().trim();
      //Debug.noteln("OCE var name is:", varName + ".");
      try {
	variable = (ItemVar)Symbol.intern(varName);
      }
      catch (Exception e) {
	variable = null;
      }
    }
    else if (varOb instanceof ItemVar) variable = (ItemVar)varOb;
    else {
      Debug.noteln("OCE: problem with variable of class", varOb.getClass());
      variable = null;
    }

    //variable gets updated here
    loadObjectConstraints(variable, (UIRefinement)currentObject);
    if (tableModel != null) {
      tableModel.setConstraints(variable, objectClass, propertyConstraints);
    }
  }

  public void variableConstraintsChanged(ItemVar variable, 
					 HashMap propConsts) {
    Debug.noteln("OCE: got new constraints for", variable);
    Debug.noteln("  :", 
		 IVUtil.printConstraints(UIUtil.getFlatValues(propConsts)));
    varConstraints.put(variable, propConsts);
  }
  public void variableNameChanged(ItemVar oldVar, ItemVar newVar) {
    Debug.noteln("OCE: ************Cannot deal with var name change*********");
  }
  //should do some checking and filtering?
  public void variableTypeChanged(ItemVar var, UIObjectClass oldType, 
				  UIObjectClass newType) {
    objectClass = newType;
    classEd.setSelectedObject(objectClass);
    tableModel.setObjectClass(newType);
  }

  //************NOTE: this does not work after var name editing!!!
  private void updateRefinement() {
    Debug.noteln("OCE: updating refinement");
    UIRefinement uir = (UIRefinement)currentObject;
    if (uir == null) return;

    Debug.noteln(" setting constraints");
    Set keys = varConstraints.keySet();  
    if (keys != null) {
      for (Iterator i = keys.iterator(); i.hasNext(); ) {
	ItemVar var = (ItemVar)i.next();
	HashMap propConsts = (HashMap)varConstraints.get(var);
	List constraints = UIUtil.getFlatValues(propConsts);
	//we now have constraints for var
	uir.setVariableConstraints(var, constraints); //will replace old
      }
    }
  }

  protected void noteType() {
    Constraint c;
    c = makeConstraint(variable, objectClass, IVUtil.TYPE_STRING_a); 
    try {
      if (c != null) {
	UIRefinement uir = ((UIRefinement)currentObject);
	if ((typeConstraint != null) && (uir != null)) {
	  uir.deleteConstraint(typeConstraint);
	}
	uir.addConstraint(c);
	typeConstraint = c;
	//update constraints for this type
	List constraints = uir.getVariableConstraints(variable);
	constraintsToProperty(variable, objectClass, constraints);
      }
    }
    catch (Exception e) { Debug.noteException(e); }
  }

  /**
   * Normally reads the constraint and updates the current object. Here this
   * is done at the time when changes are made.
   * NOTE***: perhaps here we can do a consistency check?
   */
  protected Object collectConstraint() {
      updateRefinement();
      return null;
  }

  
  private static int OWN = 1;
  private static int ANCESTOR = 2;
  private static int ALL = 3;

  /**
   * Get the current selections of the properties toggles (own/ancestors)
   */
  private int getPropertiesMode() {
    try {
      Object o = propModePan.getData();
      boolean[] data = {false, false};
      if (o instanceof Boolean[]) {
	  Boolean[] d = (Boolean[])o;
	  data[0] = d[0].booleanValue();
	  data[1] = d[1].booleanValue();
      }
      else data = (boolean[])o;
      int mode = 0;
      if (data.length == 2) {
	  if (data[0]) mode += OWN;
	  if (data[1]) mode += ANCESTOR;
	  //Debug.noteln("OCE: property mode is", mode);
      }
      else Debug.noteln("ObCE: wrong prop mode data length", data.length);
      if (mode == 0) return ALL;
      else return mode;
    }
    catch (Exception e) {
      Debug.noteException(e);
      return ALL;
    }
  }
  /**
   * Set the selections of the properties toggles (own/ancestors) to the 
   * given mode.
   */
  private void setPropertyMode(int mode) {
    Boolean[] data = {new Boolean(false), new Boolean(false)};
    if ((mode == 1) || (mode == 3)) data[0] = new Boolean(true);
    if ((mode == 2) || (mode == 3)) data[1] = new Boolean(true);
    propModePan.setData(data);
  }

  private void makeActList(Object[] children){
    //actEd.setData(children);
    //actEd.setSelectedIndex(0);
  }

    //---------------------- Useful Stuff -------------------------------

  /**
   * Make a type constraint for the variable 
   */
  protected Constraint makeConstraint(ItemVar var, UIObjectClass oClass, 
				      String typeProperty) {
    if ((var == null) || (oClass == null) || 
	(typeProperty == null) || (typeProperty == ""))
      return null;

    LList pat = Lisp.list(Symbol.intern(typeProperty), var); 
    PatternAssignment pa = new PatternAssignment(pat, oClass.getName());
    Constraint newC = new Constraint(Refinement.S_WORLD_STATE, 
				     Refinement.S_CONDITION, Lisp.list(pa));
    return newC;
    /*
    String bits = typeProperty + " " + variable.toString() 
      + " = " + oClass.getName();
    Constraint newC = IVUtil.readCondition(Refinement.S_WORLD_STATE, 
					   REL_DEFAULT, bits);
    if (newC == null) 
      Debug.noteln("OCE: failed to make type constraint from", bits);
    return newC;
    */
  }

  //delete the property constraint in the refinement and in the notes
  protected void deleteConstraint(ObjectProperty prop, Constraint c) {
    Debug.noteln("OCE: deleting constraint", c);
    Set propset = (Set)propertyConstraints.get(prop);
    if (propset != null) {
      Debug.noteln(" from", UIUtil.show(propset));
      propset.remove(c);
      if (propset.isEmpty()) propertyConstraints.remove(prop);
    }
    ((UIRefinement)currentObject).deleteConstraint(c);
    tableModel.setConstraints(propertyConstraints);
  }

  //only call this with constraints that have the same type, prop, and var
  private void replaceConstraint(Constraint oldC, Constraint newC) {
    try {
      UIRefinement uir = (UIRefinement)currentObject;
      uir.deleteConstraint(oldC);
      uir.addConstraint(newC);
      loadObjectConstraints(variable, uir);
    }
    catch (Exception e) {}
  }

  /**
   * Note the constraints for the given variable according to the property
   * that is being constrained.
   * Can handle all null parameters.
   */
  private void constraintsToProperty(ItemVar var, UIObjectClass oClass,
				     List constrs) {
    //Debug.noteln("OCE: mapping constraints to property");
    propertyConstraints.clear();
    if ((var == null) || (constrs == null)) return;
    for (Iterator i = constrs.iterator(); i.hasNext(); ) {
      try { 
	Constraint c = (Constraint)i.next();
	PatternAssignment ass = c.getPatternAssignment();
	if (ass != null) {

	  ObjectProperty prop = 
	    IVUtil.propertyFromPattern(ass.getPattern(), oClass);
	  if (prop != null) 
	    if (IVUtil.isSubjectInPattern(var, ass.getPattern())) 
	      //constraint is for property of this var
	      if (IVUtil.isSimplePropertyAssignment(ass, prop, var))
		propertyConstraints = 
		    tableModel.noteProperty(propertyConstraints, oClass, 
					    prop, c);
	      else 
		Debug.noteln("OCE: not a simple assignment:",ass.getPattern());
	}
      } catch (Exception e) {} //only interested in constraints
    }
  }



  //-----------------UIRefinement access

  private void loadObjectConstraints(ItemVar var, UIRefinement r) {
    if (r != null)
      loadObjectConstraints(var, r.getVariableConstraints(var));    
      //loadObjectConstraints(var, r.getConstraints());    
  }
  //turn constraints in UIR into type and property constraints
  private void loadObjectConstraints(ItemVar var, List constrs) {
    //Debug.noteln("OCE sorting object constraints");
    if (propertyConstraints != null) propertyConstraints.clear();
    typeConstraint = null;
    if (var == null) return;
    //find object class
    typeConstraint = IVUtil.findClassConstraint(var, constrs);

    UIObjectClass oc = null;
    if (typeConstraint == null) { //no type constraint given, so set to root
      if (root != null) 
	oc = (UIObjectClass)root.getUserObject();
    }
    else {
      //Debug.noteln("OCE got object class constraint", typeConstraint);
      UIDomain uid = ((UIRefinement)currentObject).getUIDomain();
      Object oType = IVUtil.extractType(typeConstraint, uid);
      //Debug.noteln("OCE got object class of type", oType.getClass());
      if (oType instanceof UIObjectClass) oc = (UIObjectClass)oType;
    }
    if (oc != null) {
      objectClass = oc;
      classEd.setSelectedObject(oc);
      tableModel.setObjectClass(oc);
    }
    
    constraintsToProperty(var, oc, constrs); //put constraints into prop map
  }



  //-----------------ObjectClass access
  /**
   * Find all object classes in the domain
   */
  private List getObjectClasses() {
    if (uiDomain != null)
      return uiDomain.getObjectClassTree();
    else return new LinkedList();
  }

  protected List getObjectProperties() {
    return getObjectProperties(mode, objectClass);
  }
  protected static List getObjectProperties(int mode, UIObjectClass oClass) {
    if (oClass == null) return null;
    List properties = new LinkedList();
    Collection props;
    if (mode == OWN) {
      props = oClass.getOwnProperties();
      if (props != null) properties.addAll(props);
    }
    else if (mode == ANCESTOR) {
      props = oClass.getAncestorProperties();
      if (props != null) properties.addAll(props);
    }
    else  {
      props = oClass.getAllProperties();
      if (props != null) properties.addAll(props);
    }
    return properties;
  }

  protected void finishEdits() {
    try {
      JTable table = (JTable)propConstraintEd.getItemComponent();
      if (table.isEditing()) {
	Debug.noteln("OCE: finishing edits");
	CellEditor ed = table.getCellEditor();
	ed.stopCellEditing();
      }
    }
    catch (Exception e) { Debug.noteException(e);}
      
  }

  //----------------------- Listener things --------------------------------

  public void mouseClicked(MouseEvent e){
    if (e.getSource() == null) return;
    //Debug.noteln("OCE problem? Got mouse from", e.getSource());
    if (e.getSource() == super.okBut) {
      //make sure last edits are kept
      finishEdits();
      //collect the rest of the data and update UIR****************
      tableModel.noteOldConstraints();
      processConstraint();
      closeEditor(); 
      return;
    }
    if (e.getSource() == super.cancelBut) {
      if (currentObject == null) {
	closeEditor();
	return;
      }
      String[] message = {"Cancelling this editor will set all constraints",
			  "for this refinement back to when you started",
			  "the Object Constraint Editor.",
			  "(NOTE: changes to the ObjectClass specifications",
			  "will NOT be undone.)"};
      int ans = JOptionPane.showConfirmDialog(this, message, "Confirm Cancel",
					      JOptionPane.OK_CANCEL_OPTION);
      if (ans == JOptionPane.OK_OPTION) {
	//set the constraints back to what they were
	((UIRefinement)currentObject).setConstraints(oldConstraints);
	this.closeEditor();
	clear();
	return;
      }
    }
    else super.mouseClicked(e);
  }
  
  private void cannotChangeAtNode() {
    String message = "Sofar, conditions and effects can only be" +
      " attached to refinements (i.e. self)";
    JOptionPane.showMessageDialog(this,message);

  }

  public void dataChanged(EditableObject eo, String field, 
			  Object vNew, Object vOld) {
    Debug.noteln("OCE: got data change");
  }
   
  public void valueChanged(ListSelectionEvent e) {
    if (isAdjusting || (e.getSource() == null)) return;
    //Debug.noteln("OCE: got list selection event", e);
    Object source = e.getSource();
    if ((source instanceof AbstractButton) &&
	(((AbstractButton)source).getActionCommand().equals("propMode"))) {
      mode = getPropertiesMode();
      tableModel.setMode(mode);
    }
    else if ((classEd != null) 
	     && e.getSource().equals(classEd.getItemComponent())) { 
      updateProperties();
      //get and note the selection
      Object o = classEd.getSelectedObject();
      if (o instanceof UIObjectClass) {
	objectClass = (UIObjectClass)o;
	noteType();
	//fill properties
	//Debug.noteln("OCE: Selected object class", o);
	//List properties = getObjectProperties(objectClass);
	//propConstraintEd.setData(properties);
	tableModel.setObjectClass(objectClass);
      }
      else {
	Debug.noteln("OCe: Problem with object class of type", o.getClass());
      }
    }
    else if ((varNameEd != null) 
	     && e.getSource().equals(varNameEd.getItemComponent())) { 
      updateVariable();
      //get and note the selection
    }
    /*
    else if (source.equals(actEd.getItemComponent())) { 
      int selected = actEd.getSelectedIndex();
      if (selected != 0) {
	String message = "Sofar, conditions and effects can only be" +
	  " attached to refinements (i.e. self)";
	JOptionPane.showMessageDialog(this,message);
	actEd.setSelectedIndex(0);
      }
    }
    */
  }

    //----------------------- Inner Classes --------------------------------


}


/*

***the var combobox does strange things! select only works every second time

 add undo!
 ask "replace?" for overwrites with property change 
   (value change need not ask); 
   yes - as now, no - change relation (ask again?), cancel - trow new away


 options: 
  - show properties: own, ancestors, all
  - cleanup constraints with no value: remove, use "true", ask for all

 help: 
  - no properties are showing: options switched off

 */
