/****************************************************************************
 * An editor/viewer frame for domain details.
 *
 * @author Jussi Stader
 * @version 4.1
 * Updated: Thu Oct 26 10:33:57 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import java.util.*;

import java.io.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import javax.swing.*;       
import javax.swing.event.*;       
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.awt.event.*;
import java.net.*;
import ix.*;
import ix.util.*;
import ix.util.xml.XML;
import ix.util.lisp.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.domain.event.*;
import ix.iface.util.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.domain.*;
import ix.iview.util.*;
import ix.iview.event.*;
import ix.iview.domain.*;
import ix.iview.domain.event.*;
import ix.iview.igraph.*;

/****************************************************************************
 * An editor/viewer frame for domain details.
 *
 * The editor can be created with a mandatory mode (true for simple,
 * false for advanced) and an optional domain or an optional title.
 * Most of the window buidling goes on in the ConstructFramePanels and their
 * ConstructEditorPanels, except for the menubar and the toolbar.<p>
 *
 * Super-class JFrameMT defines useful menu/toolbar making things.
 * Most of the action processing goes on in the ConstructFramePanels, except
 * for domain specific things (read...) and window things (toggles).<p>
 *
 * Example for using DomainEditor:
 * <PRE><code>
 *   ...
 *   File modelDirectory = DomainParser.getLibraryDirectory();
 *   boolean isSimple = false;
 *   DomainEditor frame = new DomainEditor(isSimple);
 *   frame.setModelDirectory(modelDirectory);
 *   frame.setVisible(true);
 *   ...
 * </code></PRE>
 *
 *****************************************************************************
 */
public class DomainEditorFrame 
  extends JFrameMTP 
  implements ChangeListener, UIDomainListener, UndoChangeListener,
	     InternalDomainEditor, IDESymbols
	     //editor listens for property changes and passes them on
{

  //---------------------Fields-------------------------------------------

  protected IXAgent agent;

  //public Properties ideProperties;
  protected JPropertiesManager propMan;

  //------------------------------------Window things
  protected GlobalFramePanel domainPanel;
  protected ActionEditorPanel actionPanel;
  protected AROEditorPanel aroPanel;
  private boolean useARO = true;
  protected GrammarEditorPanel grammarPanel;
  protected AConstructFramePanel lexiconPanel;
  protected AConstructFramePanel testPanel;

  protected OrderingEditor orderingEditor;

  protected AboutFrame aboutFrame;

  protected ImageIcon checkIcon = Util.resourceImageIcon("ide-check.gif");
  protected ImageIcon revertOIcon = 
    Util.resourceImageIcon("ide-revert-orig.gif");
  protected ImageIcon modifyIcon = 
    Util.resourceImageIcon("ide-modify-draft.gif");
  protected ImageIcon varDecIcon = 
    Util.resourceImageIcon("ide-var-icon.gif");
  protected ImageIcon publishIcon = 
    Util.resourceImageIcon("ide-draft-publish.gif");
  protected ImageIcon minimalIcon = 
    Util.resourceImageIcon("ide-view-minimal.gif");
  protected ImageIcon comprehensiveIcon = 
    Util.resourceImageIcon("ide-view-comprehensive.gif");
  protected ImageIcon graphicalIcon = 
    Util.resourceImageIcon("ide-view-graphical.gif");
  protected ImageIcon preferredIcon = 
    Util.resourceImageIcon("ide-preferences.gif");
  protected ImageIcon simpleIcon = 
    Util.resourceImageIcon("ide-mode-simple.gif");
  protected ImageIcon advancedIcon = 
    Util.resourceImageIcon("ide-mode-advanced.gif");


  private JMenu editConstructMenu;
  private JMenu orderMenu;
  //private JMenuItem Item;
  private ButtonGroup modeGroup = new ButtonGroup();
  private ButtonGroup viewGroup = new ButtonGroup();
  private JRadioButtonMenuItem simpleMI;
  private JRadioButtonMenuItem advancedMI;
  private JRadioButtonMenuItem minimalMI;
  private JRadioButtonMenuItem comprehensiveMI;
  private JRadioButtonMenuItem graphicalMI;
  private JButton minimalTB;
  private JButton comprehensiveTB;
  private JButton graphicalTB;
  private JCheckBoxMenuItem defSequenceMI;

  //overwrite others
  //own items
  protected IXToolItem publishItem =
    new IXToolItem(fileMenu, toolBar, this, "publish", "Publish Draft", 
		   publishIcon, 'P', "Publish draft to application");
  protected IXToolItem insertItem =
    new IXToolItem(fileMenu, this, "insert", "Insert into draft domain...",'I',
		   "Insert all constructs from an existing domain " +
		   "into the draft");
  protected IXToolItem clearItem =
    new IXToolItem(fileMenu, this, "clearDom", "Clear draft domain", 'C',
		   "Clear draft domain");
  protected IXToolItem revertDItem =
    new IXToolItem(fileMenu, this, "revertDom", 
		   "Discard domain changes (revert to published)", 'R',
		   "Discard changes to draft domain (revert to published)");
  protected IXToolItem changesAllItem = 
    new IXToolItem(fileMenu, this, "changesAll", 
		   "View changes in the domain", 'V', 
		   "View all changes in the draft domain and its constructs");

  protected IXToolItem newEditItem =
    new IXToolItem(editMenu, toolBar, this, "newEdit", "New", 
		   newIcon, 'N', "Create new item");
  protected IXToolItem revertOItem =
    new IXToolItem(editMenu, toolBar, this, "revertO", "Revert to Published", 
		   revertOIcon, "Revert to last published version");
  protected IXToolItem modifyItem =
    new IXToolItem(editMenu, toolBar, this, "modify", "Modify Draft", 
		   modifyIcon, 'M', "Note changes from panel into draft");
  /*
  protected IXToolItem revertItem =
    new IXToolItem(editMenu, toolBar, this, "undo", "Undo Edit", revertIcon,
		   'U', "Undo last change in the panel");
  protected IXToolItem redoItem =
    new IXToolItem(editMenu, this, "redo", "Redo Edit",
		   'R', "Redo the last undo in the panel");
  */
  protected IXToolItem declareItem =
    new IXToolItem(editMenu, toolBar, this, "declareVars", "Declare variables",
		   varDecIcon, 'D', "Declare variables for the Activity");
  protected IXToolItem findVarsItem =
    new IXToolItem(editMenu, this, "findVars", "Find variables", 'V', 
		   "Find all variables used in domain refinements");
  /*
  protected IXToolItem modifyItem =
    new IXToolItem(editMenu, toolBar, this, "modify", "Note", 
		   modifyIcon, 'N', "Note changes from panel into draft");
  */

  protected JMenu modeMenu;
  protected IXToolItem refreshItem = new IXToolItem(viewMenu, this, "refresh", 
				     "Refresh", 'R', 
				     "Refresh the display (may fold trees)");
  protected IXToolItem changesItem = new IXToolItem(viewMenu, this, "changes", 
				     "Changes from Draft", 'C', 
				     "View changes not yet noted to draft");
  protected IXToolItem changesOItem = new IXToolItem(viewMenu, this,
                                     "changesO", "Changes from Published",
                                     'P', "View changes to published object");
  protected IXToolItem simpleItem =
    new IXToolItem(modeMenu, toolBar, this, "simple", "Simple Mode", 
		   simpleIcon, 'S', "Switch to simple mode");
  protected IXToolItem advancedItem = 
    new IXToolItem(modeMenu, toolBar, this, "advanced", "Advanced Mode", 
		   advancedIcon, 'A', "Switch to advanced mode");

  protected JMenu viewsMenu;
  protected IXToolItem minimalItem =
    new IXToolItem(viewsMenu, toolBar, this, "minimal", "Minimal View", 
		   "Minimal", minimalIcon, "Switch to minimal view");
  protected IXToolItem comprehensiveItem =
    new IXToolItem(viewsMenu, toolBar, this, "comprehensive", 
		   "Comprehensive View", "Comprehensive", comprehensiveIcon, 
		   "Switch to comprehensive view");
  protected IXToolItem graphicalItem =
    new IXToolItem(viewsMenu, toolBar, this, "graphical", "Graphical View", 
		   "Graphical", graphicalIcon, "Switch to graphical view");

  protected IXToolItem checkItem =
    new IXToolItem(toolsMenu, toolBar, this, "check", "Check", checkIcon, 'C',
		   "Check the current construct");

  protected IXToolItem genItem =
    new IXToolItem(toolsMenu, this, "generateOS", "Generate Object System",
		   'G', "Generate an object system from activity constraints");
  protected IXToolItem searchItem =
    new IXToolItem(toolsMenu, this, "search", "Search",
		   'S', "Search the domain");

  protected IXToolItem preferencesItem;
  protected IXToolItem preferredItem = 
    new IXToolItem(viewMenu, toolBar, this, "preferred", "Preferred",
		   preferredIcon, 'P', 
		   "Use current preferences to set mode, style, and view");

  protected IXToolItem sequenceNewItem;
  protected IXToolItem sequenceItem;
  protected IXToolItem parallelItem;

  

  /*
  protected IXToolItem Item =
    new IXToolItem(editMenu, toolBar, this, "", "", Icon, '',
		   "");
  */
    //this one is not used
  public IXToolItem[] allButtons = 
     {newItem, openItem, saveItem, saveAsItem, 
      revertItem, redoItem, 
      declareItem, publishItem, 
      newEditItem, //cutItem, copyItem,
      pasteItem, revertOItem, modifyItem, 
      minimalItem, comprehensiveItem, graphicalItem,
      checkItem, 
      helpItem
     };
  public IXToolItem[] showButtons = 
    {openItem, saveItem, publishItem, null,
     newEditItem, revertItem, redoItem, declareItem, null, 
     minimalItem, comprehensiveItem, graphicalItem, null
     //checkItem, helpItem
    };
  //minimalItem, comprehensiveItem, graphicalItem,
  //newGNodeItem, deleteGNodesItem, layoutItem};
  public IXToolItem[] allMenuItems = {
    //fileMenu--------------------------------------------------
    newItem, openItem, saveItem, saveAsItem, toolbarItem, closeItem, //frame--
    exitItem,    
    publishItem, insertItem, clearItem, revertDItem, changesAllItem, //this
    //editMenu--------------------------------------------------
    revertItem, redoItem, cutItem, copyItem, pasteItem, //frame
    newEditItem, revertOItem, modifyItem, declareItem, findVarsItem, //this
     //viewMenu--------------------------------------------------
    refreshItem, changesItem, changesOItem, preferredItem, //rhis
    simpleItem, advancedItem, //view-modeMenu (this)
    sequenceNewItem, sequenceItem, parallelItem, //optionsMenu (this)
    minimalItem, comprehensiveItem, graphicalItem, //view-viewsMenu (this)
    //toolsMenu--------------------------------------------------
    checkItem, genItem, searchItem, //this
    //helpMenu--------------------------------------------------
    helpItem, manualItem, aboutItem //help-----
  };
    //also: panel styles in styleMenu
    // panels in windowsMenu	
    //	    
  

  //------------------------------------Modes and toggles
  //private String originalStyle = "tabbed";
  private boolean standAlone = false;
  private DomainEditor editor;

  //-------------preferences
  protected boolean simpleMode = true;
  protected boolean showButtonTexts = true;
  protected boolean isEditable = true;
  protected boolean listsAsText = true;
  protected boolean useNewCondEd = true;
  protected Symbol preferredView = VIEW_COMPREHENSIVE;
  protected String preferredStyle = "tabbed";
  protected Symbol preferredConstraints = C_ALL; //needed by ActionFormModel


  //------------------------------------"Globals"
  private ImageIcon frameIcon = Util.resourceImageIcon("ide-icon.gif");
  private ImageIcon logo = Util.resourceImageIcon("ide-logo.gif");
  private String title = "I-X Domain Editor";

  public File directory = DomainParser.getLibraryDirectory();
  /** The current domain in the editor. May be null */
    
  private boolean adjustingDomain = false;
  private boolean adjustingMode = false;
  private boolean adjustingView = false;

  //---------------------Constructors-------------------------------------

  /**
   * Makes the editor in stand-alone mode.
   * Subsequently, the domain can be set with the setDomain method or it can
   * be loaded via the menu. The default window title is "IX - Domain Editor".
   */
  public DomainEditorFrame(DomainEditor editor){this(editor, true);}


  /**
   * Makes the editor frame with the given stand alone mode.
   * Sets up panels and sub-editors, but does not initialise fully. This
   * should be done when properties have been read.
   * Subsequently, the domain can be set with the setDomain method or it can
   * be loaded via the menu. The default window title is "IX - Domain Editor".
   *
   * @param editor: the class that looks after domain editing
   * @param standAlone mode: true for standAlone, false for part of other 
   * application
   */
  public DomainEditorFrame(DomainEditor editor, boolean standAlone){
    this.editor = editor;
    this.standAlone = standAlone;

    helpFile = "ide-help.html";
    helpFrameIcon = Util.resourceImageIcon("ide-help-icon.gif");

    setVisuals(); //set title and icon image

    preInitMenus();
    preInitToolbar();
    preInitPanels();

    //addPanels();
    //Debug.noteln("I-DE added panels");

    setCurrentPanel(actionPanel);

    orderingEditor = actionPanel.orderingEditor;

    setUIDomainListeners(editor.getUIDomain()); //don't check for same UID!

  }

  /**
   * Makes the editor with the given domain - simple, stand-alone by default.
   * The default window title is "IX - Domain Editor".
   * 
   * @param domain the domain to use in the editor
   */
  public DomainEditorFrame(DomainEditor editor, Domain domain){
    this(editor, domain, true);
  }

  /**
   * Makes the editor with the given mode and load the given domain.
   * The default window title is "IX - Domain Editor".
   * 
   * @param domain the domain to use in the editor
   * @param standAlone mode: true for standAlone, false for part of other 
   * application
   */
  public DomainEditorFrame(DomainEditor editor, Domain domain, boolean alone){
    this(editor, alone);
    this.setDomain(domain);
    populateEditMenu();    //***needed?
    activateUndo();        //needed!
  }


  /**
   * Makes the editor frame with the given stand alone mode.
   * Subsequently, the domain can be set with the setDomain method or it can
   * be loaded via the menu. The default window title is "IX - Domain Editor".
   */
  public void setupFrame() {
    propMan = editor.ensureProperties();

    //makeDefaultMenu(); //done in preInit
    Debug.noteln("I-DE made menus");
    makeToolbar();
    Debug.noteln("I-DE made toolbar");

    // make a gap - after this there may be panel-specific buttons

    //actionPanel = new ActionEditorPanel(this,simpleMode);
    if (actionPanel == null)
      actionPanel = 
	new ActionEditorPanel(this, preferredView, preferredConstraints);
    else actionPanel.setupPanelProperties(preferredView, preferredConstraints);
    //hack to force updating the toolbar with the preferences in place
    boolean sbt = showButtonTexts;
    showButtonTexts = !sbt;
    updateToolbar(sbt);

    //if (preferredView.equals(VIEW_GRAPHICAL)) actionPanel.setGraphical();

    if (useARO && (aroPanel == null)) aroPanel = new AROEditorPanel(this);

    //all these are done in preInitPanels
    //domainPanel = new GlobalFramePanel(this);
    //grammarPanel = new GrammarEditorPanel(this);
    //actionPanel.addCurrentActionListener(domainPanel);
    //domainPanel.addCurrentActionListener(actionPanel);
    
    //testPanel = new TestEditorPanel();
    //lexiconPanel = new SimpleEditorPanel();
    //lexiconPanel.add(new JLabel("Editing of the lexicon" +
    //		       " is not yet supported."));
    addPanels(); 
    Debug.noteln("I-DE added panels");
    //if (simpleMode) setPanelStyle("single"); done in loadProperties

    fillWindowsMenu();
    //this next thing would override simple mode! dont do it!
    //setPanelStyle(ideProperties.getProperty("idePanelStyle"));
    setCurrentPanel(actionPanel);
    adaptMenusToPanel(actionPanel);  //gets missed out because no real change

    advancedMI.setSelected(!simpleMode);
    simpleMI.setSelected(simpleMode); 

    updateViewMenu(preferredView);

    orderingEditor = actionPanel.orderingEditor;

    this.addWindowListener(new IXWindow());

    //setDomain(new Domain()); //heads off into a loop somehow! set in field.
    UIDomain uiDomain = getUIDomain();
    setUIDomainListeners(uiDomain);
    //populateEditMenu();
    activateUndo();

    //Debug.noteln("DEF: width is ", getWidth());
    pack();  //this pack gives us a good size! a bit wide though
    setSize(650,750);
    //Debug.noteln("DEF: width is ", getWidth());
    propMan.updateApplicationFromProperties();
    //setBoundsFromProperties();   //100, 100, 650,650);
    //Debug.noteln("DEF: width is ", getWidth());
    invalidate(); //
    //validate(); //
  }



  /**
   * Makes the editor title and ensures not-stand-alone visuals.
   * 
   * @param theAgent the IXAgent that wants the editor to start
   */
  public void makeTitle(IXAgent theAgent){
    this.agent = theAgent;
    this.title = theAgent.getAgentDisplayName().toString() + " Domain Editor";
    exitItem.hide();
    setVisuals();
  }

  private void preInitMenus() {
    makeDefaultMenu();
  }

  private void preInitToolbar() {
    toolBar.removeAll();  //toolbar is made in JFrameMT
  }

  public DomainEditor editor() {
    return editor;
  }

  private void preInitPanels() {
    actionPanel = new ActionEditorPanel(this);
    if (useARO) aroPanel = new AROEditorPanel(this);
    domainPanel = new GlobalFramePanel(this);
    grammarPanel = new GrammarEditorPanel(this);
    actionPanel.addCurrentActionListener(domainPanel);
    domainPanel.addCurrentActionListener(actionPanel);
  }



  protected void postInitPanels() {
    if (useARO) aroPanel.populateARO();
    populateEditMenu();    //needed
    activateUndo();        //needed!
    //pack();        //makes things too small for some reason
  }

  //--------------------- Properties and Preferences -------------------------


  /** 
   * Check whether lists are to be displayed as texts. Used by
   * GrammarEditorPanel.
   * @return true if lists are to be displayed as texts, false otherwise.
   */
  public boolean listsAsText() {
    return listsAsText;
  }

  /**
   * @return true if the frame is in simple mode (single style, minimal
   * view), false otherwise.
   */
  public boolean isSimple() {
    return ((simpleMode) 
	    && (getPanelStyle().equals("single"))
	    && actionPanel.getActivityView().equals(VIEW_MINIMAL));
  }


  protected void setBoundsFromProperties() {
    int x = UIUtil.propertyToInt(propMan.getProperty("ideX"));
    int y = UIUtil.propertyToInt(propMan.getProperty("ideY"));
    int width = UIUtil.propertyToInt(propMan.getProperty("ideWidth"));
    int height = UIUtil.propertyToInt(propMan.getProperty("ideHeight"));
    setBounds(x, y, width, height);
    validate();
  }

  protected void setMode(Symbol mode) {
    if (MODE_SIMPLE.equals(mode)) {setSimpleMode(true);}
    else if (MODE_ADVANCED.equals(mode)) {setSimpleMode(false);}
    else Debug.noteln("I-DE: Cannot recognise mode ", mode.toString());
  }

  /**
   * Updates the toolbar buttons with text shown or not, according
   * to the parameter
   */
  protected void updateToolbar(boolean show) {
    //Debug.noteln("IDE: updating toolbar");
    if (showButtonTexts != show) {
      showButtonTexts = show;
      for (int i = 0; i < showButtons.length; i++) {
	try {
	   IXToolItem item = (IXToolItem)showButtons[i];
	   if (item != null) {
	       item.showButtonText(show);
	   }
	} catch (ClassCastException e) {} //only interested in IXToolItems
      }   
      if (actionPanel != null) actionPanel.updateToolbar(show);
    }
  }

  protected void showListsAsTexts(boolean show) {
    if (listsAsText != show) {
      listsAsText = show;
      if (grammarPanel != null) grammarPanel.setListsAsText(listsAsText);
      if (show) {
	String m = 
	  "Sofar, only lists in the grammar panel can be shown as texts";
	JOptionPane.showMessageDialog(this, m);
      }
    }
  }
  protected void useNewCondEditor(boolean use) {
    useNewCondEd = use;
  }


  //--------------------- Panels --------------------------------------------


  /** Adds all panels to the frame. */
  protected void addPanels() {
    addPanel((JPanel)domainPanel, "Domain", 
	     "Viewer/editor for global domain constraints and issues");
    addPanel((JPanel)actionPanel, "Activity", "Activity viewer/editor");
    addPanel((JPanel)grammarPanel, "Grammar", "Grammar viewer/editor");
    if (useARO) {
      addPanel((JPanel)aroPanel, "Object", "Object Class viewer/editor.");
      aroPanel.addUndoListener(this);
    }
    //UndoListening for updating the menu/toolbar offering undos or not
    domainPanel.addUndoListener(this);
    actionPanel.addUndoListener(this);
    grammarPanel.addUndoListener(this);
    /*
    addPanel((JPanel)testPanel, "Test", "Test panel");
    addPanel((JPanel)graphPanel, "Expansion", 
    	  "Graphical expansion viewer/editor");
    addPanel((JPanel)lexiconPanel, "Lexicon", "Lexicon viewer/editor");
    */
  }

  /** Clears all panels of the frame. */
  protected void clearPanels() {
    domainPanel.clear();
    actionPanel.clear();
    grammarPanel.clear();
    //testPanel.clear();
    if (useARO) aroPanel.clear(); 
    //lexiconPanel.clear();
  }

  /** Closes all panels of the frame. */
  protected boolean closePanels() {
    Debug.noteln("IDE-F: Closing panels");
    if (domainPanel.beforeClosing()) {
      if (actionPanel.beforeClosing()) {
	if (grammarPanel.beforeClosing()) {
	  if (!useARO || aroPanel.beforeClosing()) {
	    //testPanel.beforeClosing();
	    //lexiconPanel.beforeClosing();
	    return true;
	  }
	  else Debug.noteln(" Object panel not closing");
	}
	else Debug.noteln(" Grammar panel not closing");
      }
      else Debug.noteln(" Action panel not closing");
    }
    else Debug.noteln(" Domain panel not closing");
    return false;
  }


  /**
   * Changing current panel from old to new.
   */
  protected boolean currentPanelChanging(JPanel oldPanel, JPanel newPanel) {
    if (oldPanel == null) 
      return ((AConstructFramePanel)newPanel).beforeOpening();
    //do nothing but make sure the panel is changed to (important for cancel)
    if (oldPanel.equals(newPanel)) return true; 
    //Debug.noteln("DE: current panel changing");
    AConstructFramePanel oldACFP = (AConstructFramePanel)oldPanel;
    AConstructFramePanel newACFP = (AConstructFramePanel)newPanel;
    if (oldACFP.beforeClosing()) {
      if (newACFP.beforeOpening()) {
	//Debug.noteln("DE: current panel changing done");
	return true;
      }
    }
    //Debug.noteln("DE: CANCEL current panel changing");
    return false;
  }

  protected void setVisuals() {
    setTitle(title);
    setIconImage(frameIcon.getImage());
    //testPanel.setAgent(agent);
  }

  //---------------------Field access-------------------------------------  

    /** Changes the flat that toggles I-DE viewer-only/viewer-editor mode. */  
    public void setEditable(boolean editable) {
      Debug.noteln("IDE-F: editor is always editable. Setting ignored.");
      isEditable = editable;
    }

  /**
   * @return the current setting of the simpleMode flag
   */
  public boolean isSimpleMode(){ return simpleMode;}
  /**
   * If the state of the menu item does not correspond to the new setting,
   * it is changed.
   * If the editor is already in the given mode, nothing is done, except 
   * ensuring that the menu item is set to the corresponding state.
   * Currently, nothing else changes in the frame, so the command is passed on 
   * to the ActionEditorPanel.
   */
  public void setActivityView(Symbol view) { 
    if (adjustingView) return;
    //Debug.noteln("IDE: setting activity view to", view);
    adjustingView = true;
    //Debug.noteln("IDE: setting menu", view);
    try {
      updateViewMenu(view);

      if (!preferredView.equals(view)) {
	enableView(view);
      }
      // pass it on to panel
      if (actionPanel != null) actionPanel.setCurrentView(view); 
      adjustingView = false;
    }
    catch (Exception e) { adjustingView = false; }
  }

  public boolean setPanelStyle(String style) {
    boolean ok = super.setPanelStyle(style);
    // If in simple mode, change to non-single gets you into advanced mode
    if (ok && (simpleMode) && !style.equals("single")) {
      setSimpleMode(false);
    }
    return ok;
  }

  public void setConstraintsView(Symbol constrView) {
    if (!preferredConstraints.equals(constrView)) {
      preferredConstraints = constrView;
      if (actionPanel != null) actionPanel.setConstraintsView(constrView);
    }
  }

  /**
   * Call this to keep the Activity Views menu in line with the view that
   * is shown.
   */
  public void updateViewMenu(Symbol view) { 
    if (VIEW_GRAPHICAL.equals(view)) {
      if ((graphicalMI != null) && !graphicalMI.isSelected()) {
	//command came from elsewhere, so make menu item correspond
	//Debug.noteln(" setting menu", view);
	graphicalMI.setSelected(true); 
	minimalMI.setSelected(false); 
	comprehensiveMI.setSelected(false); 
      }
    }
    else if (VIEW_COMPREHENSIVE.equals(view)) {
      if ((comprehensiveMI != null) && !comprehensiveMI.isSelected()) {
	//command came from elsewhere, so make menu item correspond
	//Debug.noteln(" setting menu", view);
	comprehensiveMI.setSelected(true); 
	minimalMI.setSelected(false); 
	graphicalMI.setSelected(false); 
      }
    }
    else if (VIEW_MINIMAL.equals(view)) {
      if ((minimalMI != null) && !minimalMI.isSelected()) {
	//command came from elsewhere, so make menu item correspond
	//Debug.noteln(" setting menu", view);
	minimalMI.setSelected(true); 
	comprehensiveMI.setSelected(false); 
	graphicalMI.setSelected(false); 
      }
    }
  }


  /**
   * Sets the editor to simple (true) or advanced (false) mode.
   * If the state of the menu item does not correspond to the new setting,
   * it is changed.
   * If the editor is already in the given mode, nothing is done, except 
   * ensuring that the menu item is set to the corresponding state.
   * Currently, nothing else changes in the frame, so the command is passed on 
   * to the ActionEditorPanel.
   */
  public void setSimpleMode(boolean newSimple){ 
    //Debug.noteln("IDE: Setting simple mode to", new Boolean(newSimple));
    if (adjustingMode) return;
    adjustingMode = true;
    if ((simpleMI != null) && 
	(simpleMI.isSelected() != newSimple)) {
      //command came from elsewhere, so make menu item correspond to new value
      simpleMI.setSelected(newSimple); 
      advancedMI.setSelected(!newSimple); 
    }
    //if (newSimple != simpleMode) { //not already in the desired state
      simpleMode = newSimple;
      //enableSimple();
      // pass it on to panel
      if (newSimple) {
	//Debug.noteln("IDE: simple - setting panel and activity");
	setPanelStyle("single");
	setActivityView(VIEW_MINIMAL);
	if (minimalMI != null) minimalMI.setSelected(true);
      }
      else {
	//Debug.noteln("IDE: advanced - setting panel and activity back");
	propMan = editor.ensurePropertiesManager();
	setPanelStyle(propMan.getProperty("idePanelStyle"));
	String sValue = propMan.getProperty("ideActivityView"); 
	setActivityView(UIUtil.propertyToSymbol(sValue));
      }
    adjustingMode = false;
  }

  /**
   * If the state of the menu item does not correspond to the new setting,
   * it is changed.
   */
  public void setGraphical(){ 
    if ((graphicalMI != null) && !graphicalMI.isSelected()) {
      //command came from elsewhere, so make menu item correspond to new value
      graphicalMI.setSelected(true); 
      minimalMI.setSelected(false); 
      //Debug.noteln("IDE-sG: setting advanceMI to", new Boolean(false));
      comprehensiveMI.setSelected(false); 
    }
  }

  /**
   * Sets the directory used to load/save models.
   * This method just notes the given value which is used when needed
   * for the file selectors for loading and saving models
   */
  public void setModelDirectory(File directory){
    this.directory = directory;
  }
  /**
   * @return the directory used to load/save models.
   */
  public File getModelDirectory(){
    return this.directory;
  }



  //---------------------Toggles-----------------------------------------

  protected void enableView(Symbol view) { 
    preferredView = view;
    Dimension size = getSize();
    //change whether default sequence is enabled (minimal only)
    if (defSequenceMI != null) {
      if (VIEW_MINIMAL.equals(view)) defSequenceMI.setEnabled(true);
      else defSequenceMI.setEnabled(false);
    }
    // If in simple mode, change to non-minimal gets you into advanced mode
    if ((simpleMode) && !VIEW_MINIMAL.equals(view)) {
      setSimpleMode(false);
    }
    this.setSize(size);
    validate();
  }

 
  //---------------------Window building------------------------------------

  public void adaptDefaultMenu() {
    adaptMenuToAction();
  }
  protected void adaptMenuToAction() {
    /*
    optionsMenu.addSeparator();
    if (defSequenceMI == null) {
      sequenceNewItem = new IXToolItem(optionsMenu, this, "defSequence", 
				    "New Sub-Activities are in Sequence",
				    "New Sub-Activities are in Sequence");
      defSequenceMI = sequenceNewItem.makeCheckItem();
    }
    sequenceNewItem.showItem();
    defSequenceMI.setState(true);
    //defSequenceMI.setVisible(false);
    */

    if (orderMenu == null) 
      orderMenu = this.makeMenu(editMenu, "Set Constraints");
    else orderMenu.setVisible(true);
    if (sequenceItem == null) 
      sequenceItem = new IXToolItem(orderMenu, this, "makeSequence", 
				    "Put Sub-Activities in Sequence",
				    "Put Sub-Activities in Sequence");
    if (parallelItem == null)
      parallelItem = new IXToolItem(orderMenu, this, "makeParallel", 
				    "Put Sub-Activities in Parallel",
				    "Put Sub-Activities in Parallel");
    sequenceItem.showItem();
    parallelItem.showItem();
    //orderMenu.setVisible(false);
    populateEditMenu();    
    activateUndo();        //needed?

    makeViewsMenu();

  }
  /**
   * Makes a the file menu and toolbar buttons. Replaces that of the superclass
   * Menu items: new, open, save, save as, S, Options, S, close, exit
   * Toolbar buttons: new, open, save
   */
  public void makeFileMenu() {
    //fileMenu = this.makeMenu(parentMenu,"File");
    
    openItem.setItemLabel("Open as draft domain...");
    openItem.setToolTip("Open an existing file as a draft domain");
    openItem.hideItem();
    openItem.showItem();
    //openItem.show();
    insertItem.hideItem();
    insertItem.showItem();
    clearItem.showItem();
    revertDItem.showItem();
    changesAllItem.showItem();
    saveItem.setItemLabel("Save draft to file");
    saveItem.showItem();
    saveAsItem.setItemLabel("Save draft as...");
    saveAsItem.showItem();
    publishItem.showItem();
    
    //fileMenu.addSeparator();
    //makeOptionsMenu();

    makeFileMenuExits();
  }

  /**
   * Makes default close/exit menu options and and toolbar buttons.
   * Menu items: S, close, exit
   * Toolbar buttons: none
   */
  public void makeFileMenuExits() {
    fileMenu.addSeparator();
    if (standAlone) exitItem.showItem();
    else closeItem.showItem();
  }
  
  /** 
   * Overrides the one in JFrameMTP and JFrameMT - goes into View menu, not
   * File.
   */
  public void makeOptionsMenu() {
    //JFrameMT.makeOptionsMenu();
  }

  /**
   * Makes an edit menu
   * <nl><em>Menu items:</em> edit construct
   * <nl><em>Toolbar buttons:</em> none
   */
  public void makeEditMenu() {
    //editMenu = makeMenu(parentMenu,"Edit"); 
    //Debug.noteln("DomainEditor: Setting editConstruct to Edit Construct");

    newEditItem.setLabel("New construct");
    newEditItem.setToolTip("New construct");
    newEditItem.makeItem();
    newEditItem.show();
    cutItem.setToolTip("Delete");
    cutItem.showItem();
    editConstructMenu = makeMenu(editMenu,"Edit Construct");
    editConstructMenu.setEnabled(false);
    copyItem.showItem();

    editMenu.addSeparator();
    modifyItem.showItem();

    revertItem.setToolTip("Undo last change in the panel");
    revertItem.showItem();
    redoItem.showItem();
    revertOItem.showItem();

    editMenu.addSeparator();
    //saveItem.setEnabled(false);
  }

  public void makeViewMenu() {
    changesItem.show();
    changesOItem.show();
    viewMenu.addSeparator();
    refreshItem.show();
    viewMenu.addSeparator();
    makeToolbarItem(viewMenu);
    viewMenu.addSeparator();
    preferredItem.showItem();
    makeModeMenu();
    makeStyleMenu(viewMenu);
    makeWindowsMenu(viewMenu);
    //makeViewsMenu(); done in adaptMenuToAction
  }

  public void makeModeMenu() {
    modeMenu = makeMenu(viewMenu, "Editor Mode");
    modeMenu.setMnemonic('M');
    simpleItem.setMenu(modeMenu);
    advancedItem.setMenu(modeMenu);
    if (simpleMI == null) simpleMI = simpleItem.makeRadioItem();
    if (advancedMI == null) advancedMI = advancedItem.makeRadioItem();
    modeGroup.add(simpleMI);
    modeGroup.add(advancedMI);
    //Debug.noteln("Showing mode items");
    simpleItem.showItem();
    advancedItem.showItem();
  }

  public void makeViewsMenu() {
    viewsMenu = makeMenu(viewMenu, "Activity Views");
    minimalItem.setMenu(viewsMenu);
    comprehensiveItem.setMenu(viewsMenu);
    graphicalItem.setMenu(viewsMenu);

    if (minimalMI == null) 
      minimalMI = minimalItem.makeRadioItem();
    if (comprehensiveMI == null) 
      comprehensiveMI = comprehensiveItem.makeRadioItem();
    if (graphicalMI == null) 
      graphicalMI = graphicalItem.makeRadioItem();
    viewGroup.add(minimalMI);
    viewGroup.add(comprehensiveMI);
    viewGroup.add(graphicalMI);
    //these get shown in adaptMenusToPanel
  }

  /**
   * Makes a tools menu and toolbar buttons.
   * <nl><em>Menu items:</em> none
   * <nl><em>Toolbar buttons:</em> none
   */
  public void makeToolsMenu() {
    //toolsMenu = makeMenu(parentMenu,"Tools");
    checkItem.showItem();
    checkItem.setEnabled(false);
    toolsMenu.addSeparator();
    makePreferences();
  }

  public void makeHelpMenu() {
    super.makeHelpMenu();
    manualItem.setEnabled(false);
  }




  protected void makeToolbar() {
    toolBar.removeAll();
    //setup all buttons with(out) text and hide all.
    for (int i = 0; i < allButtons.length; i++) {
      IXToolItem item = (IXToolItem)allButtons[i];
      if (item != null) {
	item.showButtonText(true);
	item.hideButton();
      }
    }

    LayoutManager layout = toolBar.getLayout();
    //Debug.noteln("DE: toolbar layout:", layout);

    //set specific things
    openItem.setButtonLabel("Open Draft");
    saveItem.setButtonLabel("Save Draft");
    //cutItem.setButtonLabel("Delete");
    declareItem.setButtonLabel("Declare");

    //show show-buttons and add separators
    for (int j = 0; j < showButtons.length; j++) {
      IXToolItem item = (IXToolItem)showButtons[j];
      if (item != null) item.showButton();
      else toolBar.addSeparator();
    }

  }


  public void showButtonTexts(boolean show) {
  }

  JToolBar.Separator tbs = new JToolBar.Separator();

  /**
   * Called whenever the current panel is updated
   */
  protected void adaptMenusToPanel(JPanel panel){  
    //Debug.noteln("Adapting menus to panel");
    if (panel == null) return;

    //things to do for all construct panels
    String name = panel.getName();

    editConstructMenu.setText("Edit " + name);

    if (panel.equals(aroPanel)) {
      newEditItem.setItemLabel("New Child");
      newEditItem.setButtonLabel("New Child");
      newEditItem.setToolTip("New Child");
      genItem.showItem();
      genItem.setEnabled((getUIDomain() != null) 
			 && !getUIDomain().isEmpty()
			 && !getUIDomain().hasObjectSystem());
     }
    else {
      newEditItem.setItemLabel("New " + name);
      newEditItem.setButtonLabel("New");
      newEditItem.setToolTip("Create a new " + name);
      genItem.hideItem();
    }
    newEditItem.setEnabled(true);
    newEditItem.hideItem();
    newEditItem.showItem();
    modifyItem.setItemLabel("Note " + name + " into Draft");
    modifyItem.setEnabled(true);
    modifyItem.hideItem();
    modifyItem.showItem();
    checkItem.setItemLabel("Check current " + name);
    checkItem.setEnabled(false);
    checkItem.hideItem();
    checkItem.showItem();

    findVarsItem.hideItem();
    
    populateEditMenu();


    //things to do for each individual panel (have an if-else per panel)
    //NOTE: remember that domain panel has entry in "else" part!
    if (panel.equals(actionPanel) || panel.equals(aroPanel)) {
      cutItem.setItemLabel("Delete " + name);
      cutItem.setEnabled(true);
      cutItem.hideItem();
      cutItem.showItem();
      cutItem.hideButton();

      copyItem.setItemLabel("Copy " + name);
      copyItem.setEnabled(true);
      copyItem.hideItem();
      copyItem.showItem();
      copyItem.hideButton();

      //make sure the order is preserved
      revertItem.hideButton();
      revertItem.showButton();
      redoItem.hideButton();
      redoItem.showButton();

      searchItem.showItem();
      activateDomainItems();
    }
    else {
      cutItem.hideItem();
      copyItem.hideItem();
      searchItem.hideItem();
    }
    if (panel.equals(actionPanel)) {
      declareItem.hideItem();
      declareItem.setToolTip("Declare variables for the Activity");
      declareItem.showItem();
      declareItem.showButton();


      //switch things on
      //optionsMenu.add(defSequenceMI);
      //toolBar.add(Box.createRigidArea(new Dimension(30,0)));
      manageActViewGroup();    
      editMenu.add(orderMenu);
      //toolBar.validate();
    }
    else {      //switch things off
      //optionsMenu.remove(defSequenceMI);
      minimalItem.hideItem();
      comprehensiveItem.hideItem();
      graphicalItem.hideItem();
      declareItem.hideItem();
      declareItem.hideButton();
      if (panel.equals(domainPanel)) {
	String declareText = 
	  "Declare variables as used for all Activities in the domain";
	declareItem.setToolTip(declareText);
	declareItem.showItem();
	declareItem.showButton();
	findVarsItem.showItem();
      }
      editMenu.remove(orderMenu);
      //toolBar.remove(tbs);
    }
    refreshToolbar(panel);
    activateUndo();
  }
  /**
   * Called whenever the current panel is updated
   */
  protected void adaptMenusToPanelTrial(JPanel panel){  
    //Debug.noteln("Adapting menus to panel");
    if (panel == null) return;

    //things to do for all construct panels
    String name = panel.getName();

    //first set texts according to panel
    editConstructMenu.setText("Edit " + name);

    //then show/hide menu items and buttons according to panel

    if (panel.equals(aroPanel)) newEditItem.setItemLabel("New Child");
    else newEditItem.setItemLabel("New " + name);
    modifyItem.setItemLabel("Note " + name + " into Draft");
    checkItem.setItemLabel("Check current " + name);
    checkItem.setEnabled(false);
    cutItem.setItemLabel("Delete " + name);
    copyItem.setItemLabel("Copy " + name);

    if (panel.equals(actionPanel) || panel.equals(aroPanel)) {
      cutItem.setEnabled(true);
      copyItem.setEnabled(true);
    }
    else {
      cutItem.setEnabled(false);
      copyItem.setEnabled(false);
    }
    if (panel.equals(actionPanel)) {
      declareItem.setToolTip("Declare variables for the Activity");
    }

    populateEditMenu();


    //things to do for each individual panel (have an if-else per panel)
    //NOTE: remember that domain panel has entry in "else" part!
    if (panel.equals(actionPanel) || panel.equals(aroPanel)) {
      cutItem.setItemLabel("Delete " + name);
      cutItem.setEnabled(true);
      cutItem.hide();
      cutItem.show();
      cutItem.hideButton();

      copyItem.setItemLabel("Copy " + name);
      copyItem.setEnabled(true);
      copyItem.hide();
      copyItem.show();
      copyItem.hideButton();

      //make sure the order is preserved
      revertItem.hideButton();
      revertItem.showButton();
      redoItem.hideButton();
      redoItem.showButton();
    }
    else {
      cutItem.hide();
      copyItem.hide();

    }
    if (panel.equals(actionPanel)) {
      declareItem.hide();
      declareItem.setToolTip("Declare variables for the Activity");
      declareItem.show();


      //switch things on
      //optionsMenu.add(defSequenceMI);
      //toolBar.add(Box.createRigidArea(new Dimension(30,0)));
      toolBar.add(tbs);
      Component glue = Box.createHorizontalGlue();
      toolBar.add(glue);
      //int h = toolBar.getHeight();
      //toolBar.add(new Box.Filler(new Dimension(10,h), new Dimension(50,h), 
      //			 new Dimension(Short.MAX_VALUE, h)));
      String current = "";
      try {current = viewGroup.getSelection().getActionCommand();}
      catch (Exception e) {}
      viewGroup = new ButtonGroup();
      minimalMI = (JRadioButtonMenuItem)minimalItem.show();
      comprehensiveMI = (JRadioButtonMenuItem)comprehensiveItem.show();
      graphicalMI = (JRadioButtonMenuItem)graphicalItem.show();
      viewGroup.add(minimalMI);
      viewGroup.add(comprehensiveMI);
      viewGroup.add(graphicalMI);
      if ((current == "") || (current == null))  ;
      else if (current.equals("comprehensive")) 
	comprehensiveMI.setSelected(true);
      else if (current.equals("graphical")) graphicalMI.setSelected(true);
      else if (current.equals("minimal")) minimalMI.setSelected(true);

      editMenu.add(orderMenu);
      //toolBar.validate();
    }
    else {      //switch things off
      //optionsMenu.remove(defSequenceMI);
      minimalItem.hide();
      comprehensiveItem.hide();
      graphicalItem.hide();
      declareItem.hide();
      if (panel.equals(domainPanel)) {
	String declareText = 
	  "Declare variables as used for all Activities in the domain";
	declareItem.setToolTip(declareText);
	declareItem.show();
	findVarsItem.show();
      }
      editMenu.remove(orderMenu);
      toolBar.remove(tbs);
    }
    activateUndo();
  }


  private void refreshToolbar(JPanel panel) {
    toolBar.removeAll();
    if (panel == null) return;
    for (int i = 0; i < showButtons.length; i++) {
      IXToolItem item = (IXToolItem)showButtons[i];
      if (item != null) {
	item.hideButton();
	if (item.equals(minimalItem) || item.equals(comprehensiveItem) ||
	    item.equals(graphicalItem)) {
	  item.hideButton();
	  if (panel.equals(actionPanel)) 
	      item.showButton();
	}
	else item.showButton();
      }
      else toolBar.addSeparator();
    }
  }

  private void showToolbarComponents() {
    for (int i = 0; i < toolBar.getComponentCount(); i++) {
	Debug.noteln("Component at " + i + ":", toolBar.getComponent(i));
    }

  }


  private void manageActViewGroup() {   
    toolBar.addSeparator();
    Component glue = Box.createHorizontalGlue();
    toolBar.add(glue);
    //int h = toolBar.getHeight();
    //toolBar.add(new Box.Filler(new Dimension(10,h), new Dimension(50,h), 
    //			 new Dimension(Short.MAX_VALUE, h)));
    String current = "";
    try {current = viewGroup.getSelection().getActionCommand();}
    catch (Exception e) {}
    viewGroup = new ButtonGroup();
    minimalMI = (JRadioButtonMenuItem)minimalItem.show();
    comprehensiveMI = (JRadioButtonMenuItem)comprehensiveItem.show();
    graphicalMI = (JRadioButtonMenuItem)graphicalItem.show();
    viewGroup.add(minimalMI);
    viewGroup.add(comprehensiveMI);
    viewGroup.add(graphicalMI);
    if ((current == "") || (current == null))  ;
    else if (current.equals("comprehensive")) 
	comprehensiveMI.setSelected(true);
    else if (current.equals("graphical")) graphicalMI.setSelected(true);
    else if (current.equals("minimal")) minimalMI.setSelected(true);
  }

  /**
   * Fill the edit menu with constructs from the domain.
   */
  protected void populateEditMenu() {
    AConstructFramePanel panel = (AConstructFramePanel)getCurrentPanel();
    //Debug.noteln("DomainEditor: populating edit menu for panel",panel);
    //Debug.noteln(" editMenu's text is", editConstructMenu.getText());
    //make the list of objects to go in
    if (panel != null) {
      Collection constructs = panel.getAllConstructs();
      //Debug.noteln("DE: populate has constructs", constructs);
      List elements;
      if (constructs != null) {
	constructs = IVUtil.sortNamedCollection(constructs);
	elements = new ArrayList(constructs);
      }
      else  elements = new ArrayList();
      UIUtil.populateMenu(this,editConstructMenu,"editConstruct",elements);
      //if (elements.size() > 0) editConstructMenu.setEnabled(true);
    }
    activateDomainItems();
  }

  protected void activateDomainItems() {
    AConstructFramePanel panel = (AConstructFramePanel)getCurrentPanel();
    if (panel == null) return;
    if (panel.equals(aroPanel)) {
      genItem.setEnabled((getUIDomain() != null) 
			 && !getUIDomain().isEmpty()
			 && !getUIDomain().hasObjectSystem());
      
      searchItem.setEnabled((getUIDomain() != null) 
			    && !getUIDomain().isEmpty()
			    && getUIDomain().hasObjectSystem());
    }
    else if (panel.equals(actionPanel)) {
      searchItem.setEnabled((getUIDomain() != null) 
			    && !getUIDomain().isEmpty()
			    && getUIDomain().hasRefinements());
    }

  }
  public void activateUndo() {
    //Debug.noteln("IDE: activating undo/redo buttons");
    revertItem.setEnabled(false);
    redoItem.setEnabled(false);
    AConstructFramePanel panel = (AConstructFramePanel)getCurrentPanel();
    if (panel == null) return;
    if (panel.canUndo()) revertItem.setEnabled(true);
    if (panel.canRedo()) redoItem.setEnabled(true);
  }

  public void undoChanged(UndoManager manager, EditableObject uio) {
    //Debug.noteln("IDE: got undo change");
    activateUndo();
  }

  protected void makePreferences() {
    //makePreferencesMenu();
    preferencesItem = new IXToolItem(toolsMenu, this, "preferences", 
				     "Preferences", 'P', 
				     "Edit I-DE preferences");
    preferencesItem.showItem();
  }


  //---------------------Public Services--------------------------------------

  /**
   * The name of this is a hangover from something Jeff did. This is meant
   * to set the given refinement into the editor ready to be changed.
   */
  public void saveExpansion(Refinement data) {
    setConstruct(data);
  }


  /**
   * Sets the given construct as the current one and displays it for editing.
   * All work is done in the current panel.
   *
   * @param construct the original domain object to be viewed/edited
   */
  public void setConstruct(IXObject construct) {
    setUIConstruct(getUIDomain().getUIObject((IXObject)construct));
  }
  /**
   * Sets the given UIObject as the current one and displays it for editing.
   * All work is done in the current panel.
   *
   * @param construct the UI object to be viewed/edited
   */
  public void setUIConstruct(UIObject construct) {
    //find a suitable panel for the object and set it
    if (construct instanceof UIRefinement)
      setCurrentPanel(actionPanel);
    else if (construct instanceof UIDomain)
      setCurrentPanel(domainPanel);
    else if (construct instanceof UIObjectClass)
      setCurrentPanel(aroPanel);
    //set the construct in the current panel
    AConstructFramePanel panel = (AConstructFramePanel)getCurrentPanel();
    if (panel != null) panel.setUIConstruct((UIObject)construct);
  }


  /**
   * Sets the given domain as current both in this frame and the domain
   * editor panel. 
   *
   * @param domain the domain to use
   */
  protected void setDomain(Domain domain) {
    if (adjustingDomain) return;
    adjustingDomain = true;
    //Debug.noteln("DE-F: Setting domain", domain.toString());
    if (!domain.equals(getDomain())) 
      getUIDomain().setDomain(domain);

    //UIDomain newUID = getUIDomain().setDomain(domain);
    //Debug.noteln(" set uiDomain", newUID.print());
    //setUIDomain(newUID);
    //populateEditMenu();  //done by setUIDomain
    
    adjustingDomain = false;
  } 

  /**
   * Sets the given UIDomain as current both in this frame and the domain
   * editor panel.
   *
   * @param uiDomain the UIDomain to use
   */
  public void setUIDomain(UIDomain uiDomain) {
    if (getUIDomain().equals(uiDomain)) return;
    setUIDomainListeners(uiDomain);
    //global editing panel is added during init of global frame panel
    //uiDomain.fireDomainSet(uiDomain.getDomain());
    //populateEditMenu(); //domainSet event!
    uiDomain.fireDomainSet(uiDomain.getDomain());
  } 
  /** transfer listeners and tell them things have changed */
  protected void ensureUIDomainListeners(UIDomain uiDomain) {
    UIDomain currentUID = getUIDomain();
    if (currentUID.equals(uiDomain)) return;
    uiDomain.setUIDomainListeners(currentUID.getUIDomainListeners());
    uiDomain.addUIDomainListener(this);
    uiDomain.addUIDomainListener(grammarPanel);
    uiDomain.addUIDomainListener(actionPanel);
    uiDomain.addUIDomainListener(aroPanel);
  }
  /** transfer listeners and tell them things have changed */
  protected void setUIDomainListeners(UIDomain uiDomain) {
    UIDomain currentUID = getUIDomain();
    uiDomain.setUIDomainListeners(currentUID.getUIDomainListeners());
    uiDomain.addUIDomainListener(this);
    uiDomain.addUIDomainListener(grammarPanel);
    uiDomain.addUIDomainListener(actionPanel);
    uiDomain.addUIDomainListener(aroPanel);
  }

  public UIDomain getUIDomain() { return editor.getUIDomain(); }
  public Domain getDomain() { return getUIDomain().getDomain(); }

  /**
   * Sets the current panel and sets the domain in the panel
   *
   * @param domain the domain to use
   */
  public void updateCurrentPanel(AConstructFramePanel panel) {
    super.updateCurrentPanel(panel);
    //Debug.noteln("DE: Current panel is now", getACurrentPanel().getName());
    //Debug.noteln("DE: populating edit menu");
    populateEditMenu();
    activateUndo();        //needed?
  } 

  /**
   * Sets the toggle for default sub-activity sequencing behaviour.
   *
   * @param on true: put new sub-activities in sequence, false: put them in
   *   parallel.
   */
  public void setDefaultSequence(boolean on){
    if (defSequenceMI != null) defSequenceMI.setState(on);
    this.actionPanel.setDefaultSequence(on);
  }


  //---------------------Private Services--------------------------------------

  //---------------------User Services---------------------------------------

  /**
   * Clears the current domain.
   * Gets confirmation from the user if the domain is not empty.
   */
  protected void userClearDomain() {
    int y = JOptionPane.YES_OPTION;
    if (!getUIDomain().isEmpty()) {
      String[] message = {"This will clear all constructs in the draft.",
			  "Are you sure you want to do this?"};
      y = JOptionPane.showConfirmDialog(this, message, "Confirm",
					JOptionPane.YES_NO_OPTION);
    }
    if (y == JOptionPane.YES_OPTION) {
      getUIDomain().clearDraft();
      //this.clearPanels(); //do this listenening to domain!!
      //***should be a domain event
      //((GlobalFormPanel)domainPanel.formPanel).domainCleared(); 
      this.setUIDomain(getUIDomain());  //make changes take effect
    }
  }
  /**
   * Reverts the whole draft domain to the last published (or original) one.
   * Gets confirmation from the user if the domain has changed.
   */
  protected void userRevertDomain() {
    //UIUtil.notImplemented(null, "Reverting the whole domain to original");

    int y = JOptionPane.YES_OPTION;
    if (getUIDomain().hasChangedConstructsFromOriginal()) {
      String[] message = {"This will clear all changes to the draft" +
			  " (including open/insert)",
			  "since it was last published.",
			  "If the domain has not been published yet," +
			  " it will revert to the original domain.",
			  "Are you sure you want to do this?"};
      y = JOptionPane.showConfirmDialog(this, message, "Confirm",
					JOptionPane.YES_NO_OPTION);
    }
    if (y == JOptionPane.YES_OPTION) {
      //this.clearPanels();//do this listenening to domain!!
      getUIDomain().revertToOriginal();
      this.setUIDomain(getUIDomain());  //make changes take effect
    }

  }
  /**
   * Gets a domain file name, reads it, and replaces the contents of the
   * current domain with the read ones.
   */
  protected void userOpenDomain() {
    int y = JOptionPane.YES_OPTION;
    if (!getUIDomain().isEmpty()) {
      String[] message = {"This will clear all current domain constructs.",
			  "Are you sure you want to do this?",
			  "(Use 'Insert domain' to add to the current" +
			  " constructs)"};
      y = JOptionPane.showConfirmDialog(this, message, "Confirm",
					JOptionPane.YES_NO_OPTION);
    }
    if (y == JOptionPane.YES_OPTION) {
      //this.clearPanels();//do this listenening to domain!!
      getUIDomain().loadDomain();
      this.setUIDomain(getUIDomain());  //make changes take effect
    }
  }

  /**
   * Gets a domain file name and reads it.
   */
  protected void userInsertDomain() {
    getUIDomain().insertDomain();
    //this.setDomain(uiDomain.getDomain());
  }
  protected File userSaveDomain() {
    //***this first line is a hack - should just noteConstruct
    ((AConstructFramePanel)getACurrentPanel()).beforeClosing();
    return getUIDomain().saveDomain();
  }
  protected File userSaveDomainAs() {
    ((AConstructFramePanel)getACurrentPanel()).beforeClosing();
    return getUIDomain().saveDomain(null);
  }
  protected void userPublishDomain() {
    ((AConstructFramePanel)getACurrentPanel()).beforeClosing();
   getUIDomain().publishToDomain();
  }


  private void userPropSimple(boolean isSimple) {
    setSimpleMode(isSimple);
    propMan = editor.ensurePropertiesManager();
    if (isSimple) propMan.setProperty("ideMode", MODE_SIMPLE.toString());
    else propMan.setProperty("ideMode", MODE_ADVANCED.toString());
  }

  

  public void refinementAdded(RefinementEvent e) {
    //Debug.noteln("DE-F: got refinement-added event");
    populateEditMenu();
  }
  public void constructAdded(DomainEvent e) {
    //Debug.noteln("DE-F: got construct-added event");
    populateEditMenu();
  }
  public void constructAdded(UIDomainEvent e) {
    //Debug.noteln("DE-F: got construct-added event");
    populateEditMenu();
  }
  public void constructEdited(UIDomainEvent e) {
    //Debug.noteln("DE-F: got construct-edited event");
    populateEditMenu();
  }
  public void constructRemoved(UIDomainEvent e) {
    //Debug.noteln("DE-F: got construct-removed event");
    populateEditMenu();
  }
  public void domainCleared(UIDomainEvent e) {
    //Debug.noteln("DE-F: got domain-cleared event");
    populateEditMenu();
  }
  public void domainEdited(UIDomainEvent se) {
    //Debug.noteln("DE-F: got domain-edited event");
    populateEditMenu();
  }
  public void domainSet(UIDomainEvent se) {
    //Debug.noteln("DE-F: got event to set domain to", getDomain());
    //(new Throwable()).printStackTrace();
    populateEditMenu();
  }

  //---------------------Actions-------------------------------------------


  /**
   * Implements the ActionListener interface.
   * If this cannot recognise the event, it passes it to the current 
   * AConstructFramePanel. If the panel does not recognise the event either,
   * this passes it to its superclass (JFrameMTP, which will pass the
   * event to the current panel first). <p>
   *
   * Catch specific menu/toolbar events here. Let the panels process their
   * own events, let the superclass process panel-swapping events (inc. style)
   * and, via its superclass, general window events (close, exit, ...)
   */
  public void actionPerformed(ActionEvent event) {
      Object object = event.getSource();
      String command = ((AbstractButton)object).getActionCommand();
      //Debug.noteln("Domain Editor: event is",event.paramString());
      //Debug.noteln("Domain Editor: action object is",object);
      //Debug.noteln("Domain Editor: actionObject class is",object.getClass());
      Debug.noteln("Domain Editor: action command is",command);
      if (command.equals("close")) this.userClose();
      else if (command.equals("clearDom")) this.userClearDomain();
      else if (command.equals("open")) this.userOpenDomain();
      else if (command.equals("revertDom")) this.userRevertDomain();
      else if (command.equals("insert")) this.userInsertDomain();
      else if (command.equals("save")) this.userSaveDomain();
      else if (command.equals("saveAs")) this.userSaveDomainAs();
      else if (command.equals("publish")) this.userPublishDomain();

      else if (command.equals("refresh")) userRefresh();
      else if (command.equals("changes")) userViewChanges();
      else if (command.equals("changesAll")) userViewAllChanges();
      else if (command.equals("changesO")) userViewChangesFromOriginal();
      else if (command.equals("undo")) {
	try {
	  if (! ((AConstructFramePanel)getCurrentPanel()).undo())
	    //cannot undo, so must be displaying wrong!
	    Debug.noteln("I-DE: Something went wrong during undo");
	    activateUndo();
	} catch (Exception e) {}
      }
      else if (command.equals("redo")) {
	try {
	  if (! (((AConstructFramePanel)getCurrentPanel()).redo()))
	    //cannot redo, so must be displaying wrong!
	    Debug.noteln("I-DE: Something went wrong during redo");
	    activateUndo();
	} catch (Exception e) {}
      }

      else if (command.equals("preferences")) userPreferences();
      //else if (command.equals("toolbar")) 
      //  super.toggleToolbar(((JCheckBoxMenuItem)object).getState());
      else if (command.equals("modeSimple")) 
	this.userPropSimple(((JCheckBoxMenuItem)object).getState());

      else if (command.equals("preferred")) {
	propMan = editor.ensurePropertiesManager();
	propMan.updateApplicationFromProperties();
      }
      else if (command.equals("simple")) setSimpleMode(true);
      else if (command.equals("advanced")) setSimpleMode(false);

      else {
	//see whether the super classes JFrameMT(Panels) can do it
	super.actionPerformed(event);
      }
      //Debug.noteln("IDE: done command");
  }

  private boolean userCheckConstruct() {
    AConstructFramePanel panel = (AConstructFramePanel)getCurrentPanel();
    if (panel != null) return panel.checkConstruct();    
    else return true;
  }
  public void userViewAllChanges() {
    UIDomain uiDomain = getUIDomain();
    if (uiDomain == null) 
      JOptionPane.showMessageDialog(this,"No domain to check");
    else {
      List changes = uiDomain.collectAllChanges();
      if ((changes == null) || (changes.size() == 0))
	JOptionPane.showMessageDialog(this,"No changes found");
      else {
	TextAreaFrame taf = new 
	  TextAreaFrame("Changes for domain " + uiDomain.getName());
	for (Iterator i=changes.iterator(); i.hasNext();)
	  taf.appendLine((String)i.next());
	taf.setVisible(true);
      }
    }
  }
  public void userViewChanges() {
    AConstructFramePanel panel = (AConstructFramePanel)getCurrentPanel();
    if (panel != null) panel.viewChanges();
  }
  public void userViewChangesFromOriginal() {
    AConstructFramePanel panel = (AConstructFramePanel)getCurrentPanel();
    if (panel != null) panel.viewChangesFromOriginal();
  }


  private void closeIDE() {
    if (standAlone) System.exit(0); 
    else setVisible(false);
  }
  public void exitApplication() {
    userClose();
  }
    
  /**
   * Close the editor or exit it if it is running stand-alone.
   * Will check if edits have been noted (and put up dialogues) and will
   * check whether changes have been saved (and published if not running
   * stand-alone). 
   * @return true if the editor is closed, false if the user canceled.
   */
  public boolean userClose() {
    if (fileChooser != null) fileChooser.setVisible(false);
    if (helpFrame != null) helpFrame.setVisible(false);
    if (closePanels()) { //note changes into draft
      //will do this if dialogs are not canceled
      if (getUIDomain().hasChangedAllFromOriginal()) {
	Debug.noteln("Domain Changes are:", getUIDomain().collectAllChanges());
	Object o;
	String message;
	if (standAlone) {
	  message = "Do you want to save unsaved changes?";
	  //Object[] options = {"Save", "Exit", "Cancel"};
	  int i = 
	    JOptionPane.showConfirmDialog(this, message, "", 
					  JOptionPane.YES_NO_CANCEL_OPTION,
					  JOptionPane.QUESTION_MESSAGE);
	  if (i == JOptionPane.CANCEL_OPTION) return false;
	  else if (i == JOptionPane.NO_OPTION) o = "Exit";
	  else if (i == JOptionPane.YES_OPTION) o = "Save";
	  else o = null;
	}
	else {
	  message = "Do you want to save/publish changes?";
	  Object[] options = {"Save and Publish", "Save", "Publish", 
			      "Close only"};
	  o = JOptionPane.showInputDialog(this, message, "", 
					  JOptionPane.QUESTION_MESSAGE, null,
					  options, options[0]);
	}
	if (o == null) return false;  //canceled
	else if (o.equals("Save")) getUIDomain().saveDomain();
	else if (o.equals("Publish")) getUIDomain().publishToDomain();
	else if (o.equals("Save and Publish")) {
	  getUIDomain().saveDomain();
	  getUIDomain().publishToDomain();
	}
      }
      closeIDE();
      return true;
    }
    else return false;
  }

  public void aboutDialog() {
    if (aboutFrame == null) {
      String[] message = {"About " + getTitle(),
			  IVUtil.ideReleaseToString()};
      aboutFrame = new AboutFrame("About " + getTitle(), message);
    }
    aboutFrame.setVisible(true);
    /*
    JOptionPane.showMessageDialog(this, message,
				  "About " + getTitle(),
				  JOptionPane.INFORMATION_MESSAGE);
    */
  }

  private void userRefresh() {
    AConstructFramePanel panel = (AConstructFramePanel)getCurrentPanel();
    if (panel != null) panel.refresh();
    //pack(); //makes things too small for some reason
  }

  private void userPreferences() {
    propMan = editor.ensurePropertiesManager();
    propMan.editPreferences();
  }
  

  //---------------------Inner Classes----------------------------------------
    
  class IXWindow extends java.awt.event.WindowAdapter
  {
    public void windowClosing(WindowEvent e) { 
        userClose();
    }	  
  }
  

  //---------------------Stand Alone----------------------------------------

  static public void main(String[] args) {
    File directory = DomainParser.getLibraryDirectory(); //"domain-library" 
    
    // process command line args (domains)
    Parameters.processCommandLineArguments(args);
    Domain domain = new Domain();
    readDomain(domain);

    new ix.test.SimpleIXAgent();  //***HACK*** needed for us to use ToolFrame

    //make a stand-alone editor
    DomainEditor frame = new DomainEditor(domain, true);
    frame.setModelDirectory(directory);
 
    frame.setVisible(true);
  }

  //the two methods below are adapted from Jeff's Ip2 class
  /**
   *
   */
  protected static void readDomain(Domain domain) {
    List domains = Parameters.getList("domain");
    for (Iterator i = domains.iterator(); i.hasNext();) {
      String domainName = (String)i.next();
      try {
	readDomain(domain, domainName);
      }
      catch (Exception e) {
	Debug.displayException
	  ("Problem with domain " + domainName, e);
	if (!Parameters.isInteractive())
	  throw new RethrownException(e);
      }
    }
  }

  /**
   * Adds definitions from a given resource to a given domain. Does
   * *not* perform consistency checks on the way.
   */
  public static void readDomain(Domain domain, String resourceName) {
    // /\/: This used to be so simple ...
    // DomainParser.makeParser(domainName).readDomain(domain);
    Debug.noteln("Loading a domain from", resourceName);
    URL url = XML.toURL(resourceName);
    if (url == null) url = resolveDomainFile(resourceName);
    if (url == null)
      throw new IllegalArgumentException
	("Can't find a domain named " + Strings.quote(resourceName));
    Domain tempDomain = (Domain)XML.readObject(Domain.class, url);
    //tempDomain.checkConsistency();
    domain.takeFrom(tempDomain);
  }

  public static URL resolveDomainFile(String resourceName) {
    URL url = XML.toURL(resourceName);
    if (url == null) {
      // /\/: For backwards compatibility, try it as a file
      // in the library directory.
      //Debug.noteln("Trying " + resourceName + " as file in lib dir");
      File libDir = DomainParser.getLibraryDirectory();
      File file = new File(resourceName);
      if (!file.exists() && (file.getParentFile() == null)) {
	Debug.noteln("Resolving " + file + " against " + libDir);
	file = new File(libDir, file.getPath());
      }
      if (!file.exists()) {
	//try appending input to libDir
	String filesep = System.getProperty("file.separator");
	String libDirResource = libDir.getPath() + filesep + resourceName;
	//Debug.noteln("Trying file", libDirResource);
	file = new File(libDirResource);
      }
      if (file.exists()) {
	try {
	  url = file.toURL();
	}
	catch (MalformedURLException e) {
	  Debug.noteException(e);
	  throw new ConsistencyException("Unexpected", e);
	}
      }
    }  
    return url;
  }



}
