/****************************************************************************
 * An editor/viewer frame for domain details.
 *
 * @author Jussi Stader
 * @version 4.1
 * Updated: Thu Oct 12 13:04:49 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import java.util.*;

import java.io.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import javax.swing.*;       
import javax.swing.event.*;       
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.awt.event.*;
import java.net.*;
import ix.*;
import ix.util.*;
import ix.util.xml.XML;
import ix.util.lisp.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.domain.event.*;
import ix.iface.util.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.domain.*;
import ix.iview.util.*;
import ix.iview.event.*;
import ix.iview.domain.*;
import ix.iview.domain.event.*;
import ix.iview.igraph.*;

/****************************************************************************
 * An editor/viewer frame for domain details.
 *
 * The editor can be created with a mandatory mode (true for simple,
 * false for advanced) and an optional domain or an optional title.
 * Most of the window buidling goes on in the ConstructFramePanels and their
 * ConstructEditorPanels, except for the menubar and the toolbar.<p>
 *
 * Super-class JFrameMT defines useful menu/toolbar making things.
 * Most of the action processing goes on in the ConstructFramePanels, except
 * for domain specific things (read...) and window things (toggles).<p>
 *
 * Example for using DomainEditor:
 * <PRE><code>
 *   ...
 *   File modelDirectory = DomainParser.getLibraryDirectory();
 *   boolean isSimple = false;
 *   DomainEditor frame = new DomainEditor(isSimple);
 *   frame.setModelDirectory(modelDirectory);
 *   frame.setVisible(true);
 *   ...
 * </code></PRE>
 *
 *****************************************************************************
 */
public class DomainEditor 
  implements DomainListener, 
	     //UIDomainListener, 
	     PropertyChangeListener, 
	     InternalDomainEditor, IDESymbols, PropertiesApplication
{

  //---------------------Fields-------------------------------------------

  protected IXAgent agent;
  public DomainEditorFrame deFrame;

  //public Properties ideProperties;
  protected JPropertiesManager propMan;

  //------------------------------------Modes and toggles
  //private String originalStyle = "tabbed";
  private boolean standAlone = false;

  private boolean isEditable = true;

  //------------------------------------"Globals"
  public File directory = DomainParser.getLibraryDirectory();
  /** The current domain in the editor. May be null */

  public UIDomain uiDomain = new UIDomain(new Domain(), this);

    
  private boolean adjustingDomain = false;
  private boolean adjustingMode = false;
  private boolean adjustingView = false;

  //---------------------Constructors-------------------------------------

  /**
   * Makes the editor with the default (simple) mode.
   * Subsequently, the domain can be set with the setDomain method or it can
   * be loaded via the menu. The default window title is "IX - Domain Editor".
   */
  public DomainEditor(){this(false, true);}


  /**
   * Makes the editor with the given mode and the given stand-alone or not.
   * The given mode will override any defaults or preferences loaded from file.
   *
   * @param isSimple mode: true for simple, false for advanced
   * @param standAlone mode: true for standAlone, false for part of other 
   * application
   */
  public DomainEditor(boolean isSimple, boolean standAlone){
    this(standAlone);
    deFrame.setSimpleMode(isSimple);
    deFrame.activateUndo();
  }

  /**
   * Makes the editor with the given stand alone mode.
   * Subsequently, the domain can be set with the setDomain method or it can
   * be loaded via the menu. The default window title is "IX - Domain Editor".
   *
   * @param standAlone mode: true for standAlone, false for part of other 
   * application
   */
  public DomainEditor(boolean standAlone){
    this(new Domain(), standAlone);
  }

  /**
   * Makes the editor with the given domain - simple, stand-alone by default.
   * The default window title is "IX - Domain Editor".
   * 
   * @param domain the domain to use in the editor
   */
  public DomainEditor(Domain domain){
    this(domain, true);
  }

  /**
   * Makes the editor with the given mode and load the given domain.
   * The default window title is "IX - Domain Editor".
   * 
   * @param domain the domain to use in the editor
   * @param standAlone mode: true for standAlone, false for part of other 
   * application
   */
  public DomainEditor(Domain domain, boolean standAlone){
    IVUtil.printIDEReleaseGreeting();

    //Debug.noteln("DE: setting stand-alone to", standAlone);
    this.standAlone = standAlone;
    ensurePropertiesManager();
    //Debug.noteln("DE: got properties manager");
    propMan.initProperties();
    //Debug.noteln("DE: initialised properties");

    deFrame = new DomainEditorFrame(this, standAlone);

    deFrame.setupFrame();
    //setDomain(new Domain()); //heads off into a loop somehow! set in field.
    //setUIDomainListeners(getUIDomain()); //happens in deFrame
    //the next is redundant if called from DomainEditor(Domain d) or similar

    setDomain(domain);

    //deFrame.activateUndo(); part of postInitPanels
    deFrame.postInitPanels();
    //deFrame.pack();        
    //Debug.noteln("DE: width is ", deFrame.getWidth());

  }

  /**
   * Makes the editor with the given domain in not-stand-alone mode.
   * 
   * @param theAgent the IXAgent that wants the editor to start
   * @param domain the domain to use in the editor
   */
  public DomainEditor(IXAgent theAgent, Domain domain){
    this(domain, false);
    this.agent = theAgent;
    deFrame.makeTitle(theAgent);
  }

  /**
   * Needed for PropertiesApplication
   */
  public String getName() {
    if (deFrame == null) return "I-DE";
    else return deFrame.getName();
  }
  public JFrame getFrame() {
    return deFrame;
  }


  //--------------------- Properties and Preferences -------------------------

  /**
   * @return the current value within the application for the given property
   */
  public String getProperty(String which) {
    if ((which == null) || (which == "")) return "";
    else if (which.equals("ideMode")) {
      if ((deFrame != null) && (!deFrame.isSimple()))
	return MODE_ADVANCED.toString();
      else return MODE_SIMPLE.toString();      
    }
    else if (which.equals("ideEditable"))
      return new Boolean(isEditable).toString();

    else if (deFrame == null) return null; //----------rest for frame
    else if (which.equals("idePanelStyle")) 
      return deFrame.preferredStyle;
    else if (which.equals("ideActivityView")) 
      return deFrame.preferredView.toString();
    else if (which.equals("ideShowButtonTexts"))
      return new Boolean(deFrame.showButtonTexts).toString();
    else if (which.equals("ideNewCondEditor"))
      return new Boolean(deFrame.useNewCondEd).toString();
    else if (which.equals("ideListsAsTexts")) 
      return new Boolean(deFrame.listsAsText).toString();
    else if (which.equals("ideConstraints")) 
      return deFrame.preferredConstraints.toString();
    else if (which.equals("ideX")) 
      return UIUtil.intToProperty(deFrame.getX());
    else if (which.equals("ideY")) 
      return UIUtil.intToProperty(deFrame.getY());
    else if (which.equals("ideWidth")) 
      return UIUtil.intToProperty(deFrame.getWidth());
    else if (which.equals("ideHeight")) 
      return UIUtil.intToProperty(deFrame.getHeight());
    else if (which.equals("ideBounds")) return "";
    
    else Debug.noteln("****Warning**** IDE: cannot recognise property", which);
    return "";
  }

  /**
   * Sets the given property to the given value within the application
   */
  public void setProperty(String key, String newValue) {
    if ((key == null) || (key == "")) return;
    if (key.equals("ideEditable")) {
      isEditable = UIUtil.propertyToBoolean(newValue);
      if (deFrame != null) deFrame.setEditable(isEditable);
    }
    else if (deFrame == null) return; //----------rest for frame
    else if (key.equals("idePanelStyle")) 
      deFrame.setPanelStyle(newValue);
    else if (key.equals("ideActivityView")) 
      deFrame.setActivityView(UIUtil.propertyToSymbol(newValue));
    else if (key.equals("ideShowButtonTexts")) 
      deFrame.updateToolbar(UIUtil.propertyToBoolean(newValue));
    else if (key.equals("ideNewCondEditor")) 
      deFrame.useNewCondEditor(UIUtil.propertyToBoolean(newValue));
    else if (key.equals("ideListsAsTexts")) 
      deFrame.showListsAsTexts(UIUtil.propertyToBoolean(newValue));
    else if (key.equals("ideConstraints")) 
      deFrame.setConstraintsView(UIUtil.propertyToSymbol(newValue));
    else if (key.equals("ideMode")) //order matters! do this last
      deFrame.setMode(UIUtil.propertyToSymbol(newValue));
    else if (key.equals("ideBounds")) 
      deFrame.setBoundsFromProperties();
    else if (key.equals("ideX")) {}
    else if (key.equals("ideY")) {}
    else if (key.equals("ideWidth")) {}
    else if (key.equals("ideHeight")) {}
    else Debug.noteln("****Warning**** IDE: cannot recognise property", key);
  }



  //---------------------Field access-------------------------------------  

    /** Changes the flat that toggles I-DE viewer-only/viewer-editor mode. */  
    public void setEditable(boolean editable) {
	isEditable = editable;
    }

  /**
   * Sets the directory used to load/save models.
   * This method just notes the given value which is used when needed
   * for the file selectors for loading and saving models
   */
  public void setModelDirectory(File directory){
    this.directory = directory;
  }
  /**
   * @return the directory used to load/save models.
   */
  public File getModelDirectory(){
    return this.directory;
  }



  //---------------------Toggles-----------------------------------------

 
  //---------------------Window building------------------------------------

  //---------------------Public Services--------------------------------------

  /**
   * The name of this is a hangover from something Jeff did. This is meant
   * to set the given refinement into the editor ready to be changed.
   */
  public void saveExpansion(Refinement data) {
    setConstruct(data);
  }


  /**
   * Sets the given construct as the current one and displays it for editing.
   * All work is done in the current panel.
   *
   * @param construct the original domain object to be viewed/edited
   */
  public void setConstruct(IXObject construct) {
    setUIConstruct(uiDomain.getUIObject((IXObject)construct));
  }
  /**
   * Sets the given UIObject as the current one and displays it for editing.
   * All work is done in the current panel.
   *
   * @param construct the UI object to be viewed/edited
   */
  public void setUIConstruct(UIObject construct) {
    if (deFrame != null) deFrame.setUIConstruct(construct);
  }


  /**
   * Sets the given domain as current in te uiDomain. Interested
   * parties should be listening to uiDomain.
   *
   * @param domain the domain to use
   */
  public void setDomain(Domain domain) {
    Debug.noteln("DE: setting domain from", getDomain());
    Debug.noteln("   to", domain);
    if (adjustingDomain) return;
    adjustingDomain = true;
    //Debug.noteln("DE: Setting domain", domain.toString());
    if (!domain.equals(getDomain()))
	uiDomain.setDomain(domain);
    else Debug.noteln("DE: resetting same domain");

    //if (domain.equals(getDomain())) return;
    //UIDomain newUID = uiDomain.setDomain(domain);
    //Debug.noteln(" set uiDomain", uiDomain.print());
    //Debug.noteln(" domain changes now", uiDomain.collectAllChanges());
    //setUIDomain(newUID);
    //populateEditMenu();  //done by setUIDomain
    adjustingDomain = false;
  } 

  /**
   * Sets the given UIDomain as current both in this frame and the domain
   * editor panel.
   *
   * @param uiDomain the UIDomain to use
   */
  public void setUIDomain(UIDomain uiDomain) {
    if (this.uiDomain.equals(uiDomain)) return;
    this.uiDomain = uiDomain;
    //global editing panel is added during init of global frame panel
    //uiDomain.fireDomainSet(uiDomain.getDomain());
    //populateEditMenu(); //domainSet event!
    deFrame.setUIDomain(uiDomain);
    uiDomain.fireDomainSet(uiDomain.getDomain());
  } 

  public UIDomain getUIDomain() { return uiDomain; }
  public Domain getDomain() { return uiDomain.getDomain(); }


  //---------------------Private Services--------------------------------------



  //---------------------Domain Listening--------------------------------------

  public void refinementAdded(RefinementEvent e) {
    //Debug.noteln("DomainEditor: got refinement-added event");
    if (deFrame != null) deFrame.populateEditMenu();
  }

  //---------------------Actions-------------------------------------------


  /**
   *
   */
  public void propertyChange(PropertyChangeEvent evt) {
    if (evt == null) return;
    String key = evt.getPropertyName();
    if (key.equals("all")) {
      ensurePropertiesManager();
      propMan.updateApplicationFromProperties();
      return;
    }
    Object oldValue = evt.getOldValue();
    String newValue = (String)evt.getNewValue();
    if ((oldValue != null) && oldValue.equals(newValue)) return;
    Debug.note("IDE: got property change for " + key);
    Debug.noteln(" - old: " + oldValue.toString() + " new:", newValue);
    setProperty(key, newValue);
  }

  protected JPropertiesManager ensurePropertiesManager(){
    if (propMan == null) 
      propMan = new IDEPropertiesManager(this);
    return propMan;
  }
  protected JPropertiesManager ensureProperties(){
    if (propMan == null) {
      propMan = new IDEPropertiesManager(this);
      propMan.initProperties();
    }
    return propMan;
  }
  

  public void setVisible(boolean visible) {
    if (deFrame != null) deFrame.setVisible(visible);
  }


  //---------------------Stand Alone----------------------------------------

  static public void main(String[] args) {
    File directory = DomainParser.getLibraryDirectory(); //"domain-library" 
    
    // process command line args (domains)
    Parameters.processCommandLineArguments(args);
    Domain domain = new Domain();
    readDomain(domain);

    Debug.noteln("Domain Editor finished reading domain", domain);
    new ix.test.SimpleIXAgent();  //***HACK*** needed for us to use ToolFrame

    Debug.noteln("Domain Editor starting up.....");

    //make a stand-alone editor
    DomainEditor editor = new DomainEditor(domain, true);
    editor.setModelDirectory(directory);
 
    Debug.noteln("...Domain Editor started.");
    editor.setVisible(true);
  }


  //---------------------Domain Reading----------------------------------------

  //the two methods below are adapted from Jeff's Ip2 class
  /**
   *
   */
  protected static void readDomain(Domain domain) {
    List domains = Parameters.getList("domain");
    for (Iterator i = domains.iterator(); i.hasNext();) {
      String domainName = (String)i.next();
      try {
	readDomain(domain, domainName);
      }
      catch (Exception e) {
	Debug.displayException
	  ("Problem with domain " + domainName, e);
	if (!Parameters.isInteractive())
	  throw new RethrownException(e);
      }
    }
  }

  /**
   * Adds definitions from a given resource to a given domain. Does
   * *not* perform consistency checks on the way.
   */
  public static void readDomain(Domain domain, String resourceName) {
    // /\/: This used to be so simple ...
    // DomainParser.makeParser(domainName).readDomain(domain);
    Debug.noteln("Loading a domain from", resourceName);
    URL url = XML.toURL(resourceName);
    if (url == null) url = resolveDomainFile(resourceName);
    if (url == null)
      throw new IllegalArgumentException
	("Can't find a domain named " + Strings.quote(resourceName));
    Domain tempDomain = (Domain)XML.readObject(Domain.class, url);
    //tempDomain.checkConsistency();
    domain.takeFrom(tempDomain);
  }

  public static URL resolveDomainFile(String resourceName) {
    URL url = XML.toURL(resourceName);
    if (url == null) {
      // /\/: For backwards compatibility, try it as a file
      // in the library directory.
      //Debug.noteln("Trying " + resourceName + " as file in lib dir");
      File libDir = DomainParser.getLibraryDirectory();
      File file = new File(resourceName);
      if (!file.exists() && (file.getParentFile() == null)) {
	Debug.noteln("Resolving " + file + " against " + libDir);
	file = new File(libDir, file.getPath());
      }
      if (!file.exists()) {
	//try appending input to libDir
	String filesep = System.getProperty("file.separator");
	String libDirResource = libDir.getPath() + filesep + resourceName;
	//Debug.noteln("Trying file", libDirResource);
	file = new File(libDirResource);
      }
      if (file.exists()) {
	try {
	  url = file.toURL();
	}
	catch (MalformedURLException e) {
	  Debug.noteException(e);
	  throw new ConsistencyException("Unexpected", e);
	}
      }
    }  
    return url;
  }



}
