/****************************************************************************
 * An editor for specifying conditions and effects in refinements.
 *
 * @author Jussi Stader
 * @version 3.1
 * Updated: Thu Aug 31 10:22:15 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import javax.swing.*;       
import javax.swing.event.*;       
import java.util.*;
import java.awt.Color;
import java.awt.Frame;
import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;
import ix.*;
import ix.util.*;
import ix.util.Debug;
import ix.util.lisp.*;
import ix.icore.domain.*;
import ix.iview.util.*;
import ix.iview.domain.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.domain.*;
import ix.ip2.Ip2ModelManager;

/****************************************************************************
 * An editor for specifying conditions and effects in refinements.
 *
 * Conditions and effects are attached to self (the refinement) or to 
 * a child. <p>
 *
 * The editor is implemented as a frame using JList objects, a JCheckBox
 * and JButtons.  <p>
 *
 * The editor can be created with no arguments or with a given refinement
 * which will be set as the current refinement whose conditions and effects
 * are to be specified. Subsequently, the refinement in the editor can be set
 * using the setRefinement method.
 *****************************************************************************
 */
public class ConditionEffectEditor extends ConstraintEditor 
    implements ListSelectionListener
{

  //relation items
  protected ListEditorPanel actEd;

  protected JRadioButton condBut;
  protected JRadioButton effBut;
  private ButtonGroup group;
  protected JTextField statement;
  LTF_Parser constraintParser = new LTF_Parser();
  
  /**
   * Creates an editor for specifying time point relations.
   * The editor has components for selecting two time points of two
   * sub-activities, and two buttons (Ok, Cancel)
   * "- enter pattern-value statements,",
   * "- select whether you are specifying a condition or an effect,",
   * "- select the node to which the condition/effect applies."});
   */
  public ConditionEffectEditor(Frame owner){
    super(owner, "Condition/Effect Editor",
	  "Please specify a condition or an effect");
    }
    
  /**
   * Creates an editor for specifying time point relations within the given
   * refinement.
   * As OrderingEditor() but with a given refinement.
   * @param refinement the refinement whose sub-activities are to be related
   */
  public ConditionEffectEditor(Frame owner, UIRefinement refinement) {
    this(owner);
    setObject(refinement);
  }
  
  
  /**
   * Makes all items on the relations panel.
   * These are two identical activity lists and two identical time point lists
   * and a "Before" label between the two
   */
  protected Component makeRels() {
    //make a panel that can take all the bits
    JPanel panel = new JPanel(new GridBagLayout());
    
    //make the activity list panel
    //actList = new JList();    
    //actEd = makeActPanel(new JList()); if swapping, change gridbag constr
    actEd = makeActPanel(new JComboBox());
    
    //make the statement text
    JLabel label = new JLabel("Statement. syntax: pattern = value");
    statement = new JTextField();
    IVUtil.enableVars(this, statement);

    //make the condition/effects buttons
    condBut = new JRadioButton("Condition", true);
    effBut = new JRadioButton("Effect", false);
    condBut.setActionCommand("Condition");
    effBut.setActionCommand("Effect");
    Object[] typeButs = {condBut, effBut};
    IXButPanel typeButPanel = new IXButPanel(this, BoxLayout.X_AXIS, typeButs);
    typeButPanel.setButtonGroup(true);
    group = typeButPanel.group;
    
    //add it all to the panel
    GridBagConstraints c;
    c = new GridBagConstraints(0, 0, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.NORTHWEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(typeButPanel, c);
    c = new GridBagConstraints(0, 1, //x,y
			       1, 1, //width height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(label, c);
    c = new GridBagConstraints(0, 2, //x,y
			       1, 1, //width height
			       1.0,0.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       GridBagConstraints.HORIZONTAL,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(statement, c);
    
    c = new GridBagConstraints(0, 3, //x,y
			       1, 2, //width height
			       1.0,1.0,  //weight x,y
			       GridBagConstraints.WEST, //anchor
			       //GridBagConstraints.BOTH,  //fill
			       GridBagConstraints.HORIZONTAL,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
    panel.add(actEd, c);

    return panel;
  }

  /*
   * Make a sub-panel with the given activity list to be filled later
   * @param list the JList that will hold the activities
   * Uses a NodeNumberRenderer to display the list items
   * @return the new panel
   */
  private ListEditorPanel makeActPanel(JList list){
    actEd = new ListEditorPanel(this,"Attach to node",list);
    list.setAutoscrolls(true);
    list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    //list.setBackground(Color.white);
    actEd.setRenderer(new NodeNumberRenderer(false));
    return actEd;
  }
  /*
   * Make a sub-panel with the given activity combo box to be filled later
   * @param list the JComboBox that will hold the activities
   * Uses a NodeNumberRenderer to display the list items
   * @return the new panel
   */
  private ListEditorPanel makeActPanel(JComboBox box){
    actEd = new ListEditorPanel(this,"Attach to node",box);
    //list.setBackground(Color.white);
    //actEd.setRenderer(new NodeNumberRenderer(false));
    actEd.setRenderer(new FullNodeRenderer(false));
    return actEd;
  }


  public void moreInitFromObject() throws NullPointerException {
    boolean setEffect = (effBut.isSelected());
    initFromObject(); //sets conditions button (default)
    //specifying effects, so stick with that
    if (setEffect) effBut.setSelected(true);
    getFocusComponent().requestFocus();
  }

  /**
   * Sets the refinement whose components (children) are to be related.
   * Fills the activity lists of the panel with the children of the refinement.
   * @param refinement the refinement whose children are to be related 
   *
   * @throws NullPointerException if the refinement has no children.
   */
  protected void initFromObject() throws NullPointerException {
    statement.setText("");
    condBut.setSelected(true);
    UIRefinement refinement = (UIRefinement) currentObject;
    if (refinement == null) return;
    List children = new LinkedList(refinement.getNodes());
    children.add(0, refinement);
    // Debug.noteln("Children are", children);
    makeActList(children.toArray());
    //actList.clearSelection();
    //actList.setSelectedIndex(0);
    //children.remove(0);
    //statement.requestFocus();
    getFocusComponent().requestFocus();
  }

  protected void setFromObject(Object original) {
    //Debug.noteln("CondEffEd: setting from object for editing", original);
      //actList.clearSelection();
    statement.setText("");
    if (original == null) return;
    if (!(original instanceof Constraint)) {
      String message = "Condition/Effect editor can only deal with constraints"
	+ " not objects of class " + original.getClass().getName();
      JOptionPane.showMessageDialog(this,message);
      return;
    }

    Constraint constraint = (Constraint)original;
    if (constraint.getType() != Refinement.S_WORLD_STATE) {
      String message = "Condition/Effect editor cannot deal with constraints"
	+ " Of type " + constraint.getType().toString();
      JOptionPane.showMessageDialog(this,message);
      return;
    }
    actEd.setSelectedIndex(0);
    //actList.clearSelection();
    //actList.setSelectedIndex(0); //self only
    
    condBut.setSelected(constraint.getRelation() != Refinement.S_EFFECT);
    effBut.setSelected(constraint.getRelation() == Refinement.S_EFFECT);

    //statement.setText(UIUtil.listToDisplay(constraint.getParameters()));
    statement.setText(IVUtil.printConstraintParameters(constraint));
    //statement.requestFocus();
    getFocusComponent().requestFocus();
  }

  protected JComponent getFocusComponent() {
    return statement;
  }

  boolean isAdjusting = false;
  private void makeActList(Object[] children){
    isAdjusting = true;
    actEd.setData(children);
    isAdjusting = false;
    actEd.setSelectedIndex(0);
  }


  /**
   * Reads the selected relation and let interested parties know.
   * Checks that selections have been made.
   */
  protected Object collectConstraint() {
    Object node = actEd.getSelectedObject();
    ButtonModel bm = group.getSelection();
    //Debug.noteln("CEE: Selected button is",bm);
    
    String type = bm.getActionCommand(); //Condition or Effect
    //Debug.noteln("CEE: Type is", type);
    Symbol condEff;
    if (type.equals("Effect"))
      condEff = Refinement.S_EFFECT;
    else condEff = Refinement.S_CONDITION;
    String text = statement.getText().trim();
    //Debug.noteln("CEE: Got statement", text);
    Constraint newConstraint = readCondition(condEff, text);
    if (isEditing()) {
      try {
	Constraint constraint = (Constraint)((Constraint)original).clone();
	constraint.setRelation(condEff);
	constraint.setParameters(newConstraint.getParameters());
	return constraint;
      }
      catch (CloneNotSupportedException cnse) {
	Debug.describeException(cnse);
	return null;
      }
    }
    else return newConstraint;
  }

  
  protected Constraint readCondition(Symbol condEff, String text) 
       throws SyntaxException {
    return IVUtil.readCondition(Refinement.S_WORLD_STATE, condEff, text);
  }
  protected Constraint readCondition(String text) throws SyntaxException {
    return readCondition(Refinement.S_CONDITION, text);
  }

  /*
  public void mouseClicked(MouseEvent e){
    //Debug.noteln("Got mouse clicked from", e.getSource());
    if ((e.getSource() != null) 
	&& (e.getSource().equals(actEd.getItemComponent()))) {
      int selected = actEd.getSelectedIndex();
      if (selected != 0) {
	String message = "Sofar, conditions and effects can only be" +
	  " attached to refinements (i.e. self)";
	JOptionPane.showMessageDialog(this,message);
	actEd.setSelectedIndex(0);
      }
    }
    else super.mouseClicked(e);
  }
  */

  public void valueChanged(ListSelectionEvent e){
    if ((e.getSource() != null) 
	&& (e.getSource().equals(actEd.getItemComponent()))
	&& !isAdjusting
	&& ((JComponent)e.getSource()).isShowing()) { //comes from user
      int selected = actEd.getSelectedIndex();
      if (selected != 0) {
	String message = "Sofar, conditions and effects can only be" +
	  " attached to refinements (i.e. self)";
	JOptionPane.showMessageDialog(this,message);
	actEd.setSelectedIndex(0);
      }
    }
  }

  
}

//This would probably be neater with JComboBox objects but that didn't work.
// see RelationEditor for attempts.

