/****************************************************************************
 * An editor/viewer panel for refinement structures
 *
 * @author Jussi Stader
 * @version 4.1
 * Updated: Mon Nov  6 11:18:49 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iview;

import java.lang.Math;
import java.util.*;
import javax.swing.*;       
import javax.swing.event.ListSelectionListener;       
import javax.swing.event.ListSelectionEvent;       
import java.awt.BorderLayout;
import java.awt.Frame;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.*;
import ix.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.domain.event.*;
import ix.iface.ui.*;
import ix.iface.util.*;
import ix.iface.ui.table.*;
import ix.iface.ui.util.*;
import ix.iface.ui.event.*;
import ix.iface.domain.*;
import ix.iview.*;
import ix.iview.util.*;
import ix.iview.event.*;
import ix.iview.igraph.*;
import ix.iview.table.*;
import ix.iview.domain.*;
import ix.iview.domain.event.*;

/****************************************************************************
 * An editor/viewer panel for refinement structures.
 *
 * The editor has a flag that specifies whether it is in minimal view (true)
 * or in comprehensive view (false).<p>
 *
 * The editor is created with a minimal flag.
 * Subsequently, the refinement can be set using the setRefinement method.<p>
 *
 * The editor will not let you update a current refinement yet,
 * because we have not yet worked out how to manage the impact this
 * would have on other modules that use the domain.  The editor will
 * let you add new refinements (if you have changed an old one, change
 * the name to save it).<p>
 *
 * The refinement name, pattern, and comments are currently simply
 * strings. They can all be edited by typing into their fields. The
 * pattern should not stay as a string - it should change into
 * something a bit more constrained.  This will be changed when the
 * refinement models have been sorted out.<p>
 * 
 * Sub-activities and constraints can be added, deleted, and cleared,
 * either using the buttons or using keys (enter or insert for add,
 * delete for delete).<p>
 *
 * Sub-activities are simple LLists for the moment. Adding a sub-activity
 * expects the user to type in a pattern (currently just a string) and it will 
 * generate a number for the activity. Deleting sub-activities will also
 * delete any constraints in which the deleted activity participates.<p>
 *
 * The only constraints currently known are ordering constraints. Adding
 * constraints will bring up an editor that lets the user select activities
 * and time-points. For the moment, these "proper" constraints are translated
 * into simple versions because other modules cannot handle anything more
 * complex. A simple constraint "(<before> <after>)" stands for
 * "(<before> End Before Begin <after>). The constraint editor and the
 * DomainEditor are ready to provide and manage "proper" constraints, but we
 * will not provide them until other modules can deal with them.<p>
 *
 * Example code for using the ActionEditorPanel:
 *<PRE><code>
 *    ...
 *    editorPanel = new ActionEditorPanel(this,isMinimal);
 *    mainJPanel.add(editorPanel); 
 *    ...
 *</code></PRE>
 *****************************************************************************
 */
public class ActionEditorPanel extends AConstructFramePanel 
  implements ConstructFraming, ActionListener, DataChangeListener,
  UIDomainListener, CurrentActionListener, IDESymbols
{

  //---------------------Fields-------------------------------------------
  /** The panels parent - a DomainEditorFrame */
  private IXEditorPanel miniConstraintsPanel;
  private IXEditorPanel nodesTextPanel;
  //private IXEditorPanel conditionsPanel;
  private boolean minimalView = false;
  private IXRenderer fullNodeRenderer = new FullNodeRenderer(false);
  private IXRenderer miniNodeRenderer = new NodeNameRenderer(false);
  private IXRenderer fullIssueRenderer = new IssueRenderer(false);
  private IXRenderer miniIssueRenderer = new IssueRenderer(false);
  private boolean sequence = true;
  /** An editor for temporal relations */
  public OrderingEditor orderingEditor;
  public ConditionEffectEditor conditionEditor;
  public DefaultConstraintEditor constraintEditor;
  public SpecEditor nodeSpecEditor;
  public SpecEditor issueSpecEditor;
  public DeclarationEditor declarationEditor;
  public ObjectConstraintEditor objectConstraintEditor;
  /** An editor for temporal relations */
  //public ListRelationEditor relationEditor;

  public PanelSearcher searcher;

  private JSplitPane actionSplit = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
  private JScrollPane scrollTree = new JScrollPane();
  private IXTreeTable actionTree;
  private ActionTreeTableModel treeModel;

  IFormModel formModel;

  //private JObjectPopup treePopup = new JObjectPopup();
  //private JObjectMenu expansionsMenu = new JObjectMenu("Expansions");

  ConstructEditing currentPanel;

  private Point mousePoint = new Point(100, 100);

  //current view = VIEW_GRAPHICAL / VIEW_COMPREHENSIVE / VIEW_MINIMAL
  private Symbol currentView = VIEW_MINIMAL;
  //set to current view if it is a form view, stays that way for graph view
  private Symbol formView = VIEW_MINIMAL;
  //set true if graph view is showing
  private boolean graphMode = false;
  GExpansionEditor graphPanel;

  int comprehensiveConstrIndex = 3;
  List comprehensiveFields = new ArrayList();
  List minimalFields = new ArrayList();

  private boolean updatingView = false;
	
  private UIRefinement uiConstruct;  //this FramePanel needs it


  //---------------------Constructors----------------------------------------

  /**
   * Creates the editor panel with the given parent.
   * Creates all panel components, the mouse listener, and the relation editor.
   *
   * @param parent the DomainEditorFrame that the panel belons to. Used 
   *   mainly to keep toggle buttons and menus in synch
   */    
  public ActionEditorPanel(DomainEditorFrame theParent){
    //make all things (some are made during initialisation)
    super(theParent);
    currentPanel = (ActionFormPanel)formPanel;
    setupActionTree();
    formModel = new ActionFormModel(theParent);
    //setCurrentView(view);
    String[] cFields = {"pattern", "issues", "nodes", "orderings",
			"conditions", "constraints", "comments"};
    comprehensiveFields.addAll(Arrays.asList(cFields));
    String[] mFields = {"pattern", "issues", "nodes", "miniConstraints", 
			"comments"};
    minimalFields.addAll(Arrays.asList(mFields));
    //setConstraintsView(constr); do after pre-init
    enableComprehensive(); //this is really done during setConstraintsView
    String[] names = {"name"};
    formModel.setNameFields(names);
    //formModel.setDisplayFields(displays);//done in enableComprehensive()
    //formPanel.setModel(formModel, this);
    formPanel.setModel(formModel);
    MouseListener ml = new EdMouseAdapter();
    formPanel.addFormMouseListener(ml);
    //setCurrentView(view);
    
    //Debug.noteln("AEP setting new construct-------------");
    //newConstruct();

    makeTreePopup(new CatchingActionListener(parent));

  }

  public void setupPanelProperties(Symbol view, Symbol constr) {
    setConstraintsView(constr);
    setCurrentView(view);
  }

  /**
   * Creates the editor panel with the given parent with the given view and
   * the indicated set of constraints shown.
   * Creates all panel components, the mouse listener, and the relation editor.
   *
   * @param parent the DomainEditorFrame that the panel belons to. Used 
   *   mainly to keep toggle buttons and menus in synch
   * @param view a symbol specifying whether the editor is to display 
   *   its minimal, comprehensive, or graphical  view.
   * @param constr a symbol indicating how constraints are shown (orderings,
   *   conditions, others)
   */    
  public ActionEditorPanel(DomainEditorFrame theParent, 
			   Symbol view, Symbol constr){
    //make all things (some are made during initialisation)
    super(theParent);
    currentPanel = (ActionFormPanel)formPanel;
    setupActionTree();
    formModel = new ActionFormModel(theParent);
    //setCurrentView(view);
    String[] cFields = {"pattern", "issues", "nodes", "orderings",
			"conditions", "constraints", "comments"};
    comprehensiveFields.addAll(Arrays.asList(cFields));
    String[] mFields = {"pattern", "issues", "nodes", "miniConstraints", 
			"comments"};
    minimalFields.addAll(Arrays.asList(mFields));
    setConstraintsView(constr);
    enableComprehensive(); //this is really done during setConstraintsView
    String[] names = {"name"};
    formModel.setNameFields(names);
    //formModel.setDisplayFields(displays);//done in enableComprehensive()
    //formPanel.setModel(formModel, this);
    formPanel.setModel(formModel);
    MouseListener ml = new EdMouseAdapter();
    formPanel.addFormMouseListener(ml);
    setCurrentView(view);
    
    //Debug.noteln("AEP setting new construct-------------");
    newConstruct();

    makeTreePopup(new CatchingActionListener(parent));

  }

  protected JPanel setupMainPanel() {
    formPanel = new ActionFormPanel(this);

    //Debug.noteln("made form panel for AEP");
    graphPanel = new GExpansionEditor(this);
    //graphs get saved as lists of NodePosition objects. Tell XML
    ix.util.xml.XML.addImport("ix.iview.igraph.NodePosition");
    return formPanel;
  }

  protected void setupActionTree() {
    actionSplit.setRightComponent(scrollPane);
    actionSplit.setLeftComponent(scrollTree);
    add(actionSplit, BorderLayout.CENTER);
    JPanel treePanel = new JPanel();
    scrollTree.add(treePanel);
    treeModel = new ShortActionTreeTableModel(getUIDomain());
    //Debug.noteln("AEP: made short action tree model", treeModel);
    actionTree = new IXTreeTable(treeModel);
    treeModel.setParentsBold(true);
    ListSelectionModel lsm = actionTree.getSelectionModel();
    lsm.addListSelectionListener(new ListSelectionListener() {
	public void valueChanged(ListSelectionEvent lse) {
	  if (lse.getValueIsAdjusting()) return;
	  //Debug.noteln("AEP: lse is", lse);
	  //if (lse.getSource() != ActionEditorPanel.this) return;
	  Object selection = 
	    ActionEditorPanel.this.actionTree.getSelectedObject();
	  if ((selection != null) && !selection.equals(getUIConstruct())) {
	    if (selection instanceof UIObject)
	      ActionEditorPanel.this.editConstruct(selection);
	    //else if (selection instanceof NodeSpec) 
	  }
	}
      });
    actionTree.addMouseListener(new MouseAdapter() {
	public void mouseClicked(MouseEvent me) {
	  if (SwingUtilities.isRightMouseButton(me)) {
	    IXTreeTable aTree = ActionEditorPanel.this.actionTree;
	    AbstractIXTableModel aModel = 
	      (AbstractIXTableModel)aTree.getModel();
	    int row = aTree.rowAtPoint(me.getPoint());
	    Object construct = aModel.getRowObject(row);
	    if ((construct instanceof UINodeSpec) && (aTree.hasPopup())) {
	      JObjectPopup popup = (JObjectPopup)aTree.getPopup();
	      ActionEditorPanel.this.populatePopup(popup, construct);
	      popup.show(me.getComponent(), me.getX(), me.getY());
	    }
	  }
	}
      });

    //new ConstructTableMouser(actionTree, this);
    actionTree.setPreferredScrollableViewportSize(new Dimension(50, 500));
    scrollTree.getViewport().add(actionTree);
    actionSplit.setOneTouchExpandable(true);
    //actionSplit.setDividerLocation(0.33);
    //actionSplit.setResizeWeight(0.33);
    //actionSplit.setDividerSize(5);
    actionSplit.setDividerLocation(170);


  }

  protected void setPanel(ConstructEditing panel) {
    //Dimension vSize = scrollPane.getViewport().getSize();
    Dimension pSize = parent.getSize();
    //Debug.noteln("AEP: size is", pSize);
    if (currentPanel != null) {
      ((JPanel)currentPanel).setVisible(false);
      scrollPane.getViewport().remove((JPanel)currentPanel);
    }
    scrollPane.getViewport().add((JPanel)panel);
    currentPanel = panel;
    mainPanel = (JPanel)panel;
    ((JPanel)panel).setVisible(true);
    scrollPane.invalidate();
    //scrollPane.getViewport().setSize(vSize);
    parent.setSize(pSize);
    parent.validate();
  }

  public void updateToolbar(boolean show) {
    if (graphPanel != null) {
      graphPanel.newGNodeItem.showButtonText(show);
      graphPanel.deleteGNodesItem.showButtonText(show);
      graphPanel.layoutItem.showButtonText(show);
      graphPanel.layoutLeftItem.showButtonText(show);
    }
  }


  public void refresh() {
    updateTree();
  }

  private void makeTreePopup(CatchingActionListener catchingListener) {
    JPopupMenu popup = actionTree.getPopup();
    if (popup == null) {
      popup = new JObjectPopup();
      actionTree.setPopup(popup);
    }
    JMenuItem menuItem = new JMenuItem("New Expansion");
    menuItem.addActionListener(catchingListener);
    popup.add(menuItem);
  }



  //---------------------Field access----------------------------------------

  /**
   * @return true if the view is comprehensive, false for minimal and graphical
   */
  public boolean isComprehensive() { 
    return formView.equals(VIEW_COMPREHENSIVE);
  }
  /**
   * @return the current view (minimal/comprehensive/graphical)
   */
  public Symbol getActivityView() { 
    return currentView;
  }
  /**
   * @return the current view (minimal/comprehensive/graphical)
   */
  public void setCurrentView(Symbol view) { 
    //Debug.noteln("AEP: setCurrentView", view);
    if (view.equals(currentView)) return;  //already in view
    //Debug.noteln(" doing work");
    if (!beforeSwitching()) return;
    //graph mode first
    if (view.equals(VIEW_GRAPHICAL)) {
      updatingView = true;
      try {
	if (!graphMode) {
	  //coming from form view
	  //if (beforeSwitching()) {
	  this.setPanel(graphPanel);
	  graphMode = true;
	  currentView = view;
	  parent.setActivityView(view);
	  //graphPanel.setUIConstruct(uiConstruct);
	  setUIConstructPanel(uiConstruct);
	  //}
	}
	updatingView = false;
      } catch (Exception e) { updatingView = false; }
    }
    //form mode next.
    else if (graphMode) {
      updatingView = true;
      //coming from graph mode so change panel and change fields if required
      //noteConstruct();
      try {
	this.setPanel((ActionFormPanel)formPanel);
	graphMode = false;
	currentView = view;
	if (!formView.equals(view)) {
	  formView = view;
	  enableComprehensive();
	}
	else {
	  //((ActionFormPanel)formPanel).setUIConstruct(uiConstruct);
	  setUIConstructPanel(uiConstruct);
	  invalidate();
	  validate();
	}
	parent.setActivityView(view);
	updatingView = false;
      } catch (Exception e) { updatingView = false; }
    }
    else if (!formView.equals(view)) {
      updatingView = true;
      //Debug.noteln(" changing form view");
      try {
	formView = view;
	currentView = view;
	enableComprehensive();
	parent.setActivityView(view);
	updatingView = false;
      } catch (Exception e) { updatingView = false; }
    }
    afterSwitching();
  }

  /**
   * Changes the fields that are displayed in the form view. Currently only
   * changes comprehensive form, but would work for minimal too.
   * Does not check whether the view is different from the previous one.
   */
  public void setConstraintsView(Symbol constrView) {
    //Debug.noteln("AEP: setting constraints view to", constrView);
    //remove all constraint fields first
    int index = comprehensiveFields.indexOf("constraints");
    if (index != -1) comprehensiveFields.remove(index);
    index = comprehensiveFields.indexOf("conditions");
    if (index != -1) comprehensiveFields.remove(index);
    index = comprehensiveFields.indexOf("orderings");
    if (index != -1) comprehensiveFields.remove(index);
    //add all relevant constraint fields at the right place in the list, i.e.
    // start with the last one and add others before it.
    if (IDEUtil.showOther(constrView)) 
      comprehensiveFields.add(comprehensiveConstrIndex, "constraints");
    if (IDEUtil.showConditions(constrView)) 
      comprehensiveFields.add(comprehensiveConstrIndex, "conditions");
    if (IDEUtil.showOrder(constrView)) 
      comprehensiveFields.add(comprehensiveConstrIndex, "orderings");
    setComprehensiveFields(isComprehensive(), uiConstruct);
    //Debug.noteln("AEP: constraint fields", UIUtil.show(comprehensiveFields));
  }




  /**
   * @return the current setting of the sequence flag
   */
  public boolean isSequenceSet() { 
    return ((ActionFormPanel)formPanel).isSequenceSet();
  }
  /**
   * Sets the flag to put nodes in sequence (true) or in parallel (false)
   * If the state of the checkbox does not correspond to the new setting,
   * it is changed.
   * If the editor is already in the given mode, nothing is done.
   * Currently, nothing else changes in the panel
   */
  public void setDefaultSequence(boolean newSequence) {
    if (newSequence == sequence) return;
    else sequence = newSequence;
  }



  /** 
   * Checks whether the orderings are a sequence of the nodes.
   * @param nodes the nodes that can be part of the ordering.
   * @param orderings the orderings to be tested.
   */
  public static boolean orderingsAreSequential(LList nodes, LList orderings) {
    HashSet sequence = new HashSet(makeSequence(nodes));
    return new HashSet(orderings).equals(sequence);
    }

  public void setOrdering(boolean isSequence) {
    //Debug.noteln("AEP: Setting ordering to " + isSequence);
    if (isSequence) makeSequence();
    else makeParallel();
  }

  /**
   * Puts the nodes of the current refinement into sequence.
   */
  public void makeSequence() {
    // make all current sub-nodes sequential
    try {
      uiConstruct.setOrderings(makeSequence(uiConstruct.getNodes()));
      //***this.setOrderings();      
    }
    catch (Exception e) {
      Debug.noteException(e);
    }
  }
  /**
   * Puts the given nodes into sequence.
   */
  public static List makeSequence(List nodes) {
    //Debug.noteln("ActionEditorPanel - makeSequenceNodes");
    // make all current sub-nodes sequential
    NodeSpec previous = null;
    NodeSpec current;
    End tpBefore = End.END;
    End tpAfter = End.BEGIN;
    List orderings = new LListCollector();
    for (Iterator i = nodes.iterator(); i.hasNext(); ) {
      current = (NodeSpec)i.next();
      //Debug.noteln(" AEP: working on node", current.getId());
      if (previous != null) {
	Ordering ordering = 
	  new Ordering(new NodeEndRef(tpBefore, previous.getId()), 
		       new NodeEndRef(tpAfter, current.getId()));
	orderings.add(ordering);
	//Debug.noteln(" AEP: Orderings are", orderings);
      }
      previous = current;
    }
    return orderings;
  }


  /**
   * Puts the nodes of the current refinement in parallel.
   */
  public void makeParallel() {
    clearOrderings();
  }


  //---------------------Toggles----------------------------------------

  /*
   * If the view is not comprehensive, disable constraints and other useful
   * things
   * Comprehensive:
   *  sub-act renderer is FullNodeRenderer
   *  uses constraintsPanel
   *  uses nodesPanel
   * Minimal:
   *  sub-act renderer is NodeNameRenderer
   *  uses miniConstraintsPanel
   *  uses nodesTextPanel 
   */
  private void enableComprehensive() {
    setComprehensiveFields(isComprehensive(), uiConstruct);
  }

  /**
   * Sets the fields of the form panel to comprehensive/minimal.
   * Also sets UIConstruct (if not null) and invalidates
   */
  private void setComprehensiveFields(boolean compre, UIRefinement uir) {
    if (compre) {
      formPanel.unsetSimpleField("nodes");
      formPanel.unsetSimpleField("issues");
      formPanel.setRenderer("nodes", fullNodeRenderer);
      formPanel.setRenderer("issues", fullIssueRenderer);
      formModel.setDisplayFields(comprehensiveFields);
    }
    else {
      formPanel.setSimpleField("nodes");
      formPanel.setSimpleField("issues");
      formPanel.setRenderer("nodes", miniNodeRenderer);
      formPanel.setRenderer("issues", miniIssueRenderer);
      formModel.setDisplayFields(minimalFields);
    }
    //if (uir != null) ((ActionFormPanel)formPanel).setUIConstruct(uir);
    if (uir != null) setUIConstructPanel(uir);
    uiConstruct = uir;
    this.invalidate();
    validate();
  }

  //---------------------Window building-------------------------------------

  public void ensureRelationEditor(){
    if (orderingEditor == null){
      orderingEditor = new OrderingEditor(parent);
      //orderingEditor.addConstraintListener(this);
      orderingEditor.setVisible(false);
    } 
  }
  public void ensureConditionEditor(){
    if (conditionEditor == null){
      conditionEditor = new ConditionEffectEditor(parent);
      //conditionEditor.addConstraintListener(this);
      conditionEditor.setVisible(false);
    } 
  }
  public void ensureConstraintEditor(){
    if (constraintEditor == null){
      constraintEditor = new DefaultConstraintEditor(parent);
      //constraintEditor.addConstraintListener(this);
      constraintEditor.setVisible(false);
    } 
  }
  public void ensureObjectConstraintEditor(){
    if (objectConstraintEditor == null){
      objectConstraintEditor = 
	  new ObjectConstraintEditor(parent, getUIDomain());
      //constraintEditor.addConstraintListener(this);
      objectConstraintEditor.setVisible(false);
    } 
  }
  public void ensureNodeSpecEditor(){
    if (nodeSpecEditor == null){
      nodeSpecEditor = new SpecEditor(parent, "sub-activity");
      nodeSpecEditor.setVisible(false);
    } 
  }
  public void ensureIssueSpecEditor(){
    if (issueSpecEditor == null){
      issueSpecEditor = new SpecEditor(parent, "issue");
      issueSpecEditor.setVisible(false);
    } 
  }
  public void ensureDeclarationEditor(){
    if (declarationEditor == null){
      declarationEditor = new DeclarationEditor(parent);
      declarationEditor.setVisible(false);
    } 
  }
  public void ensureSearcher() {
    if (searcher == null) {
      searcher = new PanelSearcher(this, new ActionListRenderer(false));
      searcher.setVisible(false);
    }
  }

  public void userSearch() {
    ensureSearcher();
    searcher.setVisible(true);
  }
  private void searchOld() {
    String pat = JOptionPane.showInputDialog(this, 
					     "Search string (regex)");
    if ((pat == null) || pat.equals("")) Debug.noteln("No search string");
    else {
      Set refs = getUIDomain().getMatchingRefinements(pat);
      Debug.noteln("Matching specifications:", refs);
      Debug.noteln("  names:", IVUtil.namedListToDisplay(refs));
      
    }
  }

  private Point getEditorLocation() {
    Point panelPoint = getLocation();
    Point framePoint = parent.getLocation();
    panelPoint.translate((int)framePoint.getX(), (int)framePoint.getY());
    //Debug.noteln("AEP: mouse point", mousePoint.toString());
    //Debug.noteln("AEP: frame point", framePoint.toString());
    Point point = new Point(panelPoint);
    point.translate((int)mousePoint.getX(), (int)mousePoint.getY());
    return point;
  }


  public boolean beforeClosing() {
    if (nodeSpecEditor != null) nodeSpecEditor.setVisible(false);
    if (issueSpecEditor != null) issueSpecEditor.setVisible(false);
    if (orderingEditor != null) orderingEditor.setVisible(false);
    if (conditionEditor != null) conditionEditor.setVisible(false);
    if (constraintEditor != null) constraintEditor.setVisible(false);
    if (objectConstraintEditor != null) 
      objectConstraintEditor.setVisible(false);
    if (declarationEditor != null) declarationEditor.setVisible(false);
    if (searcher != null) searcher.close();
    return beforeSwitching();
  }
  

  //---------------------Public services--------------------------------------

  /**
   * Sets the given UI construct to be viewed/edited. 
   * Also sets the uiConstruct in the current editorPanel
   *
   * @param object the UI object to be viewed/edited
   */    
  public void setUIConstruct(UIObject object) { //***why diff from super???
    //noteConstruct();
    if (beforeSwitching()) {
      //UIObject old = getUIConstruct();
      setUIConstructPanel(object);
      //fireConstructChanged(old, object);
      afterSwitching();
    }
  }
  /**
   * Sets the given UI construct in this editor and in the current panel
   * Does not call beforeSwitching.
   *
   * @param object the UI object to be viewed/edited
   */    
  public void setUIConstructPanel(UIObject object) {
    UIRefinement old = uiConstruct;
    setUIConstructOnly(object);
    //set the object in the actual panel
    currentPanel.setUIConstruct(uiConstruct);
    //fireConstructChanged(old, uiConstruct); panel fires this
    fireCurrentActionChanged(old, uiConstruct);
  }
  /**
   * Sets the given UI construct to be viewed/edited without noting the 
   * previous construct.
   * Also sets the uiConstruct in the current editorPanel
   *
   * @param object the UI object to be viewed/edited
   */    
  public void setUIConstructOnly(UIObject object) {
    uiConstruct = (UIRefinement)object;
    updateTreeSelection();
  }
  /**
   * Clears the current construct and the panel.
   * Also sets the uiConstruct in the current editorPanel
   */    
  public void clearUIConstruct() {
    uiConstruct = (UIRefinement)makeNewConstruct();
    //set the object in the actual panel
    setUIConstructPanel(uiConstruct);
    ((ConstructEditing)graphPanel).setUIConstruct(uiConstruct);
    ((ConstructEditing)formPanel).setUIConstruct(uiConstruct);
  }

  public UIRefinement makeNewRefinement(String name, LList pattern) {
    
    UIRefinement newR = (UIRefinement)makeNewConstruct();
    newR.setName(name);
    newR.setPattern(pattern);

    return newR;
  }


  public ArrayList getAllExpansions(UIRefinement ref) {
    if (getUIDomain() == null) return null;
    LList pattern = ref.getPattern();
    //Debug.noteln("AEP: Getting refinements matching", pattern);
    Set exps = getUIDomain().getMatchingRefinements(pattern);
    return new ArrayList(exps);
  }
  public ArrayList getAllExpansions(NodeSpec node) {
    if (getUIDomain() == null) return null;
    LList pattern = node.getPattern();
    //Debug.noteln("AEP: Getting refinements matching", pattern);
    Set exps = getUIDomain().getMatchingRefinements(pattern);
    return new ArrayList(exps);
  }


  //---------------------Private services--------------------------------------

  /**
   * Note the settings in the editor in a UIObject. Do not save into doamin.
   */
  protected UIObject noteConstruct() {
    //make sure there is a current construct
    if (uiConstruct == null) 
      uiConstruct = (UIRefinement)makeNewConstruct();
    currentPanel.saveToObject(uiConstruct);

    uiConstruct.noteChange();
    //make sure the refinement is noted into the right domain
    if (!getUIDomain().equals(uiConstruct.getUIDomain())) 
      uiConstruct.moveToUIDomain(getUIDomain());
    getUIDomain().updateConstruct(uiConstruct);
    //Debug.noteln("AEP: Noted construct");
    //Debug.noteln(" ",uiConstruct.print());
    return uiConstruct;
  }

  //---------------------filling the panel details

  /*
  private void setChildren() {
    formPanel.displayFieldData("nodes");
  }
  
  private void setIssues(){
    formPanel.displayFieldData("issues");
  }
  
  private void setOrderings() {
    if (isComprehensive())
      formPanel.displayFieldData("orderings");
    else if (graphMode)
      graphPanel.displayFieldData("orderings");
    else
      formPanel.displayFieldData("miniConstraints");
  }
  private void setConditions() {    
    formPanel.displayFieldData("conditions");
  }
  private void setConstraints() {
    formPanel.displayFieldData("constraints");
  }
  */
  
  /**
   * Same as setUIConstruct, but remembers the setting of the miniConstraints
   */
  protected void resetUIConstruct(UIObject object) {
    if (currentView.equals(VIEW_MINIMAL)) {
      MiniConstraintPanel mcp = 
	(MiniConstraintPanel)formPanel.getFieldBit("miniConstraints");
      AbstractButton lastB = mcp.getLast();
      setUIConstruct(object);
      formPanel.setFieldData("miniConstraints", lastB.getText());
    }
    else setUIConstruct(object);
  }

  //---------------------User services----------------------------------------

  //************** these requests may come from the frame (menu selections)


  /**
   * User request to check the refinement - not implemented.
   */
  public boolean checkConstruct() {
    try {
      if (uiConstruct != null) uiConstruct.checkConsistency();
      return true;
    }
    catch (SyntaxException se) {
      JOptionPane.showMessageDialog(this, se.getMessage());
      return false;
    }
  }


  protected UIObject cloneConstruct(UIObject uiConstruct) {
    return ((UIRefinement)uiConstruct).cloneThis();
  }


  /**
   * User request to make a new action.
   */
  public UIObject makeNewConstruct() {
    return new UIRefinement(getUIDomain()); 
  }
  public List getAllConstructs() {
    if (getUIDomain() == null) return null;
    List constructs = getUIDomain().getAllRefinements();
    return new LinkedList(constructs);
  }
  public Named getNamedConstruct(String name) {
    if (getUIDomain() == null) return null;
    Object c = getUIDomain().getNamedRefinement(name);
    if ((c != null) && (c instanceof UIObject))
      return (Named)c;
    else return getUIDomain().newUIObject((IXObject)c);
  }


  /**
   *  User request to put the nodes of the current action into sequence.
   */
  public void userMakeSequence(){
    makeSequence();
  }
  /**
   *  User request to put the nodes of the current action in parallel.
   */
  public void userMakeParallel(){
    makeParallel();
  }

  public void userDeclareVariables() {
    //Debug.noteln("Declaring variables for refinement",uiConstruct);
    ensureDeclarationEditor();
    declarationEditor.start("Variable Declaration Editor",
			    uiConstruct, getEditorLocation());
  }


  //************** service sub-activity buttons (add, edit, delete)
  private void userAddChild() {
    //Debug.noteln("Adding children for refinement", uiConstruct);
    ensureNodeSpecEditor();
    nodeSpecEditor.start(uiConstruct, getEditorLocation());
  }
  private void userAddIssue(){
    //Debug.noteln("Adding children for refinement", uiConstruct);
    ensureIssueSpecEditor();
    issueSpecEditor.start(uiConstruct, getEditorLocation());
  }

  /**
   * Parse the nodes from the text area and store them in the current
   * refinement.
   */
  public void noteTextNodes() {
    //Debug.noteln("Noting nodes from text");
    JTextArea nt = (JTextArea)nodesTextPanel.getItemComponent();
    List nodes = UIRefinement.parseNodes(getUIDomain(), nt.getText());
    uiConstruct.updateNodes(nodes); //not setNodes; keep old ids for orderings!
    if (isSequenceSet()) makeSequence();
  }

  private void userDeleteChild(){
    IXEditorPanel nodesPanel = formPanel.getFieldBit("nodes");
    nodesPanel.minimise();
    JList list = (JList) nodesPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a sub-activity to delete first.";
      JOptionPane.showMessageDialog(this,message);} 
    else {
      //Debug.noteln("AEP: user deleting child");
      Object[] values = list.getSelectedValues();
      for (int i = 0; i < values.length; i++){
	Object child = values[i];
        uiConstruct.deleteNode((NodeSpec)child);
      }
      list.clearSelection();
      //***this.setChildren();
      //***this.setOrderings();
    }
  }
  private void userDeleteIssue(){
    IXEditorPanel issuesPanel = formPanel.getFieldBit("issues");
    JList list = (JList) issuesPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select an issue to delete first.";
      JOptionPane.showMessageDialog(this,message);} 
    else {
      Object[] values = list.getSelectedValues();
      for (int i = 0; i < values.length; i++){
	Object child = values[i];
        uiConstruct.deleteIssue((Issue)child);
      }
      list.clearSelection();
      //***this.setIssues();
    }
  }
  private void userEditChild(){
    //notImp("editing nodes");
    IXEditorPanel nodesPanel = formPanel.getFieldBit("nodes");
    JList list = (JList) nodesPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a sub-activity to edit first.";
      JOptionPane.showMessageDialog(this,message);} 
    else {
      Object[] values = list.getSelectedValues();
      ensureNodeSpecEditor();
      nodeSpecEditor.start(uiConstruct, values[0], getEditorLocation());
    }
  }
  private void userEditIssue(){
    //notImp("editing issues");
    IXEditorPanel issuesPanel = formPanel.getFieldBit("issues");
    JList list = (JList) issuesPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select an issue to edit first.";
      JOptionPane.showMessageDialog(this,message);} 
    else {
      Object[] values = list.getSelectedValues();
      ensureIssueSpecEditor();
      issueSpecEditor.start(uiConstruct, values[0], getEditorLocation());
    }
  }
  private void userEditOrdering(){
    //notImp("editing issues");
    IXEditorPanel cPanel = formPanel.getFieldBit("orderings");
    JList list = (JList) cPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a constraint to edit first.";
      JOptionPane.showMessageDialog(this,message);} 
    else {
      Object[] values = list.getSelectedValues();
      ensureRelationEditor();
      orderingEditor.start(uiConstruct, values[0], getEditorLocation()); 
    }
  }
  private void userEditCondition(){
    //Debug.noteln("AEP: editing condition/effect");
    IXEditorPanel cPanel = formPanel.getFieldBit("conditions");
    JList list = (JList) cPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a condition to edit first.";
      JOptionPane.showMessageDialog(this,message);} 
    else {
      Object[] values = list.getSelectedValues();
      ensureConditionEditor();
      conditionEditor.start(uiConstruct, values[0], getEditorLocation()); 
    }
  }
  private void userEditConstraint(){
    //notImp("editing other constraints");
    IXEditorPanel cPanel = formPanel.getFieldBit("constraints");
    JList list = (JList) cPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a constraint to edit first.";
      JOptionPane.showMessageDialog(this,message);} 
    else {
      Object[] values = list.getSelectedValues();
      ensureConstraintEditor();
      constraintEditor.start(uiConstruct, values[0], getEditorLocation()); 
    }
  }
  private void userEditObjectConstraint(){
    //notImp("editing object constraints");
    IXEditorPanel cPanel = formPanel.getFieldBit("constraints");
    JList list = (JList) cPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a constraint to edit first.";
      JOptionPane.showMessageDialog(this,message);} 
    else {
      Object[] values = list.getSelectedValues();
      ensureObjectConstraintEditor();
      objectConstraintEditor.start(uiConstruct,values[0],getEditorLocation()); 
    }
  }

  private void userClearChildren(){
    // if the refinement has no children, just reset the field to be safe
    List nodes = uiConstruct.getNodes();
    if ((nodes == null) || (nodes.isEmpty())) {
      //***this.setChildren();
      return;
    }
    // if the action has children, get the user to confirm
    String[] message = {"This will clear all sub-activities.",
			" Are you sure you want to do this?"};
    int answer = JOptionPane.showConfirmDialog(this, message, "Warning", 
					       JOptionPane.OK_CANCEL_OPTION, 
					       JOptionPane.WARNING_MESSAGE);
    if (answer == JOptionPane.OK_OPTION) {
      uiConstruct.setNodes(Lisp.NIL);
      uiConstruct.setOrderings(Lisp.NIL);
      //***this.setChildren();
      //***this.setOrderings();
    }
  }

  //************** service constraints buttons (add, edit, delete)
  private void userAddOrdering(){
    //Debug.noteln("Adding constraints for refinement",uiConstruct);
    ensureRelationEditor();
    orderingEditor.start(uiConstruct, getEditorLocation());
  }
  private void userAddCondition(){
    if (((DomainEditorFrame)parent).useNewCondEd) 
      userAddObjectConstraint();    
    else userAddSimpleCondition();
  }
  private void userAddConstraint(){
    //userAddObjectConstraint();
    //Debug.noteln("Adding free-form constraint for refinement",uiConstruct);
    ensureConstraintEditor();
    constraintEditor.start(uiConstruct, getEditorLocation());
  }
  private void userAddObjectConstraint(){
    //Debug.noteln("Adding object constraint for refinement",uiConstruct);
    ensureDeclarationEditor();
    declarationEditor.start("Object Constraint Editor", 
			    uiConstruct, getEditorLocation());
  }
  private void userAddSimpleCondition(){
    //Debug.noteln("Adding free-form condition for refinement", uiConstruct);
    ensureConditionEditor();
    conditionEditor.start(uiConstruct, getEditorLocation());
  }
  //obsolete
  private void userAddSimpleObjectConstraint(){
    //Debug.noteln("Adding simple object constraint for",uiConstruct);
    ensureObjectConstraintEditor();
    objectConstraintEditor.start(uiConstruct, getEditorLocation());
  }

  private void userDeleteOrdering(){
    IXEditorPanel cPanel = formPanel.getFieldBit("orderings");
    JList list = (JList) cPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a constraint to delete first.";
      JOptionPane.showMessageDialog(this,message);} 
    else {
      Object[] values = list.getSelectedValues();
      for (int i = 0; i < values.length; i++){
	Object relation = (Object) values[i];
	List orderings = new ArrayList(uiConstruct.getOrderings());
	orderings.remove(relation);
	uiConstruct.setOrderings(orderings);
      }
      list.clearSelection();
      //***this.setOrderings();
    }
  }
  private void userDeleteCondition(){
    IXEditorPanel cPanel = formPanel.getFieldBit("conditions");
    JList list = (JList) cPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a condition to delete first.";
      JOptionPane.showMessageDialog(this,message);} 
    else {
      Object[] values = list.getSelectedValues();
      for (int i = 0; i < values.length; i++){
	Object relation = (Object) values[i];
	List orderings = new ArrayList(uiConstruct.getConditions());
	orderings.remove(relation);
	uiConstruct.setConditions(orderings);
      }
      list.clearSelection();
      //***this.setConditions();
    }
  }
  private void userDeleteConstraint(){
    IXEditorPanel cPanel = formPanel.getFieldBit("constraints");
    JList list = (JList) cPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a constraint to delete first.";
      JOptionPane.showMessageDialog(this,message);} 
    else {
      Object[] values = list.getSelectedValues();
      for (int i = 0; i < values.length; i++){
	Object relation = (Object) values[i];
	List orderings = new ArrayList(uiConstruct.getConstraints());
	orderings.remove(relation);
	uiConstruct.setConstraints(orderings);
      }
      list.clearSelection();
      //***this.setConstraints();
    }
  }

  protected boolean clearOrderings() {
    // if the action has no constraints, just reset the field to be safe
    List orderings = uiConstruct.getOrderings();
    if ((orderings == null) || (orderings.isEmpty())) {
      //***this.setOrderings();
      return true;
    }
      // if the action has constraints, get the user to confirm
    String[] msg = {"This will clear all orderings between sub-activities.",
		    " Are you sure you want to do this?"};
    int answer = JOptionPane.showConfirmDialog(this, msg, "Warning", 
					       JOptionPane.OK_CANCEL_OPTION, 
					       JOptionPane.WARNING_MESSAGE);
    if (answer == JOptionPane.OK_OPTION) {
      uiConstruct.setOrderings(Lisp.NIL);
      //***this.setOrderings();
      return true;
    }
    return false;
  }



  //---------------------Variables----------------------------------------

  /**
   * The variable list determines whether variables are checked or not.
   * - If the list is set, new variables are checked against it.
   * - If the list is set to ampty, no variables are allowed.
   * - If the list is not set, any variable is allowed.
   * perhaps provide a "set variable list from current used" to freeze names.
   */

  //---------------------UIDomainListener things------------------------------

  /** 
   * Called when a UI construct is added. 
   * Use getSource() to get at the domain (UI/IX) in which the event occurred.
   * Use getObject() to get at the object (UI/IX) that has been added.
   */
  public void constructAdded(UIDomainEvent event){}
  public void constructAdded(DomainEvent event){}
  /** 
   * Called when a UI construct is edited. 
   * Use getSource() to get at the UIDomain in which the event occurred.
   * Use getObject() to get at the UIObject that has been edited.
   */
  public void constructEdited(UIDomainEvent event){}
  /** 
   * Called when a UI construct is removed. 
   * Use getSource() to get at the UIDomain in which the event occurred.
   * Use getObject() to get at the UIObject that has been removed.
   */
  public void constructRemoved(UIDomainEvent event) {   
    if (event.getObject().equals(uiConstruct)) 
      clearUIConstruct();
  }
  public void domainCleared(UIDomainEvent event) {
    //Debug.noteln("AEP: got cleared domain");
    clearUIConstruct();
  }
  public void domainEdited(UIDomainEvent se) {
  }
  public void domainSet(UIDomainEvent event) {
    //Debug.noteln("AEP: got domain set event. New:", getUIDomain().print());
    clearUIConstruct();
  }

  public void updateTree() {
    treeModel.reloadData();
    updateTreeSelection();
  }
  public void updateTreeSelection() {
    //Debug.noteln("AEP: updating tree selection to", uiConstruct);
    Object o = actionTree.getSelectedObject();
    if ((o != null) && o.equals(uiConstruct)) return;
    if ((o == null) && (uiConstruct == null)) return;
    actionTree.setSelectedObject(uiConstruct);
  }

  //---------------------Listeners----------------------------------------

  /** Listeners interested in the current action */
  HashSet actionListeners = new HashSet();

  public void addCurrentActionListener(CurrentActionListener cal) {
    actionListeners.add(cal);
  }

  private void fireCurrentActionChanged(UIRefinement old, UIRefinement act) {
    //Debug.noteln("AEP: current action changed", old + " to " + act);
    //if (((old == null) && (act == null)) ||
    //  ((old != null) && (old.equals(act)))) return;
    //after undefined refinements the above does not communicate current sel.
    if ((old == null) && (act == null)) return;
    //actual change, so pass on to interested parties
    //Debug.noteln("AEP: firing action changed" , old + " -> " + act);
    for (Iterator i=actionListeners.iterator(); i.hasNext();) {
      CurrentActionListener cal = (CurrentActionListener)i.next();
      cal.actionChanged(this, old, act);
    }
    if ((act == null) || act.isEmpty()) actionTree.clearSelection();
  }

  public void actionChanged(Component source, 
			    UIRefinement old, UIRefinement newAction) {
    if ((newAction != null) && newAction.equals(old)) return;
    if ((newAction == null) && (old == null)) return;
    Debug.noteln("AEP: action change in", source);
    //Debug.noteln("AEP: action changed to", newAction);
    //make sure the tree is in synch (tree clicks always in synch anyway)
    Object currentTreeOb = actionTree.getSelectedObject();
    if ((!this.equals(source) || newAction == null || newAction.isEmpty()))
	actionTree.clearSelection();
    if (((currentTreeOb == null) && !(newAction == null))
	|| !currentTreeOb.equals(newAction)) 
      actionTree.setSelectedObject(newAction);
    if (!this.equals(source)) //two things need to know, so do listen!
      setUIConstruct(newAction);
  }

  //---------------------Actions----------------------------------------

  //************** relay buttons to their services *************************
  class EdMouseAdapter extends java.awt.event.MouseAdapter 
  implements SchemaTerms {
    public void mouseClicked(MouseEvent event){
      Component component = event.getComponent();
      mousePoint = event.getPoint();
      //component.requestFocus();
      //Debug.noteln("ActionEditorPanel: mouse from ", component.getClass());
      if (!AbstractButton.class.isInstance(component)) return;
      
      AbstractButton button = (AbstractButton) component;
      //Debug.noteln("AEP: Button command " + button.getText());
      IXEditorPanel buttonParent = 
	(IXEditorPanel)button.getParent().getParent();
      //Debug.noteln(" Parent name " + buttonParent.getName());

      if (buttonParent.getName().equals("Nodes")) {
	if (button.getText() == "Add") 
	  ActionEditorPanel.this.userAddChild();
	else if (button.getText() == "Delete") 
	  ActionEditorPanel.this.userDeleteChild();
	else if (button.getText() == "Edit") 
	  ActionEditorPanel.this.userEditChild();
	else 
	  ActionEditorPanel.this.notImp(button.getText() + " for nodes");
	return;
      }
      else if (buttonParent.getName().equals("Issues")) {
	if (button.getText() == "Add") 
	  ActionEditorPanel.this.userAddIssue();
	else if (button.getText() == "Delete") 
	  ActionEditorPanel.this.userDeleteIssue();
	else if (button.getText() == "Edit") 
	  ActionEditorPanel.this.userEditIssue();
	else 
	  ActionEditorPanel.this.notImp(button.getText() + " for issues");
	return;
      }
      else if ((buttonParent.getName().equals("Ordering")) 
	       || (buttonParent.getName().equals("Orderings"))) {
	//Debug.noteln("AEP: got an ordering button");
	if (button.getText() == "Add") 
	  ActionEditorPanel.this.userAddOrdering();
	else if (button.getText() == "Delete")
	  ActionEditorPanel.this.userDeleteOrdering();
	else if (button.getText() == "Edit") 
	  ActionEditorPanel.this.userEditOrdering();
        else if (button.getText().equals(O_PARALLEL)) {
	  ((MiniConstraintPanel)buttonParent).setLast(button);
	  ActionEditorPanel.this.setOrdering(false);
	}
	else if (button.getText().equals(O_SEQUENCE)) {
	 ((MiniConstraintPanel)buttonParent).setLast(button);
	  ActionEditorPanel.this.setOrdering(true);
	}
	else if (button.getText() == "other") {}
	else if (button.getText() == "Comprehensive") {
	  //Debug.noteln("AcEP-MA: parent.setActivityView", "true");
	  ActionEditorPanel.this.parent.setActivityView(VIEW_COMPREHENSIVE);
	}
	else
	  ActionEditorPanel.this.notImp(button.getText() + " for orderings");
	return;
      }     
      else if (buttonParent.getName().equals("Conditions/Effects")) {
	//Debug.noteln("AEP: got a condition button");
	if (button.getText() == "Add") 
	  ActionEditorPanel.this.userAddCondition();
	else if (button.getText() == "Delete")
	  ActionEditorPanel.this.userDeleteCondition();
	else if (button.getText() == "Edit") 
	  ActionEditorPanel.this.userEditCondition();
	return;
      }
      else if (buttonParent.getName().equals("Constraints")
	       || buttonParent.getName().equals("Other Constraints")
	       || buttonParent.getName().equals("Constraints (Other)")) {
	
	//Debug.noteln("AEP: got a general constraints button");
	if (button.getText() == "Add") 
	  ActionEditorPanel.this.userAddConstraint();
	else if (button.getText() == "Delete")
	  ActionEditorPanel.this.userDeleteConstraint();
	else if (button.getText() == "Edit") 
	  ActionEditorPanel.this.userEditConstraint();
	return;
      }
      else ActionEditorPanel.this.notImp("Command " + button.getText());
    }
  }
 
  /**
   * Recognises panel specific events and processes them.
   * This is a separate method because it must let other event handlers know
   * whether it recognised the event.
   *
   * @param event the event to be processed.
   * @return true if the event was recognised, false if not.
   */
  public boolean frameActionPerformed(ActionEvent event){
    //deal with commands
    String command = event.getActionCommand();
    //Debug.noteln("AcEP: action command is", command);
    if (command == "newEdit") newConstruct();
    else if (command == "editConstruct") {
      //get the construct name from the item text and show it
      JMenuItem item = (JMenuItem)event.getSource();
      String name = item.getText().trim();
      setUIConstruct((UIObject)getNamedConstruct(name));  
    }
    else if (command == "cut") deleteConstruct();
    else if (command == "copy") copyConstruct();
    else if (command == "modify") saveConstruct();
    else if (command == "revert") revertConstruct();
    else if (command == "revertO") revertOConstruct();
    else if (command == "check") checkConstruct();
    else if (command == "search") userSearch();
    else if (command == "declareVars") userDeclareVariables();

    else if (command == "makeSequence") userMakeSequence();
    else if (command == "makeParallel") userMakeParallel();
    else if (command == "defSequence") {
      Object object = event.getSource();
      parent.setDefaultSequence(((JCheckBoxMenuItem)object).getState());
    }
    else if ((command == "graphical") && !updatingView) {
      updatingView = true;
      try {
	parent.setActivityView(VIEW_GRAPHICAL);
	//setGraphical();
	//notImp("The graphical activity editor");
	updatingView = false;
      }
      catch (Exception e) {	updatingView = false;      }
    }
    else if ((command == "minimal") && !updatingView) {
      updatingView = true;
      //Debug.noteln("AcEP-fAP: parent.setActivityView", VIEW_MINIMAL);
      try {
	parent.setActivityView(VIEW_MINIMAL);
	updatingView = false;
      }
      catch (Exception e) {	updatingView = false;      }
    }
    else if ((command == "comprehensive") && !updatingView) {
      updatingView = true;
      //Debug.noteln("AcEP-fAP: parent.setActivityView", VIEW_COMPREHENSIVE);
      try {
	parent.setActivityView(VIEW_COMPREHENSIVE);
	updatingView = false;
      }
      catch (Exception e) {	updatingView = false;      }
    }
    else if (command.startsWith("Expansions for ")) { //from tree popup
      try {
	return goToExpansion(((AbstractButton)event.getSource()).getText());
      }
      catch (Exception e) { Debug.noteException(e); return false;}
    }
    else if (command == "New Expansion") {
      JMenuItem item = (JMenuItem)event.getSource();
      try {
	editConstruct(((JObjectPopup)item.getParent()).getObject());
	return true;
      }
      catch (Exception e) {
	Debug.noteException(e);
	Debug.noteln("menu item's parent is", item.getParent());
	return false;
      }
    }
    else return false;
    return true;

  }

  public void populatePopup(JObjectPopup popup, Object construct) {
    if (construct instanceof UINodeSpec) {
      List exps = getAllExpansions((UINodeSpec)construct);
      String cname = NodeNameRenderer.makeText((UINodeSpec)construct);
      JObjectMenu expansionsMenu = null;
      try {
	expansionsMenu = (JObjectMenu)popup.getComponent(1);
      }
      catch (Exception e) {
	expansionsMenu = new JObjectMenu("Expansions");
	popup.add(expansionsMenu);
      }
      UIUtil.populateMenu(parent, expansionsMenu, construct, 
			  "Expansions" + " for " + cname, exps);
      popup.setObject(construct);
      popup.setLabel(construct.toString());
    }
  }

  private boolean goToExpansion(String name) {
    UIObject uir = (UIObject)getNamedConstruct(name);  
    if (uir == null) return false;
    else setUIConstruct(uir);
    return true;
  }

  /**
   * Wakes up the ActionListener with a user action.
   * This is called when a KeyStroke or other panel event happens in which
   * the ActionListener registered its interest.
   */
  public void actionPerformed(ActionEvent ae){
    // give the method above a chance to recognise the event.
    if (frameActionPerformed(ae)) return;
    String command = ae.getActionCommand();
    Debug.noteln("ActionEditorPanel: Got action command", command);
    if (command.equals("add node")) userAddChild();
    else if (command.equals("delete node")) userDeleteChild();
    else if (command.equals("add ordering")) userAddOrdering();
    else if (command.equals("delete ordering")) userDeleteOrdering();
    else Debug.noteln("ActionEditorPanel cannot recognise command", command);
  }


  protected static void notImp(String message) {
    UIUtil.notImplemented(null,message);
  }

  public void dataChanged(EditableObject object, String field, 
			  Object oldValue, Object newValue) {
    //super.dataChanged(object, field, oldValue, newValue);

    if ((field != null) && 
	(field.equals("name") || field.equals("pattern") 
	 || field.equals("nodes"))) {
      //Debug.noteln("Data changed, so update tree",object + " " + field);
      updateTree();
    }
  }

}

/*Issues***************************************************************
 *
 * This is no longer a JConstraintListener - that listening is now
 * done via DataChangeListener in AConstructFramePanel (ConstraintEditors
 * do proper updates now)
 *
 * NOTE: this does not do checkConsistency on new constructs anymore.
 *
/*Todos***************************************************************
 *
 ** Investigate setUIConstruct and setUIConstructPanel and setUIConstructOnly.
 *   also work out why they are different from AConstructFramePanel
 ** pull apart ConstructFraming, AConstructFramePanel and ConstructEditing
 *
 ** edit pattern: patterns should have constraints on their structure. 
 *   e.g. <verb> <variables>|<other things>
 ** lists etc. should use MVC 
 *
 *
 ** simple/advanced/graphical should be done via "mode" switch
 ** documentation of simple/advanced/graphical and their ConstructPanels things
 *****************************************************************************
 */
