/****************************************************************************
 * Support for editing sets of named objects (e.g. refinements or
 * object classes)
 *
 * File: AbstractUISetEditor.java
 * @author Jussi Stader
 * @version 4.3
 * Updated: Thu Oct 12 11:22:59 2006
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import java.util.*;
import java.lang.reflect.Field;
import java.awt.Component;
import java.awt.Cursor;
import javax.swing.JOptionPane;
import javax.swing.JFileChooser;
import java.io.*;

import ix.iview.*;
import ix.iview.util.*;
import ix.iview.event.*;
import ix.iview.domain.*;
import ix.iview.domain.event.*;
import ix.icore.domain.event.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.iface.domain.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.ui.event.*;
import ix.util.*;
import ix.util.match.*;
import ix.util.lisp.*;

/**
 * Support for editing sets of named objects (e.g. refinements or
 * object classes).  Looks after adding/removing/editing the objects
 * and keeping the set consistent. Handles overwrites, finding objects
 * in the set, working out whether the set has changed.
 */

public abstract class AbstractUISetEditor extends AbstractSetEditor 
{ 
  public AbstractUISetEditor(DomainObjectManager manager, LinkedList objects) {
    super((ObjectManager)manager, objects);
  }

  public void setManager(DomainObjectManager manager, LinkedList objects) {
    super.setManager(manager, objects);
  }

  public void setObjects(LinkedList objects) {
    //get all relevant objects in their order and note them in nodes
    //Debug.noteln("UISetEd: setting Objects", this);
    clear();
    members.clear();
    if (objects == null) return;
    for (Iterator i = objects.iterator(); i.hasNext(); ) {
      Named r = (Named)i.next();
      members.put(r.getName(), r); 
    }
    //Debug.noteln(" Objects now", members);
  }

  public Domain getDomain() {
    return ((DomainObjectManager)manager).getDomain();
  }
  public UIDomain getUIDomain() {
    return ((DomainObjectManager)manager).getUIDomain();
  }

    public Object getOriginal(Object uiOb) {
	if (uiOb instanceof UIObject) {
	    String base = ((UIObject)uiOb).getBaseReference().toString();
	    return getStringOriginal(base);
	}
	else return super.getOriginal(uiOb);
    }

  public void rememberClear() {
    Debug.noteln("UISetEd: remembering clear");
    if (getDomain() != null) removed.addAll(getDomainOriginals());
  }


  //-------------------------abstracts---------------------------------------
  /** Gets all original base objects, e.g. all Refinements from Domain */
  public abstract LinkedList getDomainOriginals();

  /** Makes an list of original objects */
  public abstract LinkedList makeOriginalList();
  /**
   * Finds a construct whose name matches the given string.
   * @return an UIObject, an IXObject, or null - depending on which exists.
   */
  public abstract Named getNamedObject(String name);



  public String getName(Object object) {
    if (object instanceof Named) return ((Named)object).getName();
    else return "";
  }
  public void setName(Object object, String name) {
    if (object instanceof EditableObject) 
      ((EditableObject)object).setValue("name", name);
  }


  public abstract Named getStringOriginal(String name);
  public Object getStringObject(String name) {
    return getNamedObject(name);
  }

  /** Checks whether the given object has an original base object */
  public boolean hasOriginal(Named object) {
    if (object instanceof UIObject) return ((UIObject)object).hasOriginal();
    else return false;
  }
  /** Checks whether the given object is empty */
  public boolean isEmpty(Object object) {
    if (object instanceof UIObject) return ((UIObject)object).isEmpty();
    else return false;
  }

  public boolean isUndefined(Object object) {
    if (object instanceof UIObject) return ((UIObject)object).isUndefined();
    else return true;
  }

  public IXObject getBaseObject(Named object) {
    if (object instanceof UIObject) return ((UIObject)object).getBaseObject();
    else return null;
  }

  public Object cloneConstruct(Object o) throws CloneNotSupportedException {
    if (o instanceof Refinement) return ((Refinement)o).clone();
    else if (o instanceof ObjectClass) return ((ObjectClass)o).clone();
    else return null;
  }

  //--------------------------Handling changes--------------------------------
  

  public void saveToDomain(Domain domain) {
    try {
      for (Iterator i=removed.iterator(); i.hasNext();) {
	Object o = i.next();
	//Debug.noteln("UID: removing object", o.toString());
	if ((o != null) && (o instanceof UIObject))
	  ((UIObject)o).removeFromDomain(domain);
	else removeFromDomain(domain, o);
      }
      for (Iterator i=edited.iterator(); i.hasNext();) {
	Object o = i.next();
	//Debug.noteln("UID: updating object", o.toString());
	if ((o != null) && (o instanceof UIObject))
	  ((UIObject)o).updateInDomain(domain);
      }
      for (Iterator i=added.iterator(); i.hasNext();) {
	Object o = i.next();
	//Debug.noteln("UID: adding object", o.toString());
	if ((o instanceof UIObject) && (!((UIObject)o).isEmpty()))
	  ((UIObject)o).addToDomain(domain);
      }
      //Debug.noteln("UID - finished saveToDomain(domain)");
    }
    catch (DomainReferenceException re) {
      Debug.noteln(Debug.foldException(re));
      JOptionPane.showMessageDialog(null, Debug.foldException(re));
      //Debug.describeException(re);
    }

  }


  public void removeFromDomain(Domain domain, Object object) {
    if (object == null) return;
    if (object instanceof Refinement) 
      domain.deleteNamedRefinement(((Refinement)object).getName());
    else if (object instanceof ObjectClass) {
      domain.deleteNamedObjectClass(((ObjectClass)object).getName());
    }
  }


  /** 
   * Collects descriptions of any changes in constructs in the set.
   */
  public List collectConstructChanges() {
    List changes = new ArrayList();
    if (hasChangedConstructs()) {
      changes.add("Changes in " + label);
      if (!added.isEmpty()) 
	changes.add(" Added: " + IVUtil.namedListToDisplay(added));
      if (!removed.isEmpty()) 
	changes.add(" Removed: " + IVUtil.namedListToDisplay(removed));
      if (!edited.isEmpty()) 
	changes.add(" Edited: " + IVUtil.namedListToDisplay(edited));
    }
    return changes;
  }

  //--------------------------Notifying listeners-----------------------------


  public void fireConstructRemoved(Object object) {
    //Debug.noteln("UISetEd firing construct removed");
    if (object instanceof UIObject)
      manager.fireConstructRemoved(this, (UIObject)object);
    else
      Debug.noteln("UISE: Deleted object without UI version. Not firing.",
		   object);
  }

}


// Issues:
// * tidy up load/insert domain stuff 
// * 
// * 
