/****************************************************************************
 * Support for editing sets of objects (e.g. constraints)
 *
 * File: AbstractSetEditor.java
 * @author Jussi Stader
 * @version 4.3
 * Updated: Mon Nov  6 13:01:17 2006
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iview;

import java.util.*;
import java.lang.reflect.Field;
import java.awt.Component;
import java.awt.Cursor;
import javax.swing.JOptionPane;
import javax.swing.JFileChooser;
import java.io.*;

import ix.iview.*;
import ix.iview.util.*;
import ix.iview.event.*;
import ix.iview.domain.*;
import ix.iview.domain.event.*;
import ix.icore.domain.event.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.iface.domain.*;
import ix.iface.ui.*;
import ix.iface.ui.util.*;
import ix.iface.ui.event.*;
import ix.util.*;
import ix.util.match.*;
import ix.util.lisp.*;

/**
 * Support for editing sets of objects (e.g. constraints)
 * Looks after adding/removing/editing the objects
 * and keeping the set consistent. Handles overwrites, finding objects
 * in the set, working out whether the set has changed.
 */

public abstract class AbstractSetEditor { 

  public ObjectManager manager;  //the thing interested in the members

  private boolean overwrite = false; //may be set for bulk overwrts when loadng

  protected String label = "construct"; // set for messages

  /** Constants for handling overwrite */
  protected static final int CANCEL = 0;
  protected static final int ADD = 1;
  protected static final int OVERWRITE = 2;
  protected static final int NEWNAME = 3;

  /**
   * List of objects in the editor. Needed to preserve input order.
   */
  protected StableHashMap members = new StableHashMap();

  /** list of Objects added to the set (unless removed).*/
  protected HashSet added = new HashSet();
  /** list of Objects edited in the set (unless added). */
  protected HashSet edited = new HashSet();
  /** list of Objects removed from the set (unless added). */
  protected HashSet removed = new HashSet();
  /** original to copy map; note that new Objects do *not* appear */
  protected HashMap originals = new HashMap();


  public AbstractSetEditor(ObjectManager manager, LinkedList objects) {
    super();
    setManager(manager, objects); //load objects and set up oritinals etc.
  }

  public void setManager(ObjectManager manager, LinkedList objects) {
    this.manager = manager;
    //setObjects below may need manager to know about us already. HACK
    manager.noteSetEditor(this);
    setObjects(objects); 
  }

  public void setObjects(LinkedList objects) {
    //get all relevant objects in their order and note them in nodes
    //Debug.noteln("SetEd: setting Objects", this);
    clear();
    if (manager != null) {
      members.clear();
      if (objects == null) return;
      for (Iterator i = objects.iterator(); i.hasNext(); ) {
	Object m = i.next();
	if (m != null) { //ignore null objects
	  Object copyM = m;
	  try {copyM = cloneConstruct(m);
	  } catch (CloneNotSupportedException e) { }
	  members.put(getName(copyM), copyM); 
	  originals.put(m, copyM);
	}
      }
    }
  }

  public void clear(){
    resetEdits();
    originals.clear();
  }

  public void resetEdits() {
    added.clear();
    edited.clear();
    removed.clear();
  }

  public boolean isLoading() {
    return manager.isLoading();
  }
  public void setOverwrite(boolean onOff) {
    overwrite = onOff;
  }

  //-------------------------abstracts---------------------------------------
  /** Checks whether the given object is one of this editor's own,
   * e.g. UIRefinement */
  public abstract boolean isOwnObject(Object object); // true for own objects
  /** Checks whether the two given objects have the same specifications. */
  public abstract boolean sameConstruct(Object one, Object other);
  /** Makes an own object from the given one */
  public abstract Object makeOwnObject(Object object);
  /** Makes a list of original objects */
  public abstract LinkedList makeOriginalList();
  /** Gets a string representation of the object to identify it uniquely */
  public abstract String getName(Object object);
  /** Sets a string representation of the object that identifies it uniquely */
  public abstract void setName(Object object, String name);
  /** Sets a string representation of the object that identifies it uniquely */
  public abstract void setLegalName(Object object, String name);
  /**
   * Finds a construct whose getName(Object) matches the given string.
   * @return an Object if it exists, otherwise null.
   */
  public abstract Object getStringObject(String name);
  /** Checks whether the given object is empty */
  public abstract boolean isEmpty(Object object);
  /** Checks whether the given object is undefined */
  public abstract boolean isUndefined(Object object);
  /** Clones the given construct so as not to overwrite the original. */
  public abstract Object cloneConstruct(Object o) 
      throws CloneNotSupportedException;


  public boolean isCurrentOwnObject(Object o) {
    if (removed.contains(o)) return false;
    //added or edited, check if it matches
    return (edited.contains(o) || added.contains(o));
  }


  /** Checks whether the given object has an original base object */
  public boolean hasOriginal(Object object) {
    if ((originals.values() != null) 
	&& originals.values().contains(object)) return true;
    else return false;
  }
  /** 
   * Gets all original objects from the originals map. If this map is built 
   * in a lazy way (as for Refinements), make sure that this is overwiritten 
   * to look in the original domain.
   */
  public LinkedList getOriginals() {
    if ((originals == null) || (originals.keySet() == null)) return null;
    else return new LinkedList(originals.keySet());
  }
  /** 
   * Finds the original version of the given object in the originals map
   */
  public Object getOriginal(Object object) {
    if ((object != null) && (originals.values() != null)
	&& originals.values().contains(object)) {
      for (Iterator i = originals.keySet().iterator(); i.hasNext(); ) {
	Object original = i.next();
	if (object.equals(originals.get(original))) 
	  return original;
      }
    }
    return null;
  }

  /** Checks whether there are objects in the set */
  public boolean isEmpty() {
    List objects = getAllConstructs();
    return ((objects == null) || objects.isEmpty());
  }
  

  //--------------------------Handling changes--------------------------------
  /**
   * Adds a given construct to the set unless it is already there or
   * undefined. Checks originals, added and edited ones.
   */
  public void ensureConstruct(Object object) {
    //if added, edited, or noted as having original, nothing to do.
    //Debug.noteln("AbSE: Ensuring construct", object);
    if (added.contains(object)) return;
    if (edited.contains(object)) return;
    List origs = getOriginals();
    if ((origs != null) && origs.contains(object)) return;
    if ((originals.values() != null) && originals.values().contains(object)) 
      return;
    //add unless empty or undefined
    if (!isEmpty(object) && !isUndefined(object)) {
      addConstruct(object);
    }
  }
  /**
   * Adds a given construct to the set.
   * Checks two things: whether the object is empty (nothing done), and 
   * whether it should overwrite an existing one with the same name.
   * @return false if nothing done, true if added (inc mod)
   */
  public boolean addConstruct(Object object) {
    //Debug.noteln("SE: addConstruct", object.print());
    //Debug.noteln(" Added now: ", UIUtil.listToDisplay(added));
    //Debug.noteln(" Edited now: ", UIUtil.listToDisplay(edited));
    //Debug.noteln(" Removed now: ", UIUtil.listToDisplay(removed));
    int toAdd = handleOverwrite(object);
    if ((toAdd == NEWNAME) || (toAdd == OVERWRITE)) { //start again
      //if overwriting, old one has been removed
      return addConstruct(object);
    }
    else if (toAdd == CANCEL) return false; //stop
    
    //toAdd == ADD, so add normally
    
    members.put(getName(object), object);
    //toAdd == ADD, so add normally
    //Debug.noteln(" Base object is", object.getBaseObject());
    //previously removed, so just forget the remove but remember changes
    if (removed.contains(object)) {
      removed.remove(object);
      fireConstructEdited(object);
    }
    else {
      Object original = getOriginal(object);
      if (removed.contains(original)) {
	removed.remove(original);
	fireConstructEdited(object);
      }
      else {
	if (added.add(object)) fireConstructAdded(object);
	else {
	  //Debug.noteln("SetEd: object already added");
	  fireConstructEdited(object); 
	}
      }
    }
    return true;
    //Debug.noteln(" Added now: ", UIUtil.listToDisplay(added));
    //Debug.noteln(" Edited now: ", UIUtil.listToDisplay(edited));
    //Debug.noteln(" Removed now: ", UIUtil.listToDisplay(removed));
  }

  /**
   * Determines whether the given object has a name that is already in
   * the set. Ensures that the old object has not been removed (if it has, 
   * its name can be reused).
   * @return the object with the same name if one can be found, null
   * if not.
   **/
  private Object getNameClashObject(Object object) {
    return getNameClashObject(object, getName(object)); 
  }
  /**
   * Determines whether the given object has a name that is already in
   * the set. Ensures that the old object has not been removed (if it has, 
   * its name can be reused).
   * @return the object with the same name if one can be found, null
   * if not.
   **/
  private Object getNameClashObject(Object object, String name) {
    Object oldObject = getStringObject(name); 
    if ((oldObject != null) && //there is an old object
	(!oldObject.equals(object)) && //not the new one
	(!oldObject.equals(getOriginal(object))) && //not new one's original
	(!removed.contains(oldObject))) //not removed
      return oldObject;
    else return null;
  }

  /**
   * Handles overwrites for objects that are not alerady in the domain. Use 
   * the next (handleOverwrite(object, newName)) for ones that are.
   * @returns ADD if the object is to have the new name (no clash);
   * NEWNAME if the user has been asked for a new name (check this one)
   * CANCEL if the user cancelled the whole operation,
   * OVERWRITE if the clashing object has been removed, i.e. try again.
   */
  private int handleOverwrite(Object object) {
    return handleOverwrite(object, getName(object));
  }
  /**
   * Handles overwrites for objects that may alerady be in the domain. 
   * @returns ADD if the object is to have the new name (no clash);
   * NEWNAME if the user has been asked for a new name (check this one)
   * CANCEL if the user cancelled the whole operation,
   * OVERWRITE if the clashing object has been removed, i.e. try again.
   */
  //The first dialog should not be YES_NO_CANCEL but YES_NO_YES-ALL
  private int handleOverwrite(Object object, String newName) {
    String name = newName;
    //Debug.noteln("SetEd: handleOverwrite");
    //(new Throwable()).printStackTrace();
    /*
    if (((object == null) || isEmpty(object))
	&& ((newName == null) || newName.equals("")))
      return CANCEL; //dont put in empty objects
    */
    //no name: get a name and start again (or give up)
    if ((newName == null) || newName.equals("")) {
      //(new Throwable()).printStackTrace();
      //needs a name
      String answer = 
	JOptionPane.showInputDialog("Please enter a name for the " + label);
      if (answer == null) return CANCEL; //cancel ****check what happens!
      else { //got a new name
	try {
	  //object not in domain, so update it (in domain will update later)
	  if (((getName(object) == null) && (newName == null))
	      || getName(object).equals(newName)) 
	    setName(object, answer);
	  //check whether the new name is ok
	  return handleOverwrite(object, answer); 
	}
	catch (Exception e) {
	  Debug.noteException(e);
	  return CANCEL;
	}
      }
    }
    //else (above always returns)
    Object[] optionsA = {"Replace", "Change Name", "Replace All", "Cancel"};
    Object[] optionsB = {"Replace", "Change Name", "Cancel"};
    List options = null;
    Object oldObject = getNameClashObject(object, newName);
    if (oldObject != null) {
      Debug.noteln("ASE: we do have an overwrite situation for", object);
      Debug.noteln(" old object is", oldObject);

      int y = 0; //"Replace" slot; stays if loading and overwrite is set!
      if (isLoading()) {
	//Debug.noteln("ASE: we are loading");
	options = Arrays.asList(optionsA);
	if (!overwrite) {
	  String[] message = 
	    {"There already is a " + label + " called " + 
	     newName + ". Do you want to ",
	     "Replace the existing " + label + "?",
	     "Change the name of the new " + label + "?",
	     "Replace ALL duplicate " + label + "s during this open/insert?",
	     "Cancel the overwrite, i.e. use the existing " 
	     + label + ", forgetting the new one?"};
	  y = JOptionPane.showOptionDialog(null, message, 
					   "WARNING: Name clash",
					   JOptionPane.DEFAULT_OPTION, 
					   JOptionPane.WARNING_MESSAGE,
					   null, optionsA, optionsA[0]);
	}
      }
      else { //not loading, so ask for each time
	//Debug.noteln("ASetEd: handling overwrite");
        //(new Throwable()).printStackTrace();

	String[] message = {"There already is a " + label + " called " + 
			    newName + ". Do you want to ",
			    "Replace the existing " + label + "?",
			    "Change the name of the new " + label + "?",
			    "Cancel the overwrite, i.e. use the existing " 
			    + label + ", forgetting the new one?"};
	y = JOptionPane.showOptionDialog(null, message, 
					 "WARNING: Name clash",
					 JOptionPane.DEFAULT_OPTION, 
					 JOptionPane.WARNING_MESSAGE,
					 null, optionsB, optionsB[0]);
	options = Arrays.asList(optionsB);
      }
      //Debug.noteln("SetEd: y = .", y);
      if ((y == JOptionPane.CLOSED_OPTION) || "Cancel".equals(options.get(y)))
	{
	  //Debug.noteln("SetEd: Overwrite got cancelled");
	  return CANCEL;
	}
      if ("Replace All".equals(options.get(y)) || 
	  "Replace".equals(options.get(y))) {
	if ("Replace All".equals(options.get(y))) {
	  //Debug.noteln("SetEd: Overwrite setting replace all");
	  overwrite = true;
	}
	//remove the old one and do original plan with the new one
	//Debug.noteln("SetEd: Overwrite replacing");
	//removeConstructQuiet(oldObject); //remove then add
	removeConstruct(oldObject); //remove then add
	return OVERWRITE;
      }
      else { //new name
	//change the name then do original plan with the new one
	//Debug.noteln("SetEd: Overwrite changing name");
	String answer = 
	  JOptionPane.showInputDialog("Please enter the new name");
	if (answer == null) return CANCEL; 
	try {
	  setName(object, answer);
	  return NEWNAME;
	}
	catch (Exception e2) {
	  Debug.noteException(e2);
	  return CANCEL;
	}
      }
    }
    
    return ADD;
  }

  /**
   * The two given named objects clash if changeO changes its name to
   * newName. Offer choices: 
   * - change oldO name: legal-set changeO temporary name, 
   *   get oldO name from user and set it on oldO, set newName in changeO
   * - change newName: get name from user and setName that on changeO
   * - or forget the new name and keep changeO's old name.
   * @return true if changes are made, false for last option.
   */
  public boolean handleNameClash(UIObject changeO, String newName, Object old){
    Debug.noteln("ASetEd: name clash on", newName);
    Debug.noteln(" Between " + changeO.toString() + " and", old);
    //(new Throwable()).printStackTrace();
    Object[] options = {"Change this name", "Change other name", "Cancel"};
    String[] message = 
     {"There already is a " + label + " called " +newName+ ". Do you want to ",
      "Change the name of this " + label + "?",
      "Change the name of the other " + label + "?",
      "Cancel the name change and stay with the old name " 
      + changeO.getName()}; 
    int y = JOptionPane.showOptionDialog(null, message, 
					      "WARNING: Name clash",
					      JOptionPane.DEFAULT_OPTION, 
					      JOptionPane.WARNING_MESSAGE,
					      null, options, options[0]);
    if ((y == JOptionPane.CLOSED_OPTION) || (y == 2)) {
      //Debug.noteln("SetEd: name change got cancelled");
      return false;
    }
    else {//get a new name
      String replaceName = 
	JOptionPane.showInputDialog("Please enter the new name");
      if (replaceName == null) return false; //user changed mind
      //now we have a new name
      if (y == 0) //change new object and try again
	changeO.setName(replaceName);
      else {
	String oldName = changeO.getName();
	setLegalName(changeO, newName+"temporaryNameXX");
	UIObject oldO;
	if (isOwnObject(old)) oldO = (UIObject)old;
	else oldO = (UIObject)getOwnObject(old);
	oldO.setName(replaceName);
	changeO.setName(newName); //in case user used newName again!
	//see whether all worked ok. If not, put things back and forget
	String otherName = oldO.getName();
	if ((otherName == null) && (newName == null)) return true;
	if ((otherName == null) ||
	    !otherName.equals(replaceName)) {
	  setLegalName(changeO, oldName);
	  setLegalName(oldO, newName);   
	  return false;
	}
      }
      return true;
    }
  }

  public boolean hasChangedFromOriginal(Object o) {
    Object original = getOriginal(o);
    return !sameConstruct(o, original);
  }
  
  public void updateConstruct(Object object) {
    if (!isEmpty(object)) {
      //Debug.noteln("SetEd: updateConstruct");
      int toAdd = handleOverwrite(object);
      if ((toAdd == NEWNAME) || (toAdd == OVERWRITE)) { //start again
	updateConstruct(object);
	return;
      }
      else if (toAdd == CANCEL) return;
    }
    //if it has been added, ignore edits, else remember them
    if (hasOriginal(object)) {
      if (!added.contains(object)) edited.add(object);
      //Debug.noteln("SetEd: Edited construct", getName(object));
      //back to same as original, forget changes!
      if (!hasChangedFromOriginal(object)) edited.remove(object);
      fireConstructEdited(object);
    }
    else addConstruct(object);
  }

  public void removeConstruct(Object object) {
    removeConstructQuiet(object);
    if (!hasOriginal(object) && isEmpty(object)) return;
    else fireConstructRemoved(object);
  }
  private void removeConstructQuiet(Object object) {
    Object ownObject;
    if (!isOwnObject(object)) { //request to remove base object
      ownObject = findOwnObject(object);
      if (ownObject == null) {//removing original with no own
	removed.add(object);
	return;
      } //else remove own version as below
    }
    else ownObject = object; //request to remove own object
    

    //work on own object from here
    //if it has been added, forget this and be done (do fire removed)
    if (added.contains(ownObject)) added.remove(ownObject);
    //if it is empty and has no original - just forget it.(don't fire removed)
    else if (!hasOriginal(ownObject) && isEmpty(ownObject)) return;
    //else forget all edits (if any) and remember that it is removed
    else { //has original or is not empty
      edited.remove(ownObject);
      removed.add(ownObject);
    }
    if (originals.containsValue(ownObject)) {
      originals.remove(getOriginal(ownObject));
    }
  }


  /** 
   * Finds out whether any constructs in the set have changed.
   * The set has not changed if there are no added/edited/removed notes
   */
  public boolean hasChangedConstructs() {
    if (added.isEmpty() && removed.isEmpty() && edited.isEmpty())
      return false;
    else return true;				   
  }

  /** 
   * Collects descriptions of any changes in constructs in the set.
   */
  public List collectConstructChanges() {
    List changes = new ArrayList();
    if (hasChangedConstructs()) {
      changes.add("Changes in " + label);
      if (!added.isEmpty()) changes.add(" Added: " + UIUtil.show(added));
      if (!removed.isEmpty()) changes.add(" Removed: " + UIUtil.show(removed));
      if (!edited.isEmpty()) changes.add(" Edited: " + UIUtil.show(edited));
    }
    return changes;
  }

    /*
  public void saveToDomain(Domain domain) {
    try {
      for (Iterator i=removed.iterator(); i.hasNext();) {
	Object o = i.next();
	//Debug.noteln("SetEd: removing object", o.toString());
	if (o instanceof UIObject)
	  ((UIObject)o).removeFromDomain(domain);
	else removeFromDomain(domain, o);
      }
      for (Iterator i=edited.iterator(); i.hasNext();) {
	Object o = i.next();
	//Debug.noteln("UID: updating object", o.toString());
	if (o instanceof UIObject)
	  ((UIObject)o).updateInDomain(domain);
      }
      for (Iterator i=added.iterator(); i.hasNext();) {
	Object o = i.next();
	//Debug.noteln("UID: adding object", o.toString());
	if ((o instanceof UIObject) && (!((UIObject)o).isEmpty()))
	  ((UIObject)o).addToDomain(domain);
      }
      //Debug.noteln("UID - finished saveToDomain(domain)");
    }
    catch (DomainReferenceException re) {
      Debug.noteln(Debug.foldException(re));
      JOptionPane.showMessageDialog(null, Debug.foldException(re));
      //Debug.describeException(re);
    }

  }
  

  public void removeFromDomain(Domain domain, Object object) {
    if (object instanceof Refinement) 
      domain.deleteNamedRefinement(((Refinement)object).getName());
    else if (object instanceof ObjectClass) {
      domain.deleteNamedObjectClass(((ObjectClass)object).getName());
    }
  }
    */

  //--------------------------Getting information-----------------------------


  public Object newOwnObject(Object object) {
    if (object == null) return null;
    Object ownObject = null;
    ownObject = makeOwnObject(object);
    if (ownObject != null)
      addedObject(ownObject, object);
    return ownObject;
  }


  public Object getOwnObject(Object object) {
    //find out whether there is an own version for this object
    Object oo = findOwnObject(object);
    if (oo == null) return newOwnObject(object);
    else return oo;
  }

  /**
   * Finds an ownObject from the given Object by looking it up in the
   * "originals" list.
   * @return - the ownObject found or null
   */
  public Object findOwnObject(Object object) {
    if (object == null) return null;
    if (isOwnObject(object)) return object; //already own
    String name = "";
    if (object instanceof String)  //prepare for name/symbol lookup
      name = (String)object;
    else if (object instanceof Symbol)
      name = ((Symbol)object).toString();
    
    if ((name != null) && !name.equals("")) {//look for object by name
      //Debug.noteln("SetEd: Looking for named object",name);
      Object found = findOwnObject(name, added);
      if (found == null) 
	found = findOwnObject(name, edited);
      if (found == null)
	found = findOwnObject(name, originals.values());
      return found;
    }
    else { //object given, not name
      //Debug.noteln("SetEd: originals length:-------",originals.size());
      //Debug.noteln("SetEd: finding UIobject for ", getName(object));
      Object o = originals.get(object);
      if (o == null) //not found, search by name
	o = findOwnObject(getName(object));
      if (o != null) {
	//Debug.note("SetEd: found object for " + getName(object));
	//Debug.noteln(" -- object is", o);
      }
      return o;
    }
  }

  private Object findOwnObject(String name, Collection collection) {
    for (Iterator i = collection.iterator(); i.hasNext(); ) {
      try {
	Object o = i.next();
	//Debug.noteln(" checking", getName(o));
	if (getName(o).equals(name)) return o;
      }
      catch (Exception e) {} //dont worry if the object is not named
    }
    return null;
  }

  public void addedObject(Object oo, Object original) {
    if (original == null) {
      //Debug.noteln("ASetEd: got null original for", oo);
      addConstruct(oo);
    }
    else {
      //Debug.note("SetEd: originals object " + getName(original));
      //Debug.noteln(" - with object", getName(oo));
      originals.put(original, oo);
      //Debug.noteln("SetEd:             originals length:", originals.size());
      //fireConstructAdded(oo);
    }
  }


  public LinkedList getAllConstructs() {
    LinkedList list = new LinkedList(added);
    //start with added 
    // + original, filter with own ones (use own ones in preference)
    // - removed (removed originals; removed added are already dropped)
    //get added

    //get originals
    LinkedList originalList = getOriginals();
    //Debug.noteln("SetEd originals: ",IVUtil.show(originalList));
    //use ui ones of originals and take out removed ones
    for (Iterator i = originalList.iterator(); i.hasNext(); ) {
      Object r = i.next();
      Object no = findOwnObject(r);
      if (no != null) {
        if (!removed.contains(no) && !list.contains(no)) 
         //ui version exists and is not removed so forget the domain version
	 list.add(0, no);
      }
      else if (!removed.contains(r) && !list.contains(r)) list.add(r);
    }
    //Debug.noteln("AllConstructs are:", IVUtil.show(list));
    return list;
  }

  //--------------------------Notifying listeners-----------------------------


  public void fireConstructAdded(Object object) {
    //Debug.noteln("SetEd firing construct added", object.print());
    manager.fireConstructAdded(this, object);
  }
  public void fireConstructEdited(Object object) {
    //Debug.noteln("SetEd firing construct edited");
    manager.fireConstructEdited(this, object);
  }
  public void fireConstructRemoved(Object object) {
    //Debug.noteln("SetEd firing construct removed");
    manager.fireConstructRemoved(this, object);
  }



  //-------------------------- things---------------------------------

  public String print() { 	
    List cs = getAllConstructs();
    return "[" + cs.size() + " " + label + "s]";
  }
  public String printSet() { 	
    List cs = getAllConstructs();
    String p = "[";
    for (Iterator i = cs.iterator(); i.hasNext(); ) {
      Object o = i.next();
      p = p + getName(o) + ", ";
    }
    return p;
  }




}


// Issues:
// * tidy up load/insert domain stuff 
// * 
// * 
