/****************************************************************************
 * An editor/viewer panel for AROs
 *
 * @author Jussi Stader
 * @version 2.3+
 * Updated: Thu Oct 26 10:24:29 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iview;

import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.BorderLayout;
import java.awt.Frame;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.*;
import javax.swing.table.*;
import javax.swing.tree.*;

import ix.*;
import ix.util.Debug;
import ix.util.Util;
import ix.util.lisp.*;
import ix.icore.*;
import ix.icore.info.*;
import ix.icore.domain.*;
import ix.icore.domain.event.*;
import ix.iface.ui.*;
import ix.iface.ui.event.*;
import ix.iface.ui.table.*;
import ix.iface.ui.tree.*;
import ix.iface.ui.util.*;
import ix.iview.*;
import ix.iview.util.*;
import ix.iview.domain.*;
import ix.iview.table.*;
import ix.iview.tree.*;
import ix.iview.domain.event.*;

/****************************************************************************
 * An editor/viewer panel for AROs, i.e. Activity Relatable Objects.
 *
 * Waiting to be implemented. <p>
 *
 * The panel has editable fields for the object's name, type(s), 
 * sub-types, and attributes. Other relationships between objects. <p>
 *
 * Example code for using the AROEditorPanel:
 *<PRE><code>
 *    ...
 *    ...
 *</code></PRE>
 *****************************************************************************
 */
public class AROEditorPanel extends AConstructFramePanel
implements ActionListener, TreeSelectionListener, UIDomainListener, 
	   IDESymbols {

  /*
  private UIObjectClass uiConstruct = 
    new UIObjectClass(getUIDomain(), "");  
  */

  private JSplitPane split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
  private JScrollPane scrollTree = new JScrollPane();
  //private IXTreeTable classTree;
  //  private ClassTreeTableModel classTreeModel;

  private EditableTree classTree;
  private ObjectClassTreeModel classTreeModel;

  /**
   * Make the editor panel and set up the class tree
   */    
  public AROEditorPanel(DomainEditorFrame theParent){
    super(theParent);

    //populateARO(); //this is done as part of setDomain()
    setupClassTree();
    refresh();
  }

    
  protected void populateARO() {
    //Debug.noteln("AROed: populating AROs");
    UIDomain typeManager = getUIDomain();
    List ocs = typeManager.getAllObjectClasses();
    //Debug.noteln("ARO: set is", UIUtil.show(ocs));
    if ((ocs != null) && !ocs.isEmpty()) { //we do have a model
      if (classTreeModel != null) 
	classTreeModel.reloadData();
    }
    /*
    else { //ocs empty! make root
      if (classTreeModel != null) {
	Debug.noteln("AROEd: ocs are",typeManager.getAllObjectClasses());
	UIObjectClass top = newUIObjectClass(IXTrees.ROOTNAME);
	Debug.noteln("  ocs now",typeManager.getAllObjectClasses());
	classTreeModel.setUserRoot(top);
      }
    }
    */
    //Debug.noteln("ARO: UIDomain now", typeManager.printSets());
    //refresh();
  }

  public UIObjectClass newUIObjectClass(String name) {
    //Debug.noteln("ARO: making new object class:", name);
    UIObjectClass oc = new UIObjectClass(getUIDomain(), name);
    getUIDomain().addConstruct(oc);
    return oc;
  }
  public UIObjectClass newUIObjectClass(String name, UIObjectClass parent) {
    UIObjectClass oc = new UIObjectClass(getUIDomain(), name);
    ListOfSymbol los = new LinkedListOfSymbol();
    los.add(Symbol.intern(parent.getName()));
    oc.setSuperClassNames(los);
    getUIDomain().addConstruct(oc);
    return oc;
  }

  protected JPanel setupMainPanel() {
    //Debug.noteln("ARO: setting up main panel");

    IFormModel model = new GenericIFormModel(UIObjectClass.class) {
	//this should only happen when the object system is empty
	public void setValue(String field, Object value) {
	  if (baseObject == null) {
	    //makes object, saves value and notes in panel and model
	    AROEditorPanel.this.beforeSwitching(); 
	    AROEditorPanel.this.refresh();
	  }
	  else super.setValue(field, value);
	}
	
      };
    formPanel = new AROFormPanel(this, model);

    MouseListener ml = new AROMouseAdapter();
    formPanel.addFormMouseListener(ml);
 
    return formPanel;
  }

  protected void setupClassTree() {
    split.setRightComponent(scrollPane);
    split.setLeftComponent(scrollTree);
    add(split, BorderLayout.CENTER);
    JPanel treePanel = new JPanel();
    scrollTree.add(treePanel);

    ObjectClassTreeNode dummyRoot = 
      new ObjectClassTreeNode(new UIObjectClass(getUIDomain(), "Dummy Root"));
    //classTreeModel = new ObjectClassTreeModel(getUIDomain(), dummyRoot);
    classTreeModel = new ObjectClassTreeModel(getUIDomain(), null);
    classTreeModel.reloadData();
    //classTreeModel.addTreeModelListener(this);

    classTree = new EditableTree(classTreeModel);
    classTree.addTreeSelectionListener(this);

    scrollTree.getViewport().add(classTree);
    split.setOneTouchExpandable(true);
    //split.setDividerLocation(0.33);
    //split.setResizeWeight(0.33);
    //split.setDividerSize(5);
    split.setDividerLocation(170);
    Object root = classTreeModel.getUserRoot();
    if (root != null) setUIConstruct((UIObjectClass)root);
  }
  public void refresh() {
    resetTree(); //was updateTree();
    if (classTree != null) {
      classTree.treeDidChange();
      //updateTreeSelection();
    }
  }

  public void updateTree() {
    if (classTreeModel != null)
      classTreeModel.reload(); 
    if (classTree != null)
      updateTreeSelection();
  }
  public void resetTree() {
    //Debug.noteln("AROed: resetting tree. Current class",getUIConstruct());
    if (classTreeModel != null) {
      classTreeModel.reloadData();
      if (!getUIDomain().isCurrentObject(getUIConstruct())) {
	UIObject root = (UIObject)classTreeModel.getUserRoot();
	if (!getUIDomain().isCurrentObject(root)) root = null;
	setUIConstruct(root);
      }
    }
    else Debug.noteln("AROed: no class tree model");
    if (classTree != null)
      updateTreeSelection();
  }
  public void updateTreeSelection() {
    //Debug.noteln("AEP: updating tree selection to", getUIConstruct());
    classTree.showUserNode(getUIConstruct());
  }

  /** 
   * Make a new construct of the panel's type.
   */
  public UIObject makeNewConstruct() {
    return (UIObject) new UIObjectClass(getUIDomain(), "");
  }
  /** 
   * Save a construct of the panel's type into the draft domain.
   */
  public void saveConstruct() {
    noteConstruct();
    setUIConstructPanel(getUIConstruct());
  }
  /** 
   * Note the contents of the panel into the current object.
   */
  public UIObject noteConstruct() {
    //make sure there is a current construct
    UIObject uiConstruct = getUIConstruct();
    if (uiConstruct == null)
      uiConstruct = (UIObjectClass)makeNewConstruct();
    formPanel.saveToObject(uiConstruct);

    uiConstruct.noteChange();
    //make sure the refinement is noted into the right domain
    if (!getUIDomain().equals(uiConstruct.getUIDomain())) 
      uiConstruct.moveToUIDomain(getUIDomain());
    getUIDomain().updateConstruct(uiConstruct);
    //Debug.noteln("AEP: Noted construct");
    //Debug.noteln(" ",uiConstruct.print());
    //setUIConstruct(uiConstruct);
    return uiConstruct;
  }
  /** 
   * Get a named construct of the panel's type from the domain
   */
  public Named getNamedConstruct(String name) {
    if (getUIDomain() == null) return null;
    Object c = getUIDomain().getNamedObjectClass(name);
    if ((c != null) && (c instanceof UIObject))
      return (Named)c;
    else return getUIDomain().newUIObject((IXObject)c);
  }

  /** 
   *  Check a construct of the panel's type for errors and inconsistencies
   */
  public boolean checkConstruct() {
    UIUtil.notImplemented(this,"Checking Object Classes");
    return true;
  }
  /** 
   * Get a full list of the constructs of the panel's type in the domain.
   */
  public List getAllConstructs() {
    if (getUIDomain() == null) return null;
    List constructs = getUIDomain().getAllObjectClasses();
    return new LinkedList(constructs);
  }

  

  public void clear() {
    resetTree();
  }

  public void clearUIConstruct() {}

  protected UIObject cloneConstruct(UIObject uiConstruct) {
    return ((UIObjectClass)uiConstruct).cloneThis();
  }

  /** ***do we really need this?
   * Sets the given UI construct in this editor and in the current panel
   * Does not call beforeSwitching.
   *
   * @param object the UI object to be viewed/edited
   */    
  public void setUIConstructPanel(UIObject object) {
    UIObject old = getUIConstruct();
    //uiConstruct = (UIObjectClass)object;
    //set the object in the actual panel
    ((ConstructEditing)formPanel).setUIConstruct(object);
    //***updateTreeSelection();
    //***fireCurrentClassChanged(old, uiConstruct);
  }
  /**
   * Sets the given UI construct to be viewed/edited. 
   * Also sets the uiConstruct in the current editorPanel
   *
   * @param object the UI object to be viewed/edited
   */    
  public void setUIConstruct(UIObject object) { //***why diff from super???
    //Debug.noteln("AROEd: setting construct", object);
    //noteConstruct();
    UIObject old = getUIConstruct();
    //***this loops if equal is not caught!
    if ((object != null) && !object.equals(old) && beforeSwitching()) {
      setUIConstructPanel(object);
      //fireConstructChanged(old, object);
      afterSwitching();
    }
    else super.setUIConstruct(object);
  }

  public boolean beforeClosing() {
    //if (classEditor != null) classEditor.setVisible(false);
    if (propEditor != null) propEditor.setVisible(false);
    if (searcher != null) searcher.close();
    return beforeSwitching();
  }
  
     
  //---------------------UIDomainListener things------------------------------

  //****************Note: tree model also listens, so no need to tell it

  /** 
   * Called when a UI construct is added. 
   * Use getSource() to get at the domain (UI/IX) in which the event occurred.
   * Use getObject() to get at the object (UI/IX) that has been added.
   */
  public void constructAdded(UIDomainEvent event) {
    //updateTree();
  }
  public void constructAdded(DomainEvent event){
    //updateTree();
  }
  /** 
   * Called when a UI construct is edited. 
   * Use getSource() to get at the UIDomain in which the event occurred.
   * Use getObject() to get at the UIObject that has been edited.
   */
  public void constructEdited(UIDomainEvent event){
    //updateTree();
  }
  /** 
   * Called when a UI construct is removed. 
   * Use getSource() to get at the UIDomain in which the event occurred.
   * Use getObject() to get at the UIObject that has been removed.
   */
  public void constructRemoved(UIDomainEvent event) {   
    if (event.getObject().equals(getUIConstruct())) 
      deleteConstruct();
    //updateTree();
  }
  public void domainCleared(UIDomainEvent event) {
    //Debug.noteln("AEP: got cleared domain");
    clear();
    //populateARO();
    //resetTree();
  }
  public void domainEdited(UIDomainEvent se) {
      //clear();
    populateARO();
  }
  public void domainSet(UIDomainEvent event) {
    //Debug.noteln("AEP: got domain set event");
      //clear();
    populateARO();
    //resetTree();
  }

  //-------------------- User services -----------------------

  protected UIObjectClass getClassFromUser(UIObjectClass parentClass) {
    return getClassFromUser("Select an ObjectClass", parentClass);
  }
  protected UIObjectClass getClassFromUser(String message, 
					   UIObjectClass parentClass) {
    List classes = getUIDomain().getAllObjectClasses();
    classes.add(0, new String("New class"));
    return getClassFromUser(classes, message, parentClass);
  }
  protected UIObjectClass getClassFromUser(List offers, 
					   UIObjectClass parentClass) {
    return getClassFromUser(offers, "Select an ObjectClass", parentClass);
  }

  protected UIObjectClass getClassFromUser(List offers, String message, 
					   UIObjectClass parentClass) {
    Object c = 
      JOptionPane.showInputDialog((Component)parent, message,
				  "Input", JOptionPane.QUESTION_MESSAGE, null, 
				  offers.toArray(), null);
    return processClassFromUser(c, parentClass);

  }

  private UIObjectClass processClassFromUser(Object c, UIObjectClass parentC) {
    if (c == null) return null;
    else if (c instanceof UIObjectClass) return (UIObjectClass)c;
    else if (c.equals("New class")) {
      return userNewClass(parentC);
    }
    else return (UIObjectClass)getUIDomain().getNamedObjectClass(c.toString());
  }

  private UIObjectClass userNewClass(UIObjectClass parentClass) {
    String answer = 
      JOptionPane.showInputDialog("Please enter the name of the new class");
    if (answer == null) return null;
    UIDomain uiDomain = getUIDomain();
    UIObjectClass newClass = new UIObjectClass(uiDomain, answer);
    //check that the name is ok straight away!
    boolean added = uiDomain.addConstruct(newClass);
    if (added) { //has been added ok (no cancelled name clashes)
      if (parentClass == null)
	parentClass = 
	  getClassFromUser("Please enter the parent of the new class", 
			   parentClass);
      if (parentClass == null) { //user changed their mind
	uiDomain.removeConstruct(newClass);
	return null;
      }
      else {
	//Debug.noteln("AROEd***: new class parent:", parent.print());
	newClass.addDataChangeListener(classTreeModel); //catch next change!
	newClass.addParent(parentClass);
	//Debug.noteln("AROEd***: new class:", newClass.print());
	return newClass;
      }
    }
    return null;
  }


  public void userNewChild() {
    //Debug.noteln("AROEd: new child");
    UIObjectClass current = (UIObjectClass)getUIConstruct();
    if (current != null) {
      //Debug.noteln("AROEd: making new child for", current);
      UIObjectClass newKid = (UIObjectClass)makeNewConstruct();
      newKid.addParent(current);
      if (getUIDomain().addConstruct(newKid)) { //this fires construct added
        current.addChild(newKid);
	//Debug.noteln("AROEd: newCh", classTreeModel.printSubTree());
	setUIConstruct(newKid);
      }
      //else  Debug.noteln(" not added to domain");
    }
    //else Debug.noteln(" no current class");
  }
  public void userAddChild() {
    UIObjectClass current = (UIObjectClass)getUIConstruct();
    if (current != null) {
      //Debug.noteln("AROEd: adding child for", current);
      UIObjectClass oc = getClassFromUser(current);
      if (oc != null)  {//not cancelled
	String result = current.reportLegalChild(oc);
	if (result == "") {
	  current.addChild(oc);
	  oc.addParent(current);
	  //Debug.noteln("AROEd: addCh", classTreeModel.printSubTree());
	}
	else JOptionPane.showMessageDialog(this, result);
      }
    }
  }
  public void userDeleteChild() {
    IXEditorPanel nodesPanel = formPanel.getFieldBit("subClassNames");
    JList list = (JList) nodesPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a sub class to delete first.";
      JOptionPane.showMessageDialog(this, message);} 
    else {
      //Debug.noteln("AROEd: user deleting sub class");
      Object[] values = list.getSelectedValues();
      for (int i = 0; i < values.length; i++){
	Object child = values[i];
	String cs = child.toString();
	UIObjectClass uiChild = 
	    (UIObjectClass)getUIDomain().getNamedObjectClass(cs);
	if (uiChild != null)
	  uiChild.removeParent((UIObjectClass)getUIConstruct());
        ((UIObjectClass)getUIConstruct()).removeChild(child.toString());
      }
      list.clearSelection();
    }
  }
  public void userEditChild() {
    IXEditorPanel nodesPanel = formPanel.getFieldBit("subClassNames");
    JList list = (JList) nodesPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a sub class to edit first.";
      JOptionPane.showMessageDialog(this, message);} 
    else {
      //Debug.noteln("AROEd: user editing sub class");
      Object[] values = list.getSelectedValues();
      Object child = values[0]; //just use the first selected one
      UIObjectClass childOC = 
	(UIObjectClass)getUIDomain().getNamedObjectClass(child.toString());
      if (childOC != null) setUIConstruct(childOC);
      else Debug.noteln("Cannot find object class", child);
    }
  }

  public void userAddParent() {
    UIObjectClass current = (UIObjectClass)getUIConstruct();
    if (current != null) {
      //Debug.noteln("AROEd: adding parent for", current);
      UIObjectClass oc = getClassFromUser(null);
      if (oc != null)  {//not cancelled
	String result = current.reportLegalParent(oc);
	if (result == "") current.addParent(oc);
	else JOptionPane.showMessageDialog(this, result);
      }
    }
  }
  public void userDeleteParent() {
    IXEditorPanel nodesPanel = formPanel.getFieldBit("superClassNames");
    JList list = (JList) nodesPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a super class to delete first.";
      JOptionPane.showMessageDialog(this, message);} 
    else {
      //Debug.noteln("AROEd: user deleting super class");
      Object[] values = list.getSelectedValues();
      for (int i = 0; i < values.length; i++){
	Object parent = values[i];
        ((UIObjectClass)getUIConstruct()).removeParent(parent.toString());
      }
      list.clearSelection();
    }
  }
  public void userEditParent() {
    IXEditorPanel nodesPanel = formPanel.getFieldBit("superClassNames");
    JList list = (JList) nodesPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a super class to edit first.";
      JOptionPane.showMessageDialog(this, message);} 
    else {
      //Debug.noteln("AROEd: user editing super class");
      Object[] values = list.getSelectedValues();
      Object parent = values[0]; //just use the first selected one
      UIObjectClass parentOC = 
	(UIObjectClass)getUIDomain().getNamedObjectClass(parent.toString());
      if (parentOC != null) setUIConstruct(parentOC);
      else Debug.noteln("Cannot find object class", parent);
    }
  }

  /*
  ObjectClassEditor classEditor;
  public void ensureClassEditor(){
    if (classEditor == null){
      classEditor = new ObjectClassEditor();
      classEditor.setVisible(false);
    } 
  }
  */

  public PanelSearcher searcher;
  public void ensureSearcher() {
    if (searcher == null) {
      searcher = new PanelSearcher(this); //, new SimpleTextRenderer(false)
      searcher.setVisible(false);
    }
  }
  public void userSearch() {
    ensureSearcher();
    searcher.setVisible(true);
  }
  private void searchOld() {
    String pat = JOptionPane.showInputDialog(this, 
					     "Search string (regex)");
    if ((pat == null) || pat.equals("")) Debug.noteln("No search string");
    else {
      Set classes = getUIDomain().getMatchingObjectClasses(pat);
      Debug.noteln("Matching classes:", classes);
    }
  }


  PropertyEditor propEditor;
  public void ensurePropEditor(){
    if (propEditor == null){
      propEditor = new PropertyEditor(parent);
      propEditor.setVisible(false);
    } 
  }
  private Point mousePoint = new Point(100, 100);
  private Point getEditorLocation() {
    Point panelPoint = getLocation();
    Point framePoint = parent.getLocation();
    panelPoint.translate((int)framePoint.getX(), (int)framePoint.getY());
    //Debug.noteln("AEP: mouse point", mousePoint.toString());
    //Debug.noteln("AEP: frame point", framePoint.toString());
    Point point = new Point(panelPoint);
    point.translate((int)mousePoint.getX(), (int)mousePoint.getY());
    return point;
  }

  public void userAddProperty() {
    ensurePropEditor();
    propEditor.start(getUIConstruct(), getEditorLocation());
  }
  public void userDeleteProperty() {
    IXEditorPanel nodesPanel = formPanel.getFieldBit("objectProperties");
    JList list = (JList) nodesPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a property to delete first.";
      JOptionPane.showMessageDialog(this,message);} 
    else {
      Debug.noteln("AROEd: user deleting property");
      Object[] values = list.getSelectedValues();
      for (int i = 0; i < values.length; i++){
	Object prop = values[i];
        ((UIObjectClass)getUIConstruct()).removeProperty(prop);
      }
      list.clearSelection();
    }
  }

  public void userEditProperty() {
    //UIUtil.notImplemented(null,"ARO Editor: Editing properties");   
    
    IXEditorPanel propsPanel = formPanel.getFieldBit("objectProperties");
    JList list = (JList) propsPanel.getItemComponent();
    if (list.isSelectionEmpty()) {
      String message = "Please select a property to edit first.";
      JOptionPane.showMessageDialog(this,message);} 
    else {
      Object[] values = list.getSelectedValues();
      ensurePropEditor();
      propEditor.start(getUIConstruct(), values[0], getEditorLocation());
    }    
  }


  //---------------------Services------------------------------
  /**
   * Deletes the construct, then sets its parent.
   * overwrites abstract class. Deleting the current class results not
   * in a new construct, but in selecting a parent.
   * Does not allow deletion of root node.
   */
  public void deleteConstruct() {
    deleteConstruct((IXNode)getUIConstruct());
  }
  private void deleteConstruct(IXNode construct) {
    if (construct == null) return;
    else if (((construct.getParents() == null) || 
	      (construct.getParents().isEmpty()))
	     && construct.equals(classTreeModel.getUserRoot())) {
	JOptionPane.showMessageDialog(parent, "Cannot delete root node." 
				      + " Nothing done.");
	return;
    }
    else {
      //find a suitable node to use as current after deleting
      IXTreeNode currentSelection = getCurrentTreeNode();
      IXTreeNode newSelection = null;
      if ((currentSelection != null) && 
	  !currentSelection.equals(classTreeModel.getRoot())) {
	if (construct.equals(currentSelection.getUserObject())) //to be deleted
	  newSelection = (IXTreeNode)currentSelection.getParent();
	else newSelection = currentSelection;
      }
      if (newSelection == null) 
	newSelection = (IXTreeNode)classTreeModel.getRoot();
      Object newCurrent = newSelection.getUserObject();
      classTree.showUserNode(newCurrent, construct);
      
      getUIDomain().removeConstruct((Named)construct);
      setUIConstruct((UIObject)newCurrent);
    }

  }
  protected IXTreeNode getCurrentTreeNode() {
    if (!classTree.isSelectionEmpty()) 
      return (IXTreeNode)classTree.getSelectionPath().getLastPathComponent();
    else return null;
  }

  private void generateObjectSystem() {
    ObjectClassGenerator ocg =  new ObjectClassGenerator();
    ocg.generateObjectClasses(getUIDomain());
  }

  /**
   * Recognises panel specific events and processes them.
   * This is a separate method because it must let other event handlers know
   * whether it recognised the event.
   *
   * @param event the event to be processed.
   * @return true if the event was recognised, false if not.
   */
  public boolean frameActionPerformed(ActionEvent event){
    //deal with commands
    String command = event.getActionCommand();
    //Debug.noteln("ARO-P: action command is", command);
    if ((command == "newEdit") || (command == "newChild"))
      userNewChild();
    else if (command == "editConstruct") {
      //get the construct name from the item text and show it
      JMenuItem item = (JMenuItem)event.getSource();
      String name = item.getText().trim();
      setUIConstruct((UIObject)getNamedConstruct(name));  
    }
    else if ((command == "saveConstruct") || (command == "modify")) 
      saveConstruct();
    else if ((command == "checkConstruct") || (command == "check")) 
      checkConstruct();
    else if (command == "generateOS") generateObjectSystem();
    else if (command == "search") userSearch();
    else if (command == "cut") deleteConstruct();
    else if (command == "copy") copyConstruct();
    else if (command == "revert") revertConstruct();
    else if (command == "revertO") revertOConstruct();
    else return false;
    return true;
  }

  /**
   * Wakes up the ActionListener with a user action.
   * This is called when a KeyStroke happens in which the ActionListener
   * registered its interest.
   */
  public void actionPerformed(ActionEvent ae){
    String command = ae.getActionCommand();
    Debug.noteln("ARO Editor: Got action command", command);
  }


  // ------------- TreeModelListener things --------------
  //all these monitor the model and make sure updates go through to the domain
  public void treeNodesChanged(TreeModelEvent e) {
  }
  public void treeNodesInserted(TreeModelEvent e) {
  }
  public void treeNodesRemoved(TreeModelEvent e) {
  }
  
  public void treeStructureChanged(TreeModelEvent e) {
    
  }
 

  // ------------- TreeSelectionListener things --------------
  public void valueChanged(TreeSelectionEvent e) {
    if (!classTree.isSelectionEmpty()) {
      TreePath path = e.getPath();
      Object o = path.getLastPathComponent();
      if (o instanceof IXTreeNode) {
	Object userOb = ((IXTreeNode)o).getUserObject();
	setUIConstruct((UIObject)userOb);
      }
      else Debug.noteln("AROEd: cannot process selection", o);
    }
  }

  //--------------------------------------Panel Class-----------------------

  public class AROFormPanel extends IFormPanel 
    implements ConstructEditing { //, UndoEditing

    private UIObject uiConstruct;
    
    private AConstructFramePanel parent;

    /** Listeners interested in the current action */
    HashSet actionListeners = new HashSet();

    public AROFormPanel(AConstructFramePanel parent, IFormModel model) {
      super(model);
      this.parent = parent;
      IXRenderer r = new IXLabelRenderer(false) {
	  public void setInfo(Object o) {
	    ObjectProperty prop = (ObjectProperty)o;
	    if (prop == null) setText("");
	    else {
	      String name = "";
	      String synt = "";
	      if (prop.getName() != null) name = prop.getName().toString();
	      if (prop.getSyntax() != null) synt = prop.getSyntax().toString();
	      setText(name + " [" + synt + "]");
	    }
	  }
	};
      setRenderer("objectProperties", r);
    }

    public UIDomain getUIDomain() {
      return parent.getUIDomain();
    }
    public void setConstruct(IXObject construct) {
      setUIConstruct(getUIDomain().getUIObject(construct));
    }
    public IXObject getConstruct() {
      if (uiConstruct == null) return null;
      else return uiConstruct.getBaseObject();
    }
    public void setUIConstruct(UIObject construct) {
      //Debug.noteln("AROed: setting new construct", construct);
      if (uiConstruct != null) 
	  uiConstruct.removeDataChangeListener(AROEditorPanel.this);
      setUIConstructOnly(construct);
    }
    public void setUIConstructOnly(UIObject construct) {
      UIObject old = uiConstruct;
      uiConstruct = construct;
      getModel().setObject(construct);
      parent.fireConstructChanged(old, construct, this);
      classTree.showUserNode(uiConstruct, old);
      if (construct != null)
	construct.addDataChangeListener(AROEditorPanel.this);
    }
    public UIObject getUIConstruct() {
      return uiConstruct;
    }
    
    public void loadFromObject() {
      displayModelData();
    }
    
    //public void formModelChanged(java.lang.String change) 



    //-----------------------UndoEditing things----
    // reading/updating the model will have the desired effect
    
    /** Sets the given field to the given value in the editor. */
    public void undoSetValue(String field, Object value) {
      getModel().setValue(field, value);
    }
    
    /** Gets the given field to the given value in the editor. */
    public Object undoGetValue(String field) {
      return getModel().getValue(field);
    }
    
  }

  

 
 

  //************** relay buttons to their services *************************
  class AROMouseAdapter extends java.awt.event.MouseAdapter 
  implements SchemaTerms {
    public void mouseClicked(MouseEvent event){
      Component component = event.getComponent();
      //Debug.noteln("AROEditorPanel: mouse from ", component.getClass());
      //only interested in button clicks
      if (!AbstractButton.class.isInstance(component)) return;
      
      AbstractButton button = (AbstractButton) component;
      //Debug.noteln("ARO-P: Button command " + button.getText());
      IXEditorPanel buttonParent = 
	(IXEditorPanel)button.getParent().getParent();
      //Debug.noteln(" Parent name " + buttonParent.getName());

      if (buttonParent.getName().equals("SuperClassNames")) {
	if (button.getText() == "Add") 
	  AROEditorPanel.this.userAddParent();
	else if (button.getText() == "Delete") 
	  AROEditorPanel.this.userDeleteParent();
	else if (button.getText() == "Edit") 
	  AROEditorPanel.this.userEditParent();
	else 
	  UIUtil.notImplemented(null, button.getText() + " for nodes");
	return;
      }
      else if (buttonParent.getName().equals("SubClassNames")) {
	if (button.getText() == "Add") 
	  AROEditorPanel.this.userAddChild();
	else if (button.getText() == "Delete") 
	  AROEditorPanel.this.userDeleteChild();
	else if (button.getText() == "Edit") 
	  AROEditorPanel.this.userEditChild();
	else 
	  UIUtil.notImplemented(null, button.getText() + " for nodes");
	return;
      }
      else if (buttonParent.getName().equals("ObjectProperties")) {
	if (button.getText() == "Add") 
	  AROEditorPanel.this.userAddProperty();
	else if (button.getText() == "Delete") 
	  AROEditorPanel.this.userDeleteProperty();
	else if (button.getText() == "Edit") 
	  AROEditorPanel.this.userEditProperty();
	else 
	  UIUtil.notImplemented(null,button.getText() + " for issues");
	return;
      }
      else UIUtil.notImplemented(null,"Command " + button.getText());
    }
  }
 

  /* write this for popups
  public void populatePopup(JObjectPopup popup, Object construct) {
    if (construct instanceof UINodeSpec) {
      List exps = getAllExpansions((UINodeSpec)construct);
      String cname = NodeNameRenderer.makeText((UINodeSpec)construct);
      JObjectMenu expansionsMenu = new JObjectMenu("Expansions");
      UIUtil.populateMenu(parent, expansionsMenu, construct, 
			  "Expansions" + " for " + cname, exps);
      popup.add(expansionsMenu);
      popup.setObject(construct);
      popup.setLabel(construct.toString());
    }
  }
  */

  //----------------------------Tree Model---------------------------
  public class ClassTreeTableModel extends AbstractTreeTableModel
  {
    protected UIDomain uiDomain;

    public ClassTreeTableModel(UIDomain uiDomain) { 
      super(false); //not markable
      setColumnNames();
      setDomain(uiDomain);
    }

    public void setDomain(UIDomain domain) {
      uiDomain = domain;
      uiDomain.addUIDomainListener(AROEditorPanel.this);
      List constructs = uiDomain.getAllObjectClasses();

      //constructs = 
      //	(List)IVUtil.sortNamedCollection(new ArrayList(constructs));   
      //Debug.noteln(" after sort:", UIUtil.listToDisplay(constructs));
      //setData(constructs);
      updateData(constructs);
    }

    /**
     * Makes an IXTreeTableNode from the given object.
     * If the given object is not a TreeNode already, uses the domain to get
     * the object's UIObject.
     */
    public IXTreeTableNode makeIXTreeTableNode(Object theNode){
      TreeNode tNode = null;
      if (theNode != null) {
	if (TreeNode.class.isInstance(theNode))
	  tNode = (TreeNode)theNode;
	else if (IXObject.class.isInstance(theNode))
	  tNode = (TreeNode)uiDomain.getUIObject((IXObject)theNode);
      }
      else Debug.noteln("ARO/CTTM: Trying to make tree node from null");

      if (tNode == null) return null;
      else //make a new eNode using the tNode
	return new IXTNTreeTableNode(this, tNode);
    }

    /**
     * Gets the cell value for normal cells (only one - the pattern).
     * The name cell is done in the AbstractTreeTableModel method
     * getTreeValueAt.
     * @return the value as an object, null for illegal columns.
     */
    public Object getCellValueAt(Object o, int columnIndex) {
      return null;
    }

    /**
     * Gets the name string for the given row object.
     */
    public String getNameString(TreeNode node) {
      IXTNTreeTableNode iNode = (IXTNTreeTableNode)node;
      List props = getProperties(iNode);
      //String nameString = ((Named)iNode.node).getName() + "(" + 
      //  UIUtil.listToDisplay(pattern) + ")";
      String nameString = ((Named)iNode.node).getName() + " " + 
	UIUtil.show(props);
      return nameString;
    }
    /**
     * Gets the strings that are to be used as column names.
     */
    void setColumnNames() {
      //Debug.noteln("AROEd: setting column names");
      Class[] classes = {String.class, String.class};
      String[] names = {"Name(properties)"};
      super.setColumnClasses(classes);
      super.setColumnNames(names);
    }

    protected List getProperties(IXTNTreeTableNode iNode) {
      Object node = iNode.node;
      List props = null;
      if (node instanceof UIObjectClass)
	props = ((UIObjectClass)node).getObjectProperties();
      else if (ObjectClass.class.isInstance(node))
	props = ((ObjectClass)node).getObjectProperties();
      else return null;
      //Debug.noteln("AROEd: pattern is", UIUtil.show(props));
      return props;
    }
    
    /**
     * Looks up all object classes and updates the tree model from
     * them. This method retains the expansion information.
     */
    public void reloadData() {
      //Debug.noteln("AROE/CTM: reloadData");
      //(new Throwable()).printStackTrace();
      List constructs = uiDomain.getAllObjectClasses();
      //Debug.noteln("AROE/CTM: Classes:", UIUtil.listToDisplay(constructs));
      constructs = 
	(List)IVUtil.sortNamedCollection(new ArrayList(constructs));   
      //Debug.noteln(" after sort:", UIUtil.listToDisplay(constructs));
      List data = new ArrayList(constructs);
      //Debug.noteln("AROE/CTM: as ArrayList:", UIUtil.listToDisplay(data));
      updateData(data);
    }
  }


}

/*Issues***************************************************************
 *
 * implement this?
 *
/*Todos***************************************************************
 *
 *****************************************************************************
 */
