/****************************************************************************
 * A TreeModel for Named objects that can be edited.
 *
 * @author Jussi Stader
 * @version 2.0
 * Updated: Thu Sep 28 14:08:17 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */

package ix.iface.ui.tree;

import java.util.*;
import javax.swing.*;
import javax.swing.tree.*;

import ix.util.*;
import ix.util.lisp.*;
import ix.icore.domain.*;
import ix.icore.domain.event.*;
import ix.icore.event.*;
import ix.icore.*;
import ix.iview.domain.*;
import ix.iview.util.*;
import ix.iview.domain.event.*;
import ix.iface.ui.event.*;
import ix.iface.ui.util.*;
import ix.iface.ui.*;


/****************************************************************************
 * A DefaultTreeModel for Named objects that can be edited.
 *
 * Abstract methods to define:
 *  ---finding and creating user objects and tree nodes
 *  IXTreeNode newIXTreeNode(Object userNode) create a node for this model
 *  boolean isOwnUserObject(Object object)    check for userObject of this modl
 *  List findUserObjects()                    
 *  Object findUserNode(String name)          as in getNamedObject
 *  Object makeUserObject(String name)        create a user object 
 *  Object makeUserObject(String name, List children) ditto
 *  ---to manipulate tree structure
 *  List getUserChildren(Object userParent)   
 *  void addUserChild(Object userParent, Object newChild)
 *  void removeUserChild(Object userParent, Object oldChild)
 *  List getUserParents(Object userObject)
 *  void addUserParent(Object userObject, Object newParent)
 *  void removeUserParent(Object userObject, Object oldParent)

 *
 *****************************************************************************
 */
public abstract class EditableTreeModel extends DefaultTreeModel
  implements DataChangeListener
{

  //IXTree tree;

  /** contains entries of (userNode - HashSet{IXTreeNodes}) */
  HashMap nodeMap = new HashMap();
  
  /** contains the set of user nodes that have been added to the tree */
  HashSet userNodes = new HashSet();

  public EditableTreeModel() {
    this(null);
  }

  /**
   * Constructs a new instance with the given node as the root node
   * of the tree.
   * @param root The root node of the tree model.  
   */
  public EditableTreeModel(IXTreeNode root) {
    super(root);
    this.nodeMap = new HashMap();
  }
  /**
   * Constructs a new instance with the given node as the root node
   * of the tree. Assumes that the tree has been built already.
   * @param root The root node of the tree model.  
   * @param map the tree node map.
   */
  public EditableTreeModel(IXTreeNode root, HashSet userNodes, 
			   HashMap nodeMap) {
    super(root);
    this.nodeMap = nodeMap;
    this.userNodes = userNodes;
  }

  public boolean isUserRoot(Object userNode) {
    IXTreeNode root = (IXTreeNode)getRoot();
    //Debug.noteln("EdTM: root is", root);
    if (root == null) return false;
    else if (userNode == null) return false;
    else return userNode.equals(root.getUserObject());
  }
  public Object getUserRoot() {
    IXTreeNode root = (IXTreeNode)getRoot();
    //Debug.noteln("EdTM: root is", root);
    if (root == null) return null;
    else return root.getUserObject();
  }
  public void setUserRoot(Object userNode) {
    List userParents = getUserParents(userNode);
    if ((userParents != null) && (userParents.size() > 0)) {
      Debug.noteln("EdTM: trying to set root to node with parents", userNode);
      return;
    }
    else {
      IXTreeNode root = ensureIXTreeNode(userNode, "");
      //Debug.noteln("EdTM: setting root to user object", root);
      setRoot(root);
    }
  }

  /**
   * Scroll and open the tree to make the given node visible.
   */
  /*
  public void showUserNode(Object userNode) {
    if (tree == null) return;
    //Debug.noteln("EdTM: showing node", userNode);
    TreePath oldPath = tree.getSelectionPath();
    if (oldPath != null) { //old selection any use?
      Object node = oldPath.getLastPathComponent();
      if ((node != null) && (node instanceof IXTreeNode))
	showUserNode(userNode, ((IXTreeNode)node).getUserObject());
      else showUserNode(userNode, null);
    }
  }
  */
  /**
   * Scroll and open the tree to make the given node visible.
   * The old node may have been selected before, so if it is an
   * ancestor, show the version of userNode that descends from it.
   */
  /*
  public void showUserNode(Object userNode, Object oldUserNode) {
    //Debug.noteln("EdTM: Showing node", userNode);
    if (isUserRoot(userNode)) {
      //Debug.noteln("EdTM: got user root");
      showRoot();
      return;
    }
    Collection treeNodes = findTreeNodes(userNode);
    if (treeNodes != null) 
      while (treeNodes.contains(null)) treeNodes.remove(null);
    if ((treeNodes == null) || (treeNodes.size() == 0)) {
      showRoot();
      return;
    }
    Collection oldNodes = findTreeNodes(oldUserNode);
    if (oldNodes != null) 
      while (oldNodes.contains(null)) oldNodes.remove(null);

    //Debug.noteln("EdTM: got tree nodes", treeNodes);
    IXTreeNode showNode = null;

    if ((treeNodes.size() != 1) && //only one choice, so treat simply
	(oldNodes != null) && (oldNodes.size() > 0)) {//no others; simple again
      for (Iterator i = treeNodes.iterator(); 
	   i.hasNext() && (showNode == null); ) {
	try {
	  DefaultMutableTreeNode treeNode = (DefaultMutableTreeNode)i.next(); 
	  if (treeNode != null) {
	    for (Iterator j = oldNodes.iterator();  
		 j.hasNext() && (showNode == null); ) {
	      if (treeNode.isNodeAncestor((TreeNode)j.next())) 
		showNode = (IXTreeNode)treeNode;
	    }
	  }
	}
	catch (Exception e) {}
      }
    }
    if (showNode == null) {//simple or nothing found
      try {showNode = (IXTreeNode)treeNodes.iterator().next();}
      catch (Exception e) {}
      //Debug.noteln("EdTM: got simple node", showNode);
    }
    if (showNode == null)  //this may happen after delete
      showRoot();
    else showPath(showNode.getPath());
  }
  */

  public boolean hasTreeNode(Object userNode, String parentName) {
      return (getTreeNode(userNode, parentName) != null);
  }
  public IXTreeNode getTreeNode(Object userNode, String parentName) {
    if (userNode == null) return null;
    Object notes = nodeMap.get(userNode);
    if (notes == null) return null;
    else {
      Collection treeNodes = (Collection)notes;
      if (treeNodes.isEmpty()) return null;
      else {
	//look for the parent name in the nodes' parents
	for (Iterator i = treeNodes.iterator(); i.hasNext(); ) {
	  IXTreeNode treeNode = (IXTreeNode)i.next();
	  if ((parentName == null) || (parentName == ""))
	    //not interested in parent, just return first one
	    return treeNode; 
	  else { //check whether this one has the right parent
	    TreeNode parent = treeNode.getParent();
	    if ((parent != null) && (parent instanceof IXTreeNode)) {
	      Named parentO = (Named)((IXTreeNode)parent).getUserObject();
	      String pName = parentO.getName();
	      if (parentName.equals(pName)) return treeNode;   //---->found it
	    }
	  }
	}
	return null; //not found
      }
    }
  }
  public IXTreeNode ensureIXTreeNode(Object userNode, String parentName) {
    if (userNode == null) return null;
    IXTreeNode previous = getTreeNode(userNode, parentName);
    if (previous == null) {
      IXTreeNode node = newIXTreeNode(userNode);
      ((EditableObject)userNode).addDataChangeListener(this);
      addNodeMap(userNode, node);
      return node;
    }
    else return previous;
  }

  private void addNodeMap(Object userNode, IXTreeNode treeNode) {
    Object notes = nodeMap.get(userNode);
    Collection c = null;
    if (notes != null)
      c = (Collection)notes;
    else c = new HashSet();
    c.add(treeNode);
    nodeMap.put(userNode, c);
  }

  public IXTreeNode makeIXTreeNode(Object userNode, String parentName) {
    if (userNode == null) return null;
    IXTreeNode node = newIXTreeNode(userNode);
    ((EditableObject)userNode).addDataChangeListener(this);
    if (((parentName == null) || (parentName == "")) 
	&& (root == null))
      setRoot(node);
    addNodeMap(userNode, node);
    return node;
  }
  public void ensureMapEntry(IXTreeNode node, String parentName) {
    if (node == null) return;
    if ((parentName == null) || (parentName == "")) 
      parentName = IXTrees.ROOTNAME;
    Named userNode = (Named)node.getUserObject();
    HashSet treeNodes = (HashSet)nodeMap.get(userNode);
    if (treeNodes == null) {
	treeNodes = new HashSet();
	nodeMap.put(userNode, treeNodes);
    }
    treeNodes.add(node);
  }


  public void buildTree(Object parentNode, String preParent) {
    //Debug.noteln("EdTreeM: building tree for", parentNode);
    if (parentNode == null) return;
    if (parentNode instanceof IXTreeNode) {
      //tree node already made. Make sure it's noted, do kids if not done
      IXTreeNode parent = (IXTreeNode)parentNode;
      Object userParent = parent.getUserObject();
      if (!userNodes.contains(userParent)) { // not already built this
	userNodes.add(userParent);  //building now, so remember
	ensureMapEntry(parent, preParent);
	buildSubTree(parent);
      }
    }
    else { // user node given. Check if done, else make tree node and kids
      Object userParent = parentNode; 
      if (!userNodes.contains(userParent)) { // not already built this
	userNodes.add(userParent);  //building now, so remember
	if (!hasTreeNode(userParent, preParent)) {
	  IXTreeNode parent = makeIXTreeNode(userParent, preParent);
	  if (parent != null) {
	    buildSubTree(parent);
	    fireNodeAdded(parent);
	  }
	}
      }
    }
  }

  /**
   * Only call this if the sub-tree really needs to be built.
   * Collects user children, makes their TNs, adds them to TN parent, 
   * and builds their sub-trees (if needed)
   */
  protected void buildSubTree(IXTreeNode parent) {
    //Debug.noteln("EdTreeM: building subTree for", parent);
    if (parent == null) return;
    Object userParent = parent.getUserObject();
    List userChildren = getUserChildren(userParent);
    //Debug.noteln(" EdTreeM: building children", UIUtil.show(userChildren));
    String parentName = ((Named)userParent).getName();
    for (Iterator i = userChildren.iterator(); i.hasNext(); ) {
      Object userChild = i.next();
      if (!hasTreeNode(userChild, parentName)) {
	IXTreeNode child = makeIXTreeNode(userChild, parentName);
	if (child != null) {
	  insertNodeInto(child, parent, getChildCount(parent));
	  //parent.addChild(child);
	  fireNodeAdded(child); //was parent!?
	  buildTree(child, parentName);
	}
      }
    }
  }

  public IXTreeNode findStringTreeNode(String nodeName) {
    Object userNode = findUserNode(nodeName);
    return getTreeNode(userNode, null);
  }

  public IXTreeNode findTreeNode(Object userObject) {
    return getTreeNode(userObject, null);
  }
  private Collection findAllTreeNodes(Object userObject) {
    return findTreeNodes(userObject);
  }
  protected Collection findTreeNodes(Object userObject) {
    return (Collection)nodeMap.get(userObject);
  }


  boolean isUpdating = false;
  public void startUpdate() {
    isUpdating = true;
  }
  public void stopUpdate() {
    isUpdating = false;
    reload();
  }
  /** End update without firing */
  public void ignoreUpdate() {
    isUpdating = false;
  }

  public void setRoot(IXTreeNode root) {
    //Debug.noteln("EdTM: setting root to tree node", root);
    startUpdate();
    clearModel();
    super.setRoot(root);
    ensureMapEntry(root, "");
    //Debug.noteln(" ensured map entry");
    buildTree(root, ""); //ensure map entries etc.
    // Debug.noteln(" built tree");
    stopUpdate();
    //showRoot();
  }


  public IXTreeNode newTreeChild(UIObject userParent) {
    return newTreeChild(findTreeNode(userParent));
  }
  public IXTreeNode newTreeChild(IXTreeNode parent) {
    Object userChild = makeUserObject(" ");
    UIObject userParent = (UIObject)parent.getUserObject();
    addUserParent(userChild, userParent);
    IXTreeNode newChild = makeIXTreeNode(userChild, userParent.getName());
    if (newChild != null)
      insertNodeInto(newChild, parent, getChildCount(parent));
    //reload();
    return newChild;
  }
  public IXTreeNode addTreeParent(IXTreeNode parent, UIObject userChild) {
    return addTreeParent(parent, findTreeNode(userChild));
  }
  public IXTreeNode addTreeParent(IXTreeNode parent, IXTreeNode child) {
    //sort out userObject side
    Object userChild = child.getUserObject();
    UIObject userParent = (UIObject)parent.getUserObject();
    addUserParent(userChild, userParent);
    //make new copy of child to allow one more parent
    IXTreeNode newChild = makeIXTreeNode(userChild, userParent.getName());
    if (newChild != null)
      insertNodeInto(newChild, parent, getChildCount(parent));
    return newChild;
  }


  public void reloadData() {
    clearModel();
    //Debug.noteln("EdTM: reloading model data");
    List uios = findUserObjects();
    //Debug.noteln(" found data", IVUtil.printUIObjects(uios));
    IXTrees.setupTreeModel(this, uios);  //calls setRoot()
    //Debug.noteln(" done reload");
  }


  public void clearModel() {
    //keep the tree
    userNodes.clear();
    nodeMap.clear();
  }
    
  /** Creates a new instance of a suitable IXTreeNode from the user object */
  public abstract IXTreeNode newIXTreeNode(Object userNode);

  public abstract boolean isOwnUserObject(Object object);

  public abstract List findUserObjects();
  public abstract Object findUserNode(String name);
  public abstract Object makeUserObject(String name);
  public abstract Object makeUserObject(String name, List children);

  public abstract List getUserChildren(Object userParent);
  public abstract void addUserChild(Object userParent, Object newChild);
  public abstract void removeUserChild(Object userParent, Object oldChild);

  public abstract List getUserParents(Object userObject);
  public abstract void addUserParent(Object userObject, Object newParent);
  public abstract void removeUserParent(Object userObject, Object oldParent);

  public void reload() {
    reload(root);
  }
  
  public void reload(IXTreeNode node) {
    if(node != null) {
      fireTreeStructureChanged(this, getPathToRoot(node), null, null);
    }
  }
  
  public void dataChanged(EditableObject obj, String field,
			  Object oldVal, Object newVal) {
    //Debug.noteln("EdTM: got data change event for", obj);
    userNodeChanged(new ObjectChangeEvent(this, obj, field, oldVal, newVal));
  }


  /**
   * If a node has a new parent, we have to make a new IXTreeNode and add that.
   * A node that did not previously have a parent, now can be added to the tree
   */
  private boolean handleNewParents(ObjectChangeEvent oce) {
    //Debug.noteln("EdTM: got change event", oce.print());
    if (oce.getField().equals("superClassNames")) {
      List oldVal = (List)oce.getOldValue();
      if ((oldVal == null) || (oldVal.size() == 0)) { //no old parent, so add 
	//Debug.noteln("  first parent");
	userNodeAdded(oce.getObject(), oce);
	return true;
      }
      else {
	List userParents = getUserParents(oce.getObject());

	if (userParents.size() > oldVal.size()) { //added parents
	  //add a node for each new parent
	  if (userParents == null) return true;
	  ArrayList newParents = new ArrayList();
	  for (Iterator i = userParents.iterator(); i.hasNext();) {//remove old
	    Named parent = (Named)i.next();
	    Symbol name = Symbol.intern(parent.getName());
	    if (!oldVal.contains(name)) newParents.add(parent);
	  }
	  addChildToParents(oce.getObject(), newParents);
	  return true;
	}
	
	else { //removed parents or same as before
	    //Debug.noteln(" old value is", oldVal);
	    //Debug.noteln(" new value is", userParents);
	  UIUtil.without(oldVal, userParents);
	  for (Iterator i = userParents.iterator(); i.hasNext();) {
	    Named parent = (Named)i.next();
	    Symbol name = Symbol.intern(parent.getName());
	    oldVal.remove(name);
	  }
	  //Debug.noteln(" old without new is", UIUtil.show(oldVal));
	  removeChildFromParents(oce.getObject(), oldVal);
	  return true;
	}
      }
    }
    else return false;
  }
  
  public void userNodeChanged(ObjectChangeEvent oce) {
    //Debug.noteln("EdTM: got node change", oce.getField());
    Object userNode = oce.getObject();
    //Debug.noteln(" for node", userNode);
    if (isOwnUserObject(userNode)) {
      if (!handleNewParents(oce)) { //previous fail to add without parent
	Collection nodes = findAllTreeNodes(userNode);
	if (nodes != null) {
	  for (Iterator i = nodes.iterator(); i.hasNext(); ) {
	    IXTreeNode treeNode = (IXTreeNode)i.next();
	    fireNodeChanged(treeNode, oce);
	  }
	}
      }
    }
  }
  public void userNodeRemoved(Object userNode) {
    if (isOwnUserObject(userNode)) {
      Collection tns = findAllTreeNodes(userNode);
      if (tns != null) {
	Collection nodes = new ArrayList(findAllTreeNodes(userNode));
	for (Iterator i = nodes.iterator(); i.hasNext(); ) {
	  IXTreeNode treeNode = (IXTreeNode)i.next();
	  if (treeNode != null) {
	    removeNodeFromMaps(treeNode);
	    TreeNode parent = treeNode.getParent();
	    if (parent != null) 
	      removeNodeFromParent(treeNode);
	  }
	}
      }
    }
    userNodes.remove(userNode);
  }

  private void removeNodeFromMaps(IXTreeNode treeNode) {
    Object userNode = treeNode.getUserObject();
    Collection treeNodes = (Collection)nodeMap.get(userNode);
    treeNodes.remove(treeNode);
  }

  /**
   * This only works if the added node already has a parent
   */
  public void userNodeAdded(Object userNode, ObjectChangeEvent oce) {
    //Debug.noteln("EdTM: got node added event for", userNode);
    if (isOwnUserObject(userNode)) {
      //hook into tree for all parents
      List parents = getUserParents(userNode);
      //for each parent, make a child IX node and insert at end.
      addChildToParents(userNode, parents);
    }
  }

  private void removeChildFromParents(Object userNode, List userParentNames) {
    //if (userParentNames == null) Debug.noteln("  got null parents");
    //else Debug.noteln("  got parents ", UIUtil.show(userParentNames));
    for (Iterator i = userParentNames.iterator(); i.hasNext(); ) {
      Object userParent = i.next();
      if (userParent != null) {
	//Debug.noteln("EdTM: got parent of class", userParent.getClass());
        String parentName = userParent.toString();
	IXTreeNode childTreeNode = getTreeNode(userNode, parentName);
	//Debug.noteln(" removing child frm parent "+parentName,childTreeNode);
	if (childTreeNode != null) removeNodeFromParent(childTreeNode);
      }
    }
  }

  private boolean addChildToParents(Object userNode, List userParents) {
    //Debug.noteln("EdTM: adding user child", userNode);
    if (userParents == null) {
      //Debug.noteln("  got null user parents");
      if (root == null) {
	setUserRoot(userNode);
	return true;
      }
      else return false;
    }
    //else Debug.noteln("  to user parents", UIUtil.show(userParents));
    for (Iterator i = userParents.iterator(); i.hasNext(); ) {
      Object userParent = i.next();
      if (userParent != null) {
        String parentName = ((Named)userParent).getName();
	IXTreeNode child = makeIXTreeNode(userNode, parentName);
	if (child != null) {
	  //Debug.noteln("  made child");
	  Collection treeParents = findTreeNodes(userParent);
	  //Debug.noteln("  got tree parents ", UIUtil.show(treeParents));
	  if (treeParents == null) return false; //treeParents not made yet
	  for (Iterator j = treeParents.iterator(); j.hasNext(); ) {
	    IXTreeNode parent = (IXTreeNode)j.next();
	    if (parent != null) {
	      insertNodeInto(child, parent, getChildCount(parent));
	      //parent.addChild(child);
	      fireNodeAdded(child);
	      buildTree(child, parentName);
	    }
	  }
	}
      }
    }
    return true;
  }

  public void insertNodeInto(MutableTreeNode c,  MutableTreeNode p, int i) {
    //Debug.noteln("EdTM: inserting node", c);
    super.insertNodeInto(c, p, i);
    //Debug.noteln("  EdTM: inserted node into", p);
    //Debug.noteln("  depth of child is", getPathToRoot(c).length);

  }

  public IXTreeNode getTreeParent(IXTreeNode treeNode) {
    TreeNode[] path = getPathToRoot(treeNode);
    if (path == null) {
      Debug.noteln("EdTM: no path for finding parent", treeNode);
      return null;
    }
    else {
      int l = path.length;
      if (l == 1) {
	//Debug.noteln("EdTM: root has no parent", treeNode);
	return null; //root - has no parent
      }
      else {
	return (IXTreeNode)path[l-2];
      }
    }
  }

  //---------------------printing-----------------------------------------

  public String printSubTree() {
    return printSubTree(getUserRoot());
  }
  public String printSubTree(Object userNode) {
    if (userNode == null) return "";
    Collection treeNodes = findAllTreeNodes(userNode);
    if (treeNodes == null) 
      return userNode.toString() + " has no tree node";
    String text = "";
    for (Iterator i = treeNodes.iterator(); i.hasNext(); ) {
      text = text + printSubTreeNode(i.next());
    }
    return text;

  }
  private String printSubTreeNode(Object treeNode) {
    if (treeNode == null) return "null";
    String text = treeNode.toString();
    if (treeNode instanceof IXTreeNode) {
      IXTreeNode tn = (IXTreeNode)treeNode;
      text = text + " " + tn.getChildCount() + " children. ";
      for (int i = 0; i < tn.getChildCount(); i++) {
	text = text + ",  " + printSubTreeNode(tn.getChildAt(i));
      }
    }
    else text = text + " of class " + treeNode.getClass();
    return text;
  }

  //---------------------firing-----------------------------------------

  public void fireNodeChanged(IXTreeNode treeNode, ObjectChangeEvent oce) {
    if (isUpdating) return;
    if (treeNode == null) return;
    IXTreeNode parent = getTreeParent(treeNode);
    if (parent != null) { //the node's parent looks after this node
      int[] inds = {parent.getIndex(treeNode)};
      Object[] kids = {treeNode};
      //Debug.noteln("EdTM: firing tree node change");
      fireTreeNodesChanged(parent, parent.getPath(), inds, kids);
    }
    //in case node's children changed
    //handleChildChange(treeNode, oce);
    else reload();
  }
  public void fireNodeAdded(IXTreeNode treeNode) {
    if (isUpdating) return;
    //Debug.noteln("EdTM: Firing node added for", treeNode);
    if (treeNode == null) return;
    IXTreeNode parent = getTreeParent(treeNode);
    //IXTreeNode parent = (IXTreeNode)treeNode.getParent();
    //Debug.noteln("EdTM: node inserted into", parent);
    if (parent != null) {
      int[] inds = {parent.getIndex(treeNode)};
      Object[] kids = {treeNode};
      fireTreeNodesInserted(parent, parent.getPath(), inds, kids);
    }
    else reload();
  }
  public void fireTreeNodesInserted(Object s, Object[] p, 
				    int[] i, Object[] c) {
    if (isUpdating) return;
    //Debug.noteln("EdTM: Firing node inserted for", s);
    super.fireTreeNodesInserted(s, p, i, c);
  }
  public void fireTreeNodesChanged(Object s, Object[] p, int[] i, Object[] c) {
    if (isUpdating) return;
    //Debug.noteln("EdTM: Firing node change for parent", s);
    super.fireTreeNodesChanged(s, p, i, c);
  }
  
}




