/****************************************************************************
 * Abstract table that listens to the mouse, has values with up to 4D and
 * sensible column widths.
 * 
 * @author Jussi Stader
 * @version 1.0
 * Updated: Mon Sep 25 09:40:09 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 ****************************************************************************
 */
package ix.iface.ui.table;


import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;

import ix.iface.ui.NDRenderer;
import ix.iface.ui.IconRenderer;
import ix.iface.ui.FourField;
//import ix.iface.ui.*;
import ix.util.Debug;
import ix.util.ConsistencyException;
//import ix.util.*;

/****************************************************************************
 * Abstract table that listens to the mouse, has values with up to 4D and
 * sensible column widths.
 *
 * The IXTable is used to provide a standard table behaviour for displaying
 * up to fourD displays of fields (as provided by the NDRenderer), adjusted
 * column widths, no re-ordering of columns, adding the model as a
 * TableModelListener if it is one.<p>
 * Can be used for normal tables or tree tables. <p>
 *
 * Example code for making a TreeTable happen:
 * <PRE><code>
 *   ...
 *     TreeNode[] tna = {};
 *     TreeNode[] nodes = domain.getAllRefinements().toArray(tna);
 *     ActionTreeTableModel tableModel = new ActionTreeTableModel(nodes);
 *     IXTable table = new IXTable(tableModel);
 *     table.setPreferredScrollableViewportSize(new Dimension(200, 100));
 *     JScrollPane jsp = new JScrollPane();
 *     jsp.getViewport().add(table);
 *     panel.add(jsp);
 *   ...
 * </code></PRE>
 ****************************************************************************
 */

public class IXTable extends JTable implements MouseListener, FocusListener {

  public Hashtable columnWidths;
  public Hashtable columnFixedWidths;
  public int defaultWidth = 50;
  public int minWidth = 15;

  private JPopupMenu popup;

  /** The TableModelListeners that have been added */
  private ArrayList tmListeners = new ArrayList();
  
  /** 
   * Makes a single-selection table that listens to mouse, shows grid,
   * renders 4D.
   */
  public IXTable(){
    //    Insets ins = getInsets();
    //    setSize(ins.left + ins.right + 430,ins.top + ins.bottom + 270);
    
    setShowGrid(true);
    this.addMouseListener(this);
    this.addFocusListener(this);
    this.getSelectionModel()
      .setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    
    NDRenderer ndr = new NDRenderer(false);
    ndr.setToolTipText("Click for choices");
    IconRenderer ir = new IconRenderer(false);
    //ir.setToolTipText("click to expand/collapse");
    this.setDefaultRenderer(FourField.class, ndr);
    this.setDefaultRenderer(Icon.class, ir);
    //by default columns are not reorderable
    this.getTableHeader().setReorderingAllowed(false);
    setRowHeight(getRowHeight() + 5);
  }

  /** 
   * Makes a table and sets the given model, adjusts column widths, registers
   * model as mouse listener (if appropriate).
   */
  public IXTable(AbstractTableModel model) {
    this();
    this.setModel(model);
  }

  public void setModel(TableModel model) {
    super.setModel(model);
    if (model instanceof TableMouseListener) {
      //Debug.noteln("IXT: adding table mouse listener", model);
      this.addTableMouseListener((TableMouseListener)model);
    }
    setWidths();
    adjustWidths();
  }

  public void setSelectedRow(int row) {
    ListSelectionModel lsm = getSelectionModel();
    if ((row >= 0) && (row < getModel().getRowCount())) {
      lsm.clearSelection();
      lsm.setSelectionInterval(row, row);
    }
  }

  public void setSelectedObject(Object object) {
    if (AbstractIXTableModel.class.isInstance(getModel())) {
      int row = ((AbstractIXTableModel)getModel()).getObjectRow(object);
      if (row != -1) setSelectedRow(row);
      else clearSelection();
    }
  }

  public Object getSelectedObject() {
    if (AbstractIXTableModel.class.isInstance(getModel())) {
      int row = getSelectedRow();
      try {
	return ((AbstractIXTableModel)getModel()).getRowObject(row);
      } catch (Exception e) {}
    }
    return null;
  }

  public TableColumn findColumn(String header) {
    for (Enumeration e = this.getColumnModel().getColumns()
	   ; e.hasMoreElements() ; ) {
      TableColumn col = (TableColumn)e.nextElement();
      if (col.getHeaderValue().equals(header))
	return col;
    }
    throw new ConsistencyException("Can't find column", header);
  }


  protected LinkedList columnToolTips = new LinkedList();
  public void setToolTipTexts(String[] columnTips) {
    columnToolTips = new LinkedList(Arrays.asList(columnTips));
  }
  public void setToolTipTexts(Collection columnTips) {
    columnToolTips = new LinkedList(columnTips);
  }

  public void setToolTipText(int columnModelIndex, String columnTip) {
    TableColumnModel columnModel = getColumnModel();
    int cols = columnModel.getColumnCount();
    if (columnModelIndex >= cols) {
      Debug.noteln("IXTable: max columns is", cols +" not "+ columnModelIndex);
      return;
    }
    if (columnToolTips == null) columnToolTips = new LinkedList();
    if (columnToolTips.size() <= cols) {
      for (int i = columnToolTips.size()+1; i <= cols; i++) {
	columnToolTips.add(null);
      }
    }
    columnToolTips.set(columnModelIndex, columnTip);
  }

  protected JTableHeader createDefaultTableHeader() {
    TableColumnModel columnModel = getColumnModel();
    return new JTableHeader(columnModel) {
	public String getToolTipText(MouseEvent me) {
	  try {
	    java.awt.Point p = me.getPoint();
	    int index = columnModel.getColumnIndexAtX(p.x);
	    int realIndex = columnModel.getColumn(index).getModelIndex();
	    return (String)columnToolTips.get(realIndex);
	  }
	  catch (Exception e) {return null;}
	}
      };
  }


  /**
   * Find out whether this table has a popup.
   */
  public boolean hasPopup() { return popup != null;}
  /**
   * Return the popup of the table if it has one
   */
  public JPopupMenu getPopup() { return popup;}
  /**
   * Set the popup of the table
   */
  public void setPopup(JPopupMenu pop) { popup = pop;}





  public void setWidths() {
    setDefaultWidths();
  }
  public void setDefaultWidths() {
    if (columnWidths == null) {
      //Debug.noteln("IXT: columnWidths is null");
      columnWidths = new Hashtable();
    }
    if (columnFixedWidths == null) {
      //Debug.noteln("IXT: columnFixedWidths is null");
      columnFixedWidths = new Hashtable();
    }
    setWidth(String.class,  new Integer(100));
    setWidth(FourField.class,  new Integer(100));
    setFixedWidth(Boolean.class,  new Integer(15));
    setFixedWidth(Icon.class, new Integer(15));
    setFixedWidth(ImageIcon.class,  new Integer(15));
    setFixedWidth(Character.class,  new Integer(15));
  }

  public void setWidth(Class colClass, Integer width) {
    columnWidths.remove(colClass);
    columnWidths.put(colClass, width);
  }
  public void setWidth(String title, Integer width) {
    columnWidths.remove(title);
    columnWidths.put(title, width);
  }
  public void setFixedWidth(Class colClass, Integer width) {
    columnWidths.remove(colClass);
    columnFixedWidths.remove(colClass);
    columnFixedWidths.put(colClass, width);
  }
  public void setFixedWidth(String title, Integer width) {
    columnWidths.remove(title);
    columnFixedWidths.remove(title);
    columnFixedWidths.put(title, width);
  }

  /**
   * Adjusts column widths to sensible values.
   * String: 100, FourField: 100, boolean: 25, others 50.
   * Relative width is preserved on resizing.
   */
  public void adjustWidths() {
    if (columnWidths == null) {
      //Debug.noteln("IXT: Cannot adjust widths with null map");
      return;
    }
    TableColumn column;
    Class colClass;
    boolean fixed = false;
    String colName;
    TableModel model = getModel();
    for (int i = 0; i < getColumnModel().getColumnCount(); i++) {
      column = getColumnModel().getColumn(i);   //work on column i
      colName = model.getColumnName(i);
      //try this order: name fixed, name flex, class fixed, class flex
      Object oWidth = columnFixedWidths.get(colName);
      if (oWidth != null) fixed = true;
      else {
	oWidth = columnWidths.get(colName);
	if (oWidth != null) fixed = false;
	else {
	  colClass = model.getColumnClass(i);   //get class for width
	  oWidth = columnFixedWidths.get(colClass);
	  if (oWidth != null) fixed = true;
	  else {
	    fixed = false;
	    oWidth = columnWidths.get(colClass);
	    if (oWidth == null) oWidth = new Integer(defaultWidth);
	  }
	}
      }
      int width = ((Integer)oWidth).intValue();
      column.setPreferredWidth(width);
      if (fixed) {
	column.setMinWidth(width);
	column.setMaxWidth(width);
	column.setResizable(false);
      }
      /*
      //Debug.noteln("IXTable: doing column class " + colClass);
      if ((String.class.isAssignableFrom(colClass)) ||
	  (FourField.class.isAssignableFrom(colClass)))
	column.setPreferredWidth(100);
      else if (Boolean.class.isAssignableFrom(colClass)) 
	column.setPreferredWidth(15);
      else if ((Icon.class.isAssignableFrom(colClass)) ||
	       (ImageIcon.class.isAssignableFrom(colClass))) {
	column.setPreferredWidth(15);
	column.setMinWidth(15);
	column.setMaxWidth(15);
	column.setResizable(false);
      }
      else if (Character.class.isAssignableFrom(colClass))
	column.setPreferredWidth(15);
      else column.setPreferredWidth(50);
      //Debug.noteln(" set width to "+ column.getPreferredWidth());
      */
    }
  }




  public boolean isMarkable() {
    try { 
      return ((AbstractIXTableModel)getModel()).isMarkable();
    }
    catch (Exception e) {
      Debug.noteException(e);
    }
    return false;
  }

  public void markRow(int row) {
    if (isMarkable()) ((AbstractIXTableModel)getModel()).markRow(row);
  }
  public void unmarkRow(int row) {
    if (isMarkable()) ((AbstractIXTableModel)getModel()).unmarkRow(row);
  }
  public void clearMarks() {
    if (isMarkable()) ((AbstractIXTableModel)getModel()).clearMarks();
  }
  public void markAll() {
    if (isMarkable()) ((AbstractIXTableModel)getModel()).markAll();
  }

  public boolean isMarked(int row) {
    if (isMarkable()) 
      return ((AbstractIXTableModel)getModel()).isMarked(row);
    else return false;
  }
  public boolean isMarkedEmpty() {
    if (isMarkable()) 
      return ((AbstractIXTableModel)getModel()).isMarkedEmpty();
    else return false;
  }








  public void addTableMouseListener(TableMouseListener tml) {
    tmListeners.add(tml);
    //Debug.noteln("IXT: added listener", tml);
  }
  public void removeTableMouseListener(TableMouseListener tml) {
    while (tmListeners.contains(tml)) {
      tmListeners.remove(tmListeners.indexOf(tml));
    }
    //Debug.noteln("IXT: removed listener", tml);
  }

  /** 
   * Send a cellClicked notification to all registered TableMouseListeners.
   * Passes on the mouse event and the row/column index that has been clicked.
   * Works for single-selection tables only.
   */
  public void mouseClicked(MouseEvent me) {
    //Debug.noteln("IXTable - mouse clicked for listeners", tmListeners);
    int row = getSelectedRow();
    int col = getSelectedColumn();
    //Debug.noteln(" - row: " + row + " column: " + col);
    Iterator e = tmListeners.iterator();
    while (e.hasNext()) 
      ((TableMouseListener)e.next()).cellClicked(me,row,col);
  }
  public void mouseEntered(MouseEvent me) {}
  public void mouseExited(MouseEvent me) {}
  public void mousePressed(MouseEvent me) {}
  public void mouseReleased(MouseEvent me) {}

  public void focusGained(FocusEvent fe) {}
  public void focusLost(FocusEvent fe) {
    if (isEditing()) {
      CellEditor ed = getCellEditor();
      Component edComp = null;
      if (ed instanceof DefaultCellEditor) 
	edComp = (( DefaultCellEditor)ed).getComponent();
      else Debug.noteln("IXT: odd table cell editor of class", ed.getClass());
      Component otherComp = fe.getOppositeComponent();
      //Debug.noteln("IXT: lost focus to", otherComp);
      if ((otherComp != null) && !otherComp.equals(this) 
	  && !otherComp.equals(ed) && !otherComp.equals(edComp))
	ed.stopCellEditing();
    }
  }
}

/**
Changes:
  14/01/04: moved renderer tool tip to TreeTable
  05/02/04: made adjustWidths more flexible
*/
