/****************************************************************************
 * An abstract class for tree table models.
 *
 * @author Jussi Stader
 * @version 2.3+
 * Updated: Thu Oct 12 12:03:41 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui.table;

import ix.iface.ui.DefaultColourField;
import ix.iface.ui.util.UIUtil;
import ix.iface.ui.table.event.*;
import ix.util.Debug;
import ix.util.Util;
//import ix.util.*;

import java.awt.Font;
import javax.swing.*;
import javax.swing.table.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import java.awt.event.*;
import java.util.*;

/****************************************************************************
 * An abstract class for tree table models.
 *
 * Creates IXTreeTableNode objects to wrap around the data objects. These 
 * IXTreeTableNode objects hold id information (base-object and parent line)
 * and expansion information (indentation of the node and whether it is 
 * expanded or not). <p>
 *
 * Will add a column 0 or 1 that uses icons to show tree information
 * (right arrow for collapsed nodes, down arrow for expanded nodes,
 * nothing for leaf nodes). The column is 0 if the table is not
 * markable, 1 if it is (marks are always in column 0)
 *
 * Will indent the string in column 1 or 2 to show tree structure. 
 *
 * By default, the children of nodes are not displayed
 * When clicking on the node's tree-icon column, the following happens:
 * <ul>
 * <li> if the node is a leaf: nothing
 * <li> if the node has children and they are displayed: un-display them
 * <li> if the node has children and they are NOT displayed: display them
 * </ul>
 * Multiple parents are no problem because by default nodes children are not 
 * displayed. When a child is displayed more than once, it has a different 
 * parent-line and different IXTreeTableNode obejcts are used. <p>
 *
 * As an example, see ix.iview.table.ActionTreeTableModel
 *****************************************************************************
 */

public abstract class AbstractTreeTableModel extends AbstractIXTableModel 
implements TableMouseListener {

  /** the column to be used as a tree-icon column */
  public int treeIconColumn = 0;
  /** the column to be used as a name column */
  public int nameColumn = 1;
  /** the indentation steps used to indent children */
  public static final int INDENT_STEP = 4;
  /** the character used to indicate a leaf (no children) */
  public static final char LEAFC = ' ';
  /** the character used to indicate an expanded node */
  public static final char OPENC = '-';
  /** the character used to indicate an un-expanded node */
  public static final char CLOSEDC = '+';

  public static final ImageIcon LEAFI = null;
  public static final ImageIcon OPENI = 
     Util.resourceImageIcon("ix-symbol-fold.gif");
  public static final ImageIcon CLOSEDI = 
     Util.resourceImageIcon("ix-symbol-unfold.gif");

  /** a collection of IXTreeTableNodes that have been generated */
  protected ArrayList nodes = new ArrayList();

  /** Listeners that want to know about tree structure changes */
  private HashSet treeListeners = new HashSet();


  /**
   * Makes an empty model.
   */
  public AbstractTreeTableModel(boolean markable){
    super(markable);
  }
  /**
   * Makes a model that contains the given nodes at top level.
   */
  public AbstractTreeTableModel(boolean markable, TreeNode[] theNodes){
    super(markable);
    //add the node, set indentations of nodes to 0
    setData(theNodes);
  }

  public void addTreeTableModelListener(TreeTableStructureListener l) {
    treeListeners.add(l);
  }
  /** The tree lost/gained nodes (more drastic than nodeStructureChanged */
  public void fireTreeStructureChanged() {
      //Debug.noteln("ATTM: Tree structure changed");
    for (Iterator i = treeListeners.iterator(); i.hasNext(); )  {
      TreeTableStructureListener ttsl = (TreeTableStructureListener)i.next();
      ttsl.treeStructureChanged();
    }
  }
  /** A node gained or lost children */
  public void fireNodeStructureChanged(IXTreeTableNode node) {
    if (node == null) return;
    //Debug.noteln("ATTM: Node structure changed", node.toDebug());
    for (Iterator i = treeListeners.iterator(); i.hasNext(); ) {
      TreeTableStructureListener ttsl = (TreeTableStructureListener)i.next();
      ttsl.nodeStructureChanged(node);
    }
  }


  /**
   * Sets the table's rows to the given TreeNode objects and notifies
   * listeners.
   * Creates a top-level IXTreeTableNode object for each data object.
   * Overwrites the super method.
   */
  public void setData(Collection theNodes){
    //Debug.noteln("AbstractTTM: setting table data");
    clearData();
    for (Iterator i = theNodes.iterator(); i.hasNext(); ) {
      Object node = i.next();
      TreeNode eNode = makeIXTreeTableNode(node);
      if (eNode == null) {
	Debug.noteln("WARNING: Tree table is ignoring node", node.toString());
	Debug.noteln(" It is of class " + node.getClass().getName());
	Debug.noteln(" and not a valid TreeNode.");
      }
      else {
	if (!rows.contains(eNode)) {
	  rows.add(eNode);
	  nodes.add(eNode);
	}
      }
    }
    fireTableDataChanged();
  }    
  /**
   * Sets the table's rows to the given TreeNode objects and notifies
   * listeners.
   * Creates a top-level IXTreeTableNode object for each data object.
   * Overwrites the super method.
   */
  public void setData(Object[] theNodes){
    setData(Arrays.asList(theNodes));
  }  

  public void clearData() {
    rows.clear();
    nodes.clear();
    nodeMap.clear();
    clearMarks();
  }
  public void clearTable() {
    rows.clear();
    nodes.clear();
    clearMarks();
    fireTableDataChanged();
  }
  

  /**
   * Sets the table's rows to the given TreeNode objects, maintaining
   * current expansions and notifies listeners. As setData, but it
   * maintains current expansions whereas setData removes them.
   */
  public void updateData(Collection theNodes){
    //Debug.noteln("AbstractTTM: setting table data");
    //clearData();
    //***********todo: throw out stale nodes not in the given collection
    LinkedList newRows = new LinkedList();
    ArrayList newNodes = new ArrayList();
    for (Iterator i = theNodes.iterator(); i.hasNext(); ) {
      Object node = i.next();
      TreeNode eNode = getTreeNode(node);
      if (eNode == null) eNode = makeIXTreeTableNode(node);
      if (eNode == null) {
	Debug.noteln("WARNING: Tree table is ignoring node", node.toString());
	Debug.noteln(" It is of class " + node.getClass().getName());
	Debug.noteln(" and not a valid TreeNode.");
      }
      else {
	newRows.add(eNode);
	newNodes.add(eNode);
      }
      nodes = newNodes;
      rows = newRows;
    }
    fireTableDataChanged();
    // now redo the expansions
    for (Iterator ns = nodes.iterator(); ns.hasNext(); ) {
      IXTreeTableNode n = (IXTreeTableNode)ns.next();
      int row = super.getObjectRow(n);
      if (n.expanded) {
	//Debug.noteln("  sorting expanded node");    
	//unexpand, expand
	unexpandNode(n, row);
	expandNode(n, row); //fires tableDataChanged
      }
    }
  }    

  /**
   * Adds the given TreeNode object to the table's rows and notifies listeners.
   * Creates a top-level IXTreeTableNode object for the data object.
   * Overwrites the super method.
   */
  public void addData(Object data) {
    //Debug.noteln("AbstractTTM: adding table data");
    if (data == null) {
      Debug.noteln("AbstractTTM: Cannot add null data to table");
      (new Throwable()).printStackTrace();
    }
    else {
      IXTreeTableNode eNode =  makeIXTreeTableNode(data);
      if (!rows.contains(eNode)) {
	rows.add(eNode);
	nodes.add(eNode);
	fireTableDataChanged();
	//fireNodeStructureChanged((IXTreeTableNode)eNode.getParent());
	fireTreeStructureChanged();
      }
    }
  }
  /**
   * Removes the given TreeNode object from the table's rows (including its
   * children) and notifies listeners.
   * Overwrites the super method.
   */
  public void removeData(Object data) {
    //Debug.noteln("AbstractTTM: removing table data");
    removeDataNode((IXTreeTableNode)getTreeNode(data));
  }
          
  /**
   * Adds a node to the model in the given place with the given indentation.
   * Also notifies listeners. This happens when nodes are expanded.
   */
  private void addNode(IXTreeTableNode eNode, int place){
    //Debug.noteln("ATTM: addNode", eNode);
    if (eNode != null) {
      rows.add(place, eNode);  //make a new object
      if (eNode.expanded) forceExpandNode(eNode, place); 
      //normal expandNode has guard for already expanded, so use force instead
      fireTableDataChanged();
      fireNodeStructureChanged((IXTreeTableNode)eNode.getParent());
      fireNodeStructureChanged(eNode);
      //fireTreeStructureChanged();
    }
    //Debug.noteln("ATTM: trying to add empty node");
  }

  /**
   * Completely forget about the node (and its children)
   */
  public void removeDataNode(IXTreeTableNode eNode) {
    if (eNode == null) return;
    boolean change = removeChildData(eNode);
    if (rows.contains(eNode)) {
      rows.remove(eNode);
      nodes.remove(eNode);
      removeTreeNode(eNode.node);
      fireNodeStructureChanged((IXTreeTableNode)eNode.getParent());
      change = true;
    }
    if (change) {
      fireTableDataChanged();
    }
    //else Debug.noteln("ATTM: removeNode - no change to note");
  }
  
  /**
   * Removes the given number of nodes from the given index place in the table.
   * Also notifies listeners. This happens when nodes are collapsed.
   */
  public void removeNodes(int number, int place) {
    if (number == 0) return;
    for (int i = 0; i < number; i++) 
      rows.remove(place);
    fireTableDataChanged();
  }

  /**
   * Sets the column fplaces for tree info and name depending on
   * whether the table is markable or not.
   */
  public void setColumnPlaces() {
    if (isMarkable()) {
      treeIconColumn = 1;
      nameColumn = 2;
    }
    else {
      treeIconColumn = 0;
      nameColumn = 1;
    }
  }

  /**
   * Sets the column titles to be used in the table
   */
  public void setColumnNames(String[] fieldNames) {
    //Debug.noteln("ATTM: Setting column names");
    super.setColumnNames(fieldNames);
    //Debug.noteln("ATTM: Setting column places");
    setColumnPlaces();
    //Debug.noteln("ATTM: Adding tree column");
    columnNames.add(treeIconColumn,"");
    Class tClass = determineTreeTypeClass();
    columnClasses.add(treeIconColumn, tClass);
  }
  /**
   * Sets the fields to be used as columns in the table
   */
  public void setColumnFields(Class objectClass, String[] fieldNames) {
    super.setColumnFields(objectClass, fieldNames);
    setColumnPlaces();
    columnFields.add(treeIconColumn,"treeType");
    columnNames.add(treeIconColumn,"");
    Class tClass = determineTreeTypeClass();
    columnClasses.add(treeIconColumn, tClass);
  }

  private Class determineTreeTypeClass() {
    if (OPENI != null) return ImageIcon.class;
    else return Character.class;
    //return ImageIcon.class;
  }


  public Object getRowObject(int rowIndex) {
    if (rowIndex < 0) return null;
    IXTreeTableNode eNode = (IXTreeTableNode)rows.get(rowIndex); 
    return eNode.node;
  }
  public Object getRowNode(int rowIndex) {
    return rows.get(rowIndex); 
  }
  public int getObjectRow(Object object) {
    //find the node's IXTreeTableNode, return that node's index
    TreeNode tNode = getTreeNode(object);
    return super.getObjectRow(tNode);
  }
  public int getNodeRow(TreeNode tNode) {
    return rows.indexOf(tNode);
  }


  private boolean parentsAreBold = false; 
  /**
   * Sets the flag that determines whether the names of parent nodes are 
   * displayed bold or not.
   */
  public void setParentsBold(boolean boldParents) {
    parentsAreBold = boldParents;
  }
  /**
   * Returns the flag that determines whether the names of parent nodes are 
   * displayed bold or not.
   */
  public boolean getParentsBold() {
    return parentsAreBold;
  }
  /**
   * Returns the flag that determines whether the names of parent nodes are 
   * displayed bold or not.
   */
  public boolean isParentsBold() {
    return getParentsBold();
  }

  public boolean isCellEditable(int row, int col) {
    return super.isCellEditable(row, col);
  }

  /**
   * Get the value for a cell in the table defined by the parameters.
   * only the node name is different from normal node tables, so the other
   * cells can be done by any old table model.
   */ 
  public Object getValueAt(int rowIndex, int columnIndex) { 
    //Debug.noteln("ATTM: getValueAt for model", ixTTNListToString(nodes));
    IXTreeTableNode eNode = (IXTreeTableNode)rows.get(rowIndex); 
    Object tnt;
    if (columnIndex == treeIconColumn) 
      tnt = getTreeNodeType(eNode);
    else if (columnIndex == nameColumn) 
      tnt = getTreeValueAt(eNode);
    else if (isMarkable() && (columnIndex == 0))
      tnt = new Boolean(isMarked(rowIndex));
    else {
      int offset = 1; //at least one added column
      if (isMarkable()) offset++; //another added column
      tnt = getCellValueAt(eNode, columnIndex-offset); 
    }
    //Debug.noteln("  value is", tnt);
    return tnt;
  }

  /**
   * Gets the value that indicates the tree-position (indented name).
   * @return an object that can be rendered by the table.
   */
  protected Object getTreeValueAt(IXTreeTableNode eNode) { 
      //Debug.noteln("AbTTM: making indented name");
    String indented = toIndentedString(eNode);
    DefaultColourField ff = new DefaultColourField(indented.toString());
    if (isParentsBold()) {
	//Debug.noteln("AbTTM: making bold", indented);
	//Debug.noteln("  leaf is", eNode.isLeaf());
      if (eNode.isLeaf()) ff.setFont(Font.PLAIN);
      else ff.setFont(Font.BOLD);
    }
    return ff;
  }

  private boolean expanding = false;
  public boolean isExpanding() {return expanding;}
  /**
   * Gathers the node's children and splices them into the table.
   */
  public void expandNode(IXTreeTableNode eNode, int rowIndex){
    //Debug.noteln("ATTM: nodes before expand:", UIUtil.listToDisplay(nodes));
    if (eNode.expanded) return;
    forceExpandNode(eNode, rowIndex);
  }
  public void forceExpandNode(IXTreeTableNode eNode, int rowIndex){
    //get the node's children, set their indentation, and
    // splice them into the nodes list
    //Debug.noteln("ATTM: expanding node", eNode.toDebug());
    Enumeration kids = eNode.children();
    expanding = true;
    if (kids != null) {  
      //get the node parentLine to identify it,
      //get its IXTreeTableNode object, lookup the indentations
      //add a bit of indentation for children, add node to parents line
      //add the children to the rows just after, make children IXTreeTableNodes
      IXTreeTableNode eKid;
      LinkedList parents = new LinkedList(eNode.parentLine);
      parents.addLast(eNode.node);
      int indent = eNode.indent + INDENT_STEP;
      while (kids.hasMoreElements()) {
	Object kid = kids.nextElement();
	//Debug.noteln("ATreeTabMod: working on kid", kid.toString());
	eKid = (IXTreeTableNode)getTreeNode(kid, parents, indent);
	//add the kid to the table (last kid goes in first, so add in same indx
	addNode(eKid, rowIndex+1);
	eNode.kids.add(eKid);
      }
      //remember this node is expanded
      eNode.setExpanded(true);
      //Debug.noteln("ATTM: rows after expand:", UIUtil.listToDisplay(rows));
      fireTableDataChanged();
      fireTreeStructureChanged();
    }
    expanding = false;
  }
       
  
  /**
   * Removes the node's children from the table and notes it as unexpanded.
   */
  public void unexpandNode(IXTreeTableNode eNode, int index) {
    if (!eNode.expanded) return;
    //Debug.noteln("ATTM: rows BEFORE UNexpand:", UIUtil.listToDisplay(rows));
    removeChildren(eNode, index);
    eNode.setExpanded(false);
    //Debug.noteln("ATTM: rows after UNexpand:", UIUtil.listToDisplay(rows));
    fireTableDataChanged();
  }
  //Removes the children or each child, but keeps the nodes around
  protected void removeChildren(IXTreeTableNode eNode, int index) {
    if ((eNode == null) || !eNode.expanded) return;//no node / children showing
    //get rid of the displayed children and their displayed children.
    //Do not touch notes of which nodes are expanded
    int i = eNode.kids.size();
    if (i != 0) {
      int c = index+1;
      //remove the children of each child 
      for (Iterator kids = eNode.kids.iterator(); kids.hasNext(); c++) {
	IXTreeTableNode eKid = (IXTreeTableNode)kids.next();
	removeChildren(eKid, c);
	rows.remove(eKid);
      }

      // remove the right number of children just after the selected one
      //removeNodes(i, index+1);
      //eNode.kids.clear();
      
    }
  }
  //Removes the children or each child, does NOT keep the nodes around
  protected boolean removeChildData(IXTreeTableNode eNode) {
    if (eNode.isLeaf()) return false; //no children
    //get rid of the children and their children.
    int i = eNode.kids.size();
    if (i != 0) {
      //remove the children of each child, then the child
      for (Iterator kids = eNode.kids.iterator(); kids.hasNext(); ) {
	IXTreeTableNode eKid = (IXTreeTableNode)kids.next();
	removeChildData(eKid);
	rows.remove(eKid);
	nodes.remove(eKid);
	removeTreeNode(eKid.node);
      }
      eNode.kids.clear();
      return true;
    }
    return false;
  }

  /**
   * Produces a string that indicates the level in the tree, including initial
   * symbol.
   */
  private String toIndentedString(IXTreeTableNode eNode){
    String theValue = new String();
    //theValue = theValue + getExpansionChar(eNode) + " ";
    for (int i=0; i < eNode.indent; i++) 
      theValue = theValue + " ";
    theValue = theValue + getNameString(eNode);
    return theValue;
  }

  /**
   * Gets a character or icon to use as a reflection of the node's tree-status.
   * E.g. + for collapsed with children, - for expanded with children,
   * empty for no children.
   */
  public Object getTreeNodeType(IXTreeTableNode eNode) {
    if (ImageIcon.class.isAssignableFrom(getColumnClass(treeIconColumn))) 
      return getTypeIcon(eNode);
    else 
      return getTypeChar(eNode);
  }

  private ImageIcon getTypeIcon(IXTreeTableNode eNode) {
    ImageIcon icon = null;
    if (eNode.isLeaf()) icon = LEAFI;
    else if (eNode.expanded) icon = OPENI;
    else icon = CLOSEDI;
    return icon;
  }
  private Character getTypeChar(IXTreeTableNode eNode) {
    char e;
    if (eNode.isLeaf()) e = LEAFC;
    else if (eNode.expanded) e = OPENC;
    else e = CLOSEDC;
    return new Character(e);
  }

  /**
   * If the left button was clicked on the tree-column, expand/collapse node.
   */
  public void cellClicked(MouseEvent me, int rowIndex, int columnIndex) {
    //Debug.noteln("AbTTM: cell clicked " + rowIndex + ", " + columnIndex);
    int mod = me.getModifiers();
    //only interested in left button on tree column
    if (SwingUtilities.isLeftMouseButton(me)
	&& (columnIndex == treeIconColumn)) {
      IXTreeTableNode eNode = (IXTreeTableNode) this.rows.get(rowIndex);
      if (!eNode.isLeaf()) {
	if (eNode.expanded) unexpandNode(eNode, rowIndex);
	else expandNode(eNode, rowIndex);
      }
    }
  }

  /**
   * Gets the object's value for the given column index. The index has
   * been adjusted to ignore mark column and tree-icon column.
   * @return an object that can be rendered by the table.
   */
  public abstract Object getCellValueAt(Object node, int columnIndex);
  /**
   * Gets the string to use for the tree-column (the name of the node).
   */
  public abstract String getNameString(TreeNode node);



  /*-------------------------------------------------------------------------
   *
   *        Useful events for listeners
   *
   *-------------------------------------------------------------------------
   */
  /**
   * The given object changed - find its tree node, fire that and redo children
   */
  public void fireObjectChanged(Object o) {
    //Debug.noteln("AbTTM: got object change for", o);
    IXTreeTableNode node = (IXTreeTableNode)getTreeNode(o);
    if (node == null) return; //ignore cahnges not in model

    //Debug.noteln("  got node", node);    
    int row = super.getObjectRow(node);
    //Debug.noteln("  got row", row); 
    fireNodeStructureChanged(node);
    if (node.expanded) {
      //Debug.noteln("  sorting expanded node");    
      //unexpand, expand
      unexpandNode(node, row);
      expandNode(node, row);
    }
    else fireTableDataChanged(); //expand/unexpand will fire this too
  }


  /*-------------------------------------------------------------------------
   *
   *        Useful methods for generic tables
   *
   *-------------------------------------------------------------------------
   */
  

  /**
   * Collects usable fields from the objectClass; puts the name field
   * in front. Does not adjust for tree-icon or markings.
   */
  protected void deriveColumnFields() {
    super.deriveColumnFields();
    //now find a good field to use as a name field
    int nameIndex = UIUtil.findNameField(columnFields);
    if (nameIndex == -1) nameIndex = 0;
    String nameField = (String)columnFields.get(nameIndex);
    columnFields.remove(nameIndex);
    columnFields.add(0, nameField);
    String n = (String)columnNames.get(nameIndex);
    columnNames.remove(nameIndex);
    columnNames.add(0, nameField);
  }


  //--------------------IXTreeTableNode vs node object-------------------------

  /**
   * A map for looking up a TreeNode for a Object.
   * Use: put(Object, TreeNode) 
   */
  public HashMap nodeMap = new HashMap();

  /**
   * Get the (possibly new) TreeNode that is associlated with
   * the given Object.
   */
  public TreeNode getTreeNode(Object object) {
    if (object == null) return null;
    //Debug.noteln("ATTM: gTN/1 looking for treeNode for", object);

    TreeNode tNode = (TreeNode)nodeMap.get(object);
    if (tNode == null) {
      //Debug.noteln(" could not find it in map", nodeMap);    
      tNode = makeIXTreeTableNode(object);
      //Debug.noteln(" made new tree node gTN/1", tNode);
      addTreeNode(object, tNode);
    }
    //else Debug.noteln(" found tree node gTN/1", tNode);
    return tNode;
  }
  public TreeNode getTreeNode(Object object, List parents, int indentation) {
    if (object == null) return null;
    //Debug.noteln("ATTM: gTN/3 looking for treeNode for", object);

    IXTreeTableNode tNode = (IXTreeTableNode)nodeMap.get(object);
    //found none
    if (tNode == null) {
      //Debug.noteln(" could not find it in map", nodeMap);    
      tNode = makeIXTreeTableNode(object, parents, indentation);
      //Debug.noteln(" made new tree node gTN/3", tNode);
      addTreeNode(object, tNode);
    }
    //found one but does not match parents or indent
    else if ((((parents != null) && (!parents.equals(tNode.parentLine))) || 
	      (parents == null) && (tNode.parentLine != null)) //diff parents
	     || (indentation != tNode.indent)) { //diff indentation 
      //Debug.noteln(" found wrong one", tNode);    
      //Debug.noteln(" in map", nodeMap);    
      tNode = makeIXTreeTableNode(object, parents, indentation);
      //Debug.noteln(" made new tree node gTN/3", tNode);
      addTreeNode(object, tNode);      
    }
    //found right one!
    //else Debug.noteln(" found tree node gTN/3", tNode);
    return tNode;
  }

  /** Associate the given TreeNode to the given object */
  public void addTreeNode(Object object, TreeNode tNode) {
    IXTreeTableNode ttn = (IXTreeTableNode)tNode;
    //Debug.noteln("ATTM:***Adding node for object:",
    //	 ttn.toString() + " (" + ttn.toDebug() + ") ");
    nodeMap.put(object, tNode);
  }
  /** Forget the association of the given TreeNode to the given object */
  public void removeTreeNode(Object object) {
    if (nodeMap.containsKey(object))
      nodeMap.remove(object);
  }

  public abstract IXTreeTableNode makeIXTreeTableNode(Object node);
  /**
   * indentation is number of blanks to put in front of node (n*INDENT_STEP)
   */
  public IXTreeTableNode makeIXTreeTableNode(Object node, List parents,
					     int indentation) {
    IXTreeTableNode iNode = makeIXTreeTableNode(node);
    if (iNode != null) {
      iNode.setParents(parents);
      iNode.indent = indentation;
    }
    return iNode;
  }

  public String ixTTNListToString(List nodes) {
    String text = "";
    if (nodes == null) return text;
    Iterator i = nodes.iterator();
    while (i.hasNext()) {
      IXTreeTableNode ttn = (IXTreeTableNode)i.next();
      text = text + ttn.toString() + " (" + ttn.toDebug() + ") ";
    }
    return text.trim();

  }
}


/* ISSUES **************************************************************
 * 
 * - Not sure whether the parent-line stuff really works. Must check mappings
 * of objects to nodes.
 *
 * - This should probably implement the TreeModel interface instead of the 
 * TreeTableModelListener bundle!
 *
 */
