/****************************************************************************
 * A manager for application properties
 *
 * @author Jussi Stader
 * @version 3.1
 * Updated: Wed Jan 11 10:25:37 2006
 * Copyright: (c) 2005, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui;

import ix.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.iface.ui.util.*;

import java.io.*;
import java.util.*;
import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Component;
import java.awt.event.*;
import javax.swing.*;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

/**
 * An abstarct class for managing the properties and preferences of an
 * application. Provides lots of generically useful things for
 * property management, including running an editor to change
 * properties.
 *
 * Abstract methods are: 
 *   setDefaultProperties(Properties props)
 *   makePreferencesDialog()
 *
 * This used to be part of the application whose properties are to be
 * managed; it was moved to de-clutter the application and to make
 * mechanisms available to other applications.
 *
 * JPropertyManagers manage Properties for PropertyApplications. If
 * the Properties are user editable, they work together with
 * PropertyEditors like the JPreferenceDialogs. Here is how these
 * classes split the work:
 *
 * Properties hold all values under the keys. Provided by Java.
 *
 * PropertyManagers look after Properties, their default values,
 * loading/saving properties, and initiating updates to the
 * application to make the properties take effect, and updates from
 * the application to make sure current settings are saved. The
 * PropertyMangers also provide links between all property-related classes.
 *
 * PropertyEditors, e.g. JPreferencesDialogue, build the editor
 * window, let the user change property values (providing help where
 * available), let the user load, save, use defaults, apply, etc.
 *
 * PropertyApplications get changed by the values of the properties.
 *
 *
 *
 * {\bf How to make this work:}
 * Make your application implement the PropertiesApplication interface and 
 * write the setProperty and getProperty methods; the former sets a given
 * property in the application which should update the application, the latter 
 * determines the current setting of the property from the application. The 
 * application name is used to work out what the default property file should
 * be (./<application-name>.prefs)
 *
 * Write a specific properties manager for the application that subclasses
 * JPropertiesManager. Make this specific manager create a suitable
 * preferences editor (see below),  set default values, and the name and
 * location of property files (overwrite the getPropertiesFile or even the
 * getDefaultPropertiesFile method if the default ./<app-name>.prefs is not
 * suitable).
 *
 * Write a specific preferences editor, subclassing JPreferencesDialog
 * and specifying how each of the properties should be
 * edited. Remember to mark properties that cannot be edited in the
 * properties manager using the noteNoEditable method, otherwise such
 * preperties will be emptied when other properties are edited.
 *
 * {\bf Making changes}
 *
 * When adding a property, do the following:
 * Edit your application's properties manager to set the default value for the 
 * property; also note in the manager if the property cannot be edited.
 * Edit your application's getProperty and setProperty methods to include the
 * new property.
 * If the property is editable, add to your specific properties editor to
 * include components for editing the new property.
 *
 */
public abstract class JPropertiesManager //implements PropertyChangeListener
{
  protected PropertiesApplication application;
  protected Properties properties;
  protected JPreferencesDialog preferencesDialog;
  public HashSet noEditComponents = new HashSet();
  
  public JPropertiesManager(PropertiesApplication application) {
    this.application = application;
    //initProperties();
  }
  /**
   * Sets the default properties for the application into the given
   * properties object; application specific.  
   */
  public abstract void setDefaultProperties(Properties props);
  public abstract JPreferencesDialog makePreferencesDialog();

  /**
   * If there are properties that cannot be edited here, mark them as such.
   * Otherwise they will be emptied! (must allow user to set props to "").
   */
  public void noteNoEditable(String property) {
    noEditComponents.add(property);
  }
  public boolean isEditable(String key) {
    return !noEditComponents.contains(key);
  }
  /**
   * Specify the name of the properties file to use; 
   * set to getDefaultPropertiesFile() if you don't care.
   */
  public File getPropertiesFile() {
    return getDefaultPropertiesFile();
  }
  
  public Properties getProperties() { return properties; }
  public String getProperty(String key) { 
    return properties.getProperty(key); 
  }
  public void setProperty(String key, String value) { 
    properties.setProperty(key, value); 
  }

  public void editPreferences() {
    try { 
      ensurePreferencesDialog();
      preferencesDialog.setVisible(true);
    } catch (Exception e) {
      Debug.noteException(e);
    }
  }

  public JPreferencesDialog ensurePreferencesDialog() {
    if (preferencesDialog == null) {
      preferencesDialog = makePreferencesDialog();
      //preferencesDialog.addPropertyChangeListener(this);
      preferencesDialog.
	addPropertyChangeListener((PropertyChangeListener)application);
    }
    return preferencesDialog;
  }

  /**
   * Uses the application name to build a preferences file name in the
   * current directory.
   */
  public File getDefaultPropertiesFile() {
    String folder = ".";
    String filesep = System.getProperty("file.separator");
    String filename = folder + filesep + application.getName() + ".prefs";
    return new File(filename);    
  }


  /**
   * Sets up default properties, then loads the properties from file
   * and passes them to the application.
   */
  public void initProperties() {
    Debug.noteln("PropertyManager: loading properties");
    // create and load default properties
    Properties defaults = new Properties();
    setDefaultProperties(defaults);

    // create program properties with default
    properties = new Properties(defaults);

    loadProperties(properties);

    updateApplicationFromProperties(application);
    //Debug.noteln(" loaded properties");
  }


  public void saveProperties() {
    saveProperties(properties);
  }
  public void saveProperties(Properties properties) {
    Debug.noteln("Prop fields:", UIUtil.show(properties.propertyNames()));
    FileOutputStream out = null;
    try {
      File file = getPropertiesFile();
      out = new FileOutputStream(file);
      properties.store(out,  "--------- properties (preferences) " + 
			  " ---------");
      Debug.noteln("Wrote properties to file", file);
    } catch (java.io.FileNotFoundException e) {
      out = null;
      Debug.noteln("Warning:");
      Debug.noteln("Can't find properties file. Can't save.");
      Debug.noteln("");
    } catch (java.io.IOException e) {
      Debug.noteln("Can't save properties.");
      Debug.noteException(e);
    } finally {
      if (out != null) {
	try { out.close(); } catch (java.io.IOException e) { }
      }
    }
  }

  public Properties loadProperties() {
    loadProperties(properties);
    return properties;
  }
  public Properties loadProperties(Properties props) {
    FileInputStream in = null;
    try {
      File file = getPropertiesFile();
      Debug.noteln("Props file is", file);
      if ((file == null) || !file.exists()) {
	  Debug.noteln("Can't find properties file. Trying default file",
		       getDefaultPropertiesFile());
	  file = getDefaultPropertiesFile();
      }
      in = new FileInputStream(file);
      props.load(in);
      return props;
    } catch (java.io.FileNotFoundException e) {
      in = null;
      Debug.noteln("Warning:");
      Debug.noteln("Can't find properties file. Can't load.");
      Debug.noteln("");
      return null;
    } catch (java.io.IOException e) {
      Debug.noteln("Warning:");
      Debug.noteln("Can't read properties file. Can't load.");
      Debug.noteln("");
      return null;
    } finally {
      if (in != null) {
	try { 
	  in.close(); 
	  return props;
	} catch (java.io.IOException e) { return props; }
      }
    }
  }
  public void saveApplicationProperties() {
    updatePropertiesFromApplication();
    saveProperties(properties);
  }


  public void updateApplicationFromProperties() {
    updateApplicationFromProperties(application);
  }
  public void updateApplicationFromProperties(PropertiesApplication app) {
    Enumeration keys = properties.propertyNames();    
    while (keys.hasMoreElements() ) {
      String key = (String)keys.nextElement();
      String value = properties.getProperty(key);
      app.setProperty(key, value);
    }
  }
  public void updatePropertiesFromApplication() {
    updatePropertiesFromApplication(application);
  }
  public void updatePropertiesFromApplication(PropertiesApplication app) {
    Enumeration keys = properties.propertyNames();    
    while (keys.hasMoreElements() ) {
      String key = (String)keys.nextElement();
      String value = app.getProperty(key);
      properties.setProperty(key, value);
    }
  }

  public static boolean isBooleanProperty(String key, String value) {
    Boolean b = new Boolean(value);
    if (b.booleanValue()) return true;
    else if (b.toString().equals(value)) return true;
    else return false;
  }

}



/** Changes
 *
 */
