/****************************************************************************
 * A frame with a menu bar, a tool bar, file/about/exit dialogues.
 *
 * @author Jussi Stader
 * @version 4.0+
 * Updated: Thu Oct 26 10:32:43 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui;

import ix.*;
import ix.util.*;
import ix.iface.ui.util.*;
import ix.iface.util.*;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.io.File;
import javax.swing.filechooser.*;
//import java.net.*;

/**
 * A frame with a menu bar, a tool bar, and file/about/exit dialogues.
 *
 * Contains a method to create a default menu/tool bar and useful methods for
 * building menu and tool bars.
 * Creates an About dialog, file dialogs (open, save as), provides default 
 * functions for openFile, saveFileAs, toggleToolbar, exit, and aboutDialog.
 */
public class JFrameMT extends ToolFrame implements ActionListener
{

  CatchingActionListener catchingListener;
  /**
   * Creates a frame with a menu bar, a tool bar, a main panel.
   * Also creates required dialogs (open/save file, about, exit) and
   * handles layout.
   */
  public JFrameMT(){
    super();
    catchingListener = new CatchingActionListener(this);
    makeFrame();    
    makeMainPanel();    
  }
  /**
   * Creates a frame (see JFrameMT()) with a given title
   * @param title a string representing the frame's title
   */
  public JFrameMT(String title){
    this();	
    setTitle(title);
  }
  	
  public void makeFrame() {
    setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
    getContentPane().setLayout(new BorderLayout(0,0));
    setSize(488,309);
    setVisible(false);
  }


  //some sizes for toolbar buttons
  int TBB_Y = 4;
  int TBB_WIDTH = 51;
  int TBB_HEIGHT = 27;
  int TB_HEIGHT = TBB_HEIGHT+6;

  //{{DECLARE_CONTROLS
  // dialogs--------------
  public JFileChooser fileChooser;
  /** Te directory that the application's files live in */
  public File directory;
  /** The current file (for saving) */
  public File file;
  /** A list of file extensions to be used in the file chooser. E.g. "xml" */
  public ArrayList extensions = new ArrayList(); 
  /** A description of the file type to be used in the file chooser. 
   *  E.g. "Domain files" */
  public String fileType = "";
  
  public HelpFrame helpFrame;
  public String helpFile;
  public ImageIcon helpFrameIcon;

  // main panel for application stuff
  public javax.swing.JPanel mainJPanel = new javax.swing.JPanel();

  // toolbar--------------
  public JPanel JPanel2 = new javax.swing.JPanel();
  // menu bar-------------
  public JMenuBar menuBar = new javax.swing.JMenuBar();
  public JToolBar toolBar = new javax.swing.JToolBar();
  public JMenu fileMenu = makeMenu(menuBar, "File");
  public JMenu editMenu = makeMenu(menuBar, "Edit");
  public JMenu viewMenu = makeMenu(menuBar, "View");
  public JMenu toolsMenu = makeMenu(menuBar, "Tools");
  public JMenu helpMenu = makeMenu(menuBar, "Help");
  public JMenu optionsMenu;

  
  protected ImageIcon newIcon = Util.resourceImageIcon("ix-new.gif");
  protected ImageIcon openIcon = Util.resourceImageIcon("ix-open.gif");
  protected ImageIcon saveIcon = Util.resourceImageIcon("ix-save.gif");
  protected ImageIcon revertIcon = Util.resourceImageIcon("ix-revert.gif");
  protected ImageIcon redoIcon = Util.resourceImageIcon("ix-redo.gif");
  //protected ImageIcon cutIcon = Util.resourceImageIcon("ix-cut.gif");
  protected ImageIcon cutIcon = Util.resourceImageIcon("ix-delete.gif");
  protected ImageIcon copyIcon = Util.resourceImageIcon("ix-copy.gif");
  protected ImageIcon pasteIcon = Util.resourceImageIcon("ix-paste.gif");
  protected ImageIcon helpIcon = Util.resourceImageIcon("ix-help.gif");

  public IXToolItem newItem = 
    new IXToolItem(fileMenu, toolBar, this, "new", "New", newIcon, 'N', 
		   "Create new item");
  public IXToolItem openItem =
    new IXToolItem(fileMenu, toolBar, this, "open", "Open...", "Open", 
		   openIcon, 'O', "Open file and load");
  public IXToolItem saveItem =
    new IXToolItem(fileMenu, toolBar, this, "save", "Save", saveIcon, 'S',
		   "Save to file");
  public IXToolItem saveAsItem =
    new IXToolItem(fileMenu, this, "saveAs", "Save As...", 
		   "Save to new file");
  public IXToolItem toolbarItem;
  public IXToolItem closeItem =
    new IXToolItem(fileMenu, this, "close", "Close Window", 'C',
		   "Close window");
  public IXToolItem exitItem =
    new IXToolItem(fileMenu, this, "exit", "Exit", 'X',
		   "Exit application");


  public IXToolItem revertItem =
    new IXToolItem(editMenu, toolBar, this, "undo", "Undo", revertIcon, 'U',
		   "Undo (revert changes)");
  public IXToolItem redoItem =
    new IXToolItem(editMenu, toolBar, this,"redo", "Redo", redoIcon,'R',
		   "Redo the last undo");
  public IXToolItem cutItem =
    new IXToolItem(editMenu, toolBar, this,"cut", "Cut", cutIcon, 'C', "Cut");
  public IXToolItem copyItem =
    new IXToolItem(editMenu, toolBar, this,"copy","Copy", copyIcon,'C',
		   "Copy");
  public IXToolItem pasteItem =
    new IXToolItem(editMenu, toolBar, this,"paste","Paste", pasteIcon, 'P',
		   "Paste");
  public IXToolItem helpItem =
    new IXToolItem(helpMenu, toolBar, this,"help","Help", helpIcon,'H',"Help");
  public IXToolItem manualItem =
    new IXToolItem(helpMenu, this, "manual", "Manual", 'M', "Manual");
  public IXToolItem aboutItem =
    new IXToolItem(helpMenu, this,"about","About",'A',"About the application");
  



  private void makeMainPanel() {
    //mainJPanel.setLayout(null);   //works but won't resize components
    //mainJPanel.setLayout(new BorderLayout(0,0)); //works for domed not other
    mainJPanel.setLayout(new GridBagLayout());
    getContentPane().add(BorderLayout.CENTER, mainJPanel);
    //mainJPanel.setBounds(0,this.TB_HEIGHT,488,309-this.TB_HEIGHT);
  } 
  
  public void makeMenuToolBars(){
    setJMenuBar(menuBar);

    // toolbar
    JPanel2.setLayout(new FlowLayout(FlowLayout.LEFT,0,0));
    getContentPane().add(BorderLayout.NORTH, JPanel2);
    //JPanel2.setBounds(0,0,488,TB_HEIGHT);
    toolBar.setAlignmentY(0.222222F);
    JPanel2.add(toolBar);
    //toolBar.setBounds(0,0,395,TB_HEIGHT);
  }

  public void ensureFileChooser() {
    if (fileChooser == null) {
      if (directory == null) 
	fileChooser = new JFileChooser();
      else fileChooser = new JFileChooser(directory);
      fileChooser.setFileFilter(new FileFilter() {
	  public String getDescription() { return fileType; }
	  public boolean accept(File f) {
	    if (f.isDirectory())
	      return true;	// allow the user to visit subdirectories.
	    else return validFileName(f.getName());
	  }
	});
    }
  }

  /**
   * Checks whether the file name has an extension listed in the frame's
   * extensions.
   * @return true if the frame has no extensions listed or if the 
   *  file's extension is listed in the frame's extensions. False otherwise.
   */
  public boolean validFileName(String name) {
    if ((extensions == null) || (extensions.size() == 0)) return true;
    else {
      for (Iterator i = extensions.iterator(); i.hasNext(); ) {
	if (name.endsWith("." + i.next())) return true;
      }
      return false;
    }
  }


  /**
   * Makes a default menubar
   * Menus: 
   *   File (new, open, save, saveAs, Options menu, close, exit),
   *   Edit (revert, copy, cut, paste), 
   *   View (),
   *   Tools(),
   *   Help (help, manual, about)
   */
  public void makeDefaultMenu(){
    Debug.noteln("JFMT: making menu");
    makeFileMenu();
    makeEditMenu();
    makeViewMenu();
    makeToolsMenu();
    menuBar.add(Box.createHorizontalGlue());
    makeHelpMenu();
    adaptDefaultMenu();
  }
  public void makeFileMenu() {
    newItem.showItem();
    openItem.showItem();
    saveItem.showItem();
    saveAsItem.showItem();
    fileMenu.addSeparator();
    makeOptionsMenu();
    fileMenu.add(optionsMenu);
    fileMenu.addSeparator();
    makeFileMenuExits();
  }
  public void makeFileMenuExits() {
    closeItem.showItem();
    exitItem.showItem();
  }
  public void makeOptionsMenu() {
    optionsMenu = makeMenu(fileMenu,"Options");
    makeToolbarItem(optionsMenu);
  }

  public void makeToolbarItem(JMenu parentMenu) {
    toolbarItem = new IXToolItem(parentMenu, this, "toolbar", "Toolbar", 
				 "Toggle toolbar on/off");
    JCheckBoxMenuItem cmi = toolbarItem.makeCheckItem();
    cmi.setState(true);
    toolbarItem.showItem();    
  }

  public void makeEditMenu() {
    cutItem.showItem();
    copyItem.showItem();
    pasteItem.showItem();
  }
  public void makeViewMenu() {
  }
  public void makeToolsMenu() {
  }
  /**
   * Make sure that if this is overwritten, adaptDefaultMenu is called
   * elsewhere.
   */
  public void makeHelpMenu() {
    helpItem.showItem();
    manualItem.showItem();
    aboutItem.showItem();
  }

  public void MakeDefaultToolbar() {
    newItem.showButton();
    openItem.showButton();
    saveItem.showButton();
    toolBar.addSeparator();

    revertItem.showButton();
    redoItem.showButton();
    cutItem.showButton();
    copyItem.showButton();
    pasteItem.showButton();
    toolBar.addSeparator();

    toolBar.add(Box.createHorizontalGlue());
    helpItem.showButton();
    adaptDefaultToolbar();
  }
	
  /**
   * Makes a default menubar and toolbar.
   * Menus: File, Edit, View, Help
   */
  public void makeDefaultMenuToolbar(){
    makeFileMenuTool(menuBar);
    makeEditMenuTool(menuBar);
    makeViewMenuTool(menuBar);
    makeToolsMenuTool(menuBar);
    menuBar.add(Box.createHorizontalGlue());
    Debug.noteln("JFMT: made glue");
    makeHelpMenuTool(menuBar);
    adaptDefaultMenuToolbar();
  }

  /**
   * Dummy method that can be overridden by sub-classes to adapt the default.
   */
  public void adaptDefaultMenuToolbar(){
  }
  /**
   * Dummy method that can be overridden by sub-classes to adapt the default.
   */
  public void adaptDefaultMenu(){
  }
  /**
   * Dummy method that can be overridden by sub-classes to adapt the default.
   */
  public void adaptDefaultToolbar(){
  }



  /**
   * Makes a default file menu and toolbar buttons.
   * Menu items: new, open, save, save as, S, Options, S, close, exit
   * Toolbar buttons: new, open, save
   */
  public JMenu makeFileMenuTool(JComponent parentMenu) {
    //fileMenu = this.makeMenu(parentMenu,"File");
    makeFileMenuToolFiles(fileMenu);
    makeOptionsMenuTool(fileMenu);
    makeFileMenuToolExits(fileMenu);
    return fileMenu;
  }
  /**
   * Makes default file new/open/save/saveAs menu items and toolbar buttons.
   * Menu items: new, open, save, save as, S
   * Toolbar buttons: new, open, save
   */
  public void makeFileMenuToolFiles(JMenu fileMenu) {
    newItem.show();
    openItem.show();
    saveItem.show();
    saveAsItem.show();
    fileMenu.addSeparator();
  }
  /**
   * Makes default close/exit menu options and and toolbar buttons.
   * Menu items: S, close, exit
   * Toolbar buttons: none
   */
  public void makeFileMenuToolExits(JMenu fileMenu) {
    fileMenu.addSeparator();
    closeItem.show();
    exitItem.show();
  }
  /**
   * Makes a default options menu and toolbar buttons.
   * <nl><em>Menu items:</em> cut, copy, paste
   * <nl><em>Toolbar buttons:</em> cut, copy, paste
   */
  public JMenu makeOptionsMenuTool(JComponent parentMenu) {
    makeOptionsMenu();
    return optionsMenu;
  }
  /**
   * Makes a default edit menu and toolbar buttons.
   * <nl><em>Menu items:</em> cut, copy, paste
   * <nl><em>Toolbar buttons:</em> cut, copy, paste
   */
  public JMenu makeEditMenuTool(JComponent parentMenu) {
    //editMenu = this.makeMenu(parentMenu,"Edit");
    makeEditMenu();
    toolBar.addSeparator();
    cutItem.showButton();
    copyItem.showButton();
    pasteItem.showButton();
    return editMenu;
  }
  /**
   * Makes a default view menu and toolbar buttons.
   * <nl><em>Menu items:</em> none
   * <nl><em>Toolbar buttons:</em> none
   */
  public JMenu makeViewMenuTool(JComponent parentMenu) {
    //viewMenu = makeMenu(parentMenu,"View");
    return viewMenu;
  }
  /**
   * Makes a default tools menu and toolbar buttons.
   * <nl><em>Menu items:</em> none
   * <nl><em>Toolbar buttons:</em> none
   */
  public JMenu makeToolsMenuTool(JComponent parentMenu) {
    //toolsMenu = makeMenu(parentMenu,"Tools");
    return toolsMenu;
  }
  /**
   * Makes a default help menu and toolbar buttons.
   * <nl><em>Menu items:</em> help, manual, about
   * <nl><em>Toolbar buttons:</em> help
   */
  public JMenu makeHelpMenuTool(JComponent parentMenu) {
    makeHelpMenu();
    toolBar.addSeparator();
    helpItem.showButton();
    manualItem.showButton();
    aboutItem.showButton();
    return helpMenu;
  }

  public void placeToolbarButton(JButton button, int x) {
    button.setBounds(x, this.TBB_Y, this.TBB_WIDTH, this.TBB_HEIGHT);
  }

  /**
   * Makes a radio button menu item and adds it to the given (menu) component.
   *
   * @param menuParent the menu that the item should be added to
   * @param command attached to the item for recognising actions
   * @param label displayed in the menu
   */
  public JRadioButtonMenuItem makeRadioMenuItem(JComponent parent, 
						String command, String label){
    JRadioButtonMenuItem item = new JRadioButtonMenuItem(label);	 
    item.setActionCommand(command);
    item.addActionListener(catchingListener);
    parent.add(item);
    return item;
  }

  /**
   * Makes a menu and adds it to the given (menu) component.
   * Sets the accelerator to be the first letter of the label
   *
   * @param menuParent the menu that the item should be added to
   * @param label the name of the menu
   */
  public JMenu makeMenu(JComponent parent, String label){
    JMenu menu = new javax.swing.JMenu(label);	 
    menu.setActionCommand(label);
    menu.setMnemonic((int)label.charAt(0));
    parent.add(menu);
    return menu;
  }

  public void actionPerformed(java.awt.event.ActionEvent event) {
      Object object = event.getSource();
      String command = ((AbstractButton)object).getActionCommand();
      //Debug.noteln("action object is",object);
      Debug.noteln("JFrameMT: action command is",command);
      if (command == "open")	this.openFile();
      else if (command == "save") this.saveFile();
      else if (command == "toolbar") 
	this.toggleToolbar(((JCheckBoxMenuItem)object).getState());
      else if (command == "close") this.close();
      else if (command == "exit") this.exit();
      else if (command == "about") this.aboutDialog();
      else if (command == "saveAs") this.saveFileAs();
      //else if (command == "about") this.aboutDialog();
      else if (command.equals("manual")) this.userManual(); 
      else if (command.equals("help")) this.userHelp(); 
      
      else {
	String txt = ((AbstractButton)object).getText();
	if ((txt == "") || txt == null) txt = command;
	UIUtil.notImplemented(this, txt);
      }
  }

  public void close() {
    if (fileChooser != null) fileChooser.setVisible(false);
    if (helpFrame != null) helpFrame.setVisible(false);
    this.setVisible(false);
  }

  public void exitApplication() {
    try {
      // Beep, then show a confirmation dialog
      Toolkit.getDefaultToolkit().beep();
      int reply = JOptionPane.showConfirmDialog(this, 
						"Do you really want to exit?", 
						"JFC Application - Exit" , 
						JOptionPane.YES_NO_OPTION, 
						JOptionPane.QUESTION_MESSAGE);
      // If the confirmation was affirmative, handle exiting.
      if (reply == JOptionPane.YES_OPTION){
	this.close();    // hide the Frame
	this.dispose();            // free the system resources
	System.exit(0);            // close the application
      }
    } catch (Exception e) {}
  }

  public void openFile() {
    try {
      File openFile = userChooseFile();
      if (openFile != null) {
	file = openFile;
	openFile(file);
      }
    } catch (Exception e) { }
  }
  public File userChooseFile() {
    try {
      ensureFileChooser();
      int option = fileChooser.showOpenDialog(this);
      if (option == JFileChooser.APPROVE_OPTION) {
	File openFile = fileChooser.getSelectedFile();
	return openFile;
      }
      else return null;
    } catch (Exception e) { return null;}
  }
  
  public void saveFile() {
    try {
      if (file == null) saveFileAs();
      else saveToFile(file);
    } catch (Exception e) {}
  }
  
  public void saveFileAs() {
    try {
      ensureFileChooser();
      int option = fileChooser.showSaveDialog(this);
      if (option == JFileChooser.APPROVE_OPTION) {
	File file = fileChooser.getSelectedFile();
	saveToFile(file);
      }
    } catch (Exception e) {}
  }

  /**
   * Overwrite this with actual saving.
   */
  public void saveToFile(File file) {
    UIUtil.notImplemented(this,"Saving to file");
  }
  
  /**
   * Overwrite this with actual opening.
   */
  public void openFile(File file) {
    UIUtil.notImplemented(this,"Opening files");
  }
  
  public void toggleToolbar(boolean on){
    Rectangle bounds = mainJPanel.getBounds();
    int y,height,tbHeight;
    tbHeight = (int)this.JPanel2.getHeight();
    if (on) {
      y = tbHeight; 
      height = (int)bounds.getHeight() - tbHeight;}
    else {y = 0; height = (int)bounds.getHeight() + tbHeight;}
    JPanel2.setVisible(on);
    mainJPanel.setBounds(0,y,(int)bounds.getWidth(),height);
    mainJPanel.doLayout();
    this.validate();
  }
  
  public void exit(){
    // to do: code goes here.
    try {this.exitApplication();} catch (Exception e) {}
  }
  
  public void aboutDialog(){
    JOptionPane.showMessageDialog(this, "About " + getTitle(),
				  "About " + getTitle(),
				  JOptionPane.INFORMATION_MESSAGE);
  }

  public void userManual() {
    UIUtil.notImplemented(this,"The user manual");
  }
  
  public void userHelp() {
    ensureHelpFrame();
    helpFrame.setVisible(true);
  }

  public HTMLFrame ensureHelpFrame(){
    if (helpFrame == null){
      //helpFrame = new HelpFrame(title + "- Help", helpFile);
      helpFrame = new HelpFrame(helpFile, getTitle() + " - Help");
      helpFrame.setIconImage(helpFrameIcon.getImage());
      helpFrame.setVisible(false);
    } 
    return helpFrame;
  }




}
/*
 * Document all public methods
 */
