/****************************************************************************
 * A simple button panel with buttons of the same size.
 *
 * @author Jussi Stader
 * @version 1.0
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui;

import ix.util.Debug;
import ix.iface.ui.util.UIUtil;

import java.util.*;
import javax.swing.JComponent;
import javax.swing.AbstractButton;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.BoxLayout;
import java.awt.event.ActionListener;
import java.awt.event.MouseListener;
//import java.awt.GridLayout;
import java.awt.Dimension;

/**
 * A simple button panel with buttons of the same size.
 *
 * The button panel has an orientation (BoxLayout.X_AXIS or BoxLayout.Y_AXIS)
 * and any number of buttons whose strings are given at creation or with
 * a call to addButton. All buttons in the panel will have the same size.
 */
public class IXButPanel extends JPanel {
  public boolean groupFlag;
  public ButtonGroup group;
  public HashSet buttons = new HashSet();

  public int orientation = BoxLayout.Y_AXIS;

  /** Creates a panel for buttons with Y_AXIS layout. */
  public IXButPanel(){
    this(BoxLayout.Y_AXIS);
  }
  /**
   * Creates a panel for buttons with the given axis.
   * @param axis the layout orientation: 
   *    BoxLayout.X_AXIS or BoxLayout.Y_AXIS
   */
  public IXButPanel(int axis){
    super(); 
    if ((axis != BoxLayout.Y_AXIS)&&(axis != BoxLayout.X_AXIS)) 
	axis = BoxLayout.Y_AXIS;
    this.setLayout(new BoxLayout(this, axis));
    orientation = axis;
  }
  /**
   * Creates a panel with given buttons, given axis, adding the given
   * listener.
   *
   * @param listener a listener interested in the buttons. MouseListener and
   *   ActionListener are good.
   * @param axis the layout orientation: 
   *    BoxLayout.X_AXIS or BoxLayout.Y_AXIS
   * @param buttons a string array of button names or an array of buttons
   */
  public IXButPanel(EventListener listener, int axis, Object[] buts){
    this(axis);  
    int len;
    if (buts == null) len = 0; else len = buts.length;
    for (int i=0; i<len; i++) {
      AbstractButton button = addButton(listener,buts[i]);
    }
    invalidate();
  }
  /**
   * Creates a panel with given buttons, adding the given listener.
   * Creates the panel with the default layout, i.e. Y_AXIS.
   *
   * @param listener a listener interested in the buttons
   * @param buttons a string array of button names or an array of buttons
   */
  public IXButPanel(EventListener listener, Object[] buttons){
    this(listener, BoxLayout.Y_AXIS, buttons);
  }
    
  public int getOrientation() {
    return orientation;
  }


    /**
     * Adds a button with the given listener to this panel.
     * If the given object is a string, creates a new JButton with the string
     * as its label. If the button's actionCommand is empty, it is set to the
     * button's label.
     * Sets sensible minimum widths for the button and adds it to this panel.
     *
     * @param listener a listener interested in the buttons
     * @param but a string or an AbstractButton that will be added
     */
    public AbstractButton addButton(EventListener listener, Object but){
      AbstractButton button;
      //Debug.noteln("IXButP: making button with", but.getClass());
      if (but.getClass() == String.class) button = new JButton((String)but);
      else button = (AbstractButton) but;
      if (button.getActionCommand() == "") 
	button.setActionCommand(button.getText());
      java.awt.Dimension dim = button.getPreferredSize();
      double height = dim.getHeight();
      dim.setSize((double)IXGUIConsts.GOOD_BUT_WIDTH, height);
      button.setMinimumSize(dim);
      UIUtil.addListener(button, listener);
      //button.addMouseListener(listener);
      this.add(button);
      this.buttons.add(button);
      if (groupFlag) group.add(button);
      return button;
    }
  public void addListener(EventListener listener) {
    for (Iterator i = buttons.iterator(); i.hasNext();) {
      JComponent button = (JComponent)i.next();
      UIUtil.addListener(button, listener);
    }
  }
  public void addMouseListener(MouseListener listener) {
    if (buttons == null) return;
    for (Iterator i = buttons.iterator(); i.hasNext();) {
      AbstractButton button = (AbstractButton)i.next();
      button.addMouseListener(listener);
    }
  }
  public void addActionListener(ActionListener listener) {
    if (buttons == null) return;
    for (Iterator i = buttons.iterator(); i.hasNext();) {
      AbstractButton button = (AbstractButton)i.next();
      button.addActionListener(listener);
    }
  }
  public String getSelectionCommand() {
    if (groupFlag) return group.getSelection().getActionCommand();
    else return "";		 
  }

  public void setButtonGroup(boolean group) {
    this.groupFlag = group;
    if (group) {
      if (this.group == null) this.group = new ButtonGroup();
      fillButtonGroup();
    }
    else this.group = null;
  }
  public boolean isButtonGroup() {
    return groupFlag;
  }

  public ButtonGroup getButtonGroup() {
    return group;
  }

  protected void fillButtonGroup() {
    Iterator butts = buttons.iterator();
    while (butts.hasNext())
      group.add((AbstractButton)butts.next());
  }

  public void setEnabled(boolean isEditable) {
    Iterator butts = buttons.iterator();
    while (butts.hasNext()) {
      AbstractButton butt = (AbstractButton)butts.next();
      butt.setEnabled(isEditable);
    }
  }
}
    
