/****************************************************************************
 * A form-style panel for viewing/editing an object
 *
 * @author Jussi Stader
 * @version 3.1
 * Updated: Mon Oct  2 10:51:08 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 * 25/1/02: added simpleField stuff (collections are lists or test)
 *
 *****************************************************************************
 */
package ix.iface.ui;

import ix.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.icore.event.*;
import ix.iface.ui.util.*;
import ix.iface.ui.event.*;

import java.util.*;
//import java.awt.*;
import java.awt.Container;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.event.*;
import javax.swing.*;

/**
 * A form-style panel for viewing/editing an object
 * Each object field has a sub-panel associated with it that has a label, 
 * a panel item, and may have buttons (IXEditorPanel).
 *
 */
public class IFormPanel extends JPanel 
implements MouseListener, FormModelListener {
  protected IFormModel model;
  protected HashMap nameBits = new HashMap();
  protected HashMap displayBits = new HashMap();
  protected HashMap fieldRenderers = new HashMap();
  protected HashSet simpleFields = new HashSet();
  protected HashSet listeners = new HashSet();
  protected Container parent;
  
  /** Creates a panel with a GridBagLayout and sets renderers (if any). */
  public IFormPanel() {
    super(new GridBagLayout());
    setRenderers();
  }
  /** Creates a panel with the given model and sets renderers (if any). */
  public IFormPanel(IFormModel m) {
    super(new GridBagLayout());
    this.setModel(m);
    setRenderers();
  }

  /** Creates a panel with the given model and sets renderers (if any). */
  public IFormPanel(IFormModel m, FormDataListener l) {
    super(new GridBagLayout());
    this.setModel(m, l);
    setRenderers();
  }


  /**
   * Sets the panel's model and loads the data into the panel.
   * Ensures that the panel items are available for the object information,
   * and sets the renderers. Ensures this is notified of model changes.
   */
  public void setModel(IFormModel m, FormDataListener l) {
    setModel(m);
    model.addDataListener(l);
  }

  /**
   * Sets the panel's model and loads the data into the panel.
   * Ensures that the panel items are available for the object information,
   * and sets the renderers. Ensures this is notified of model changes.
   */
  public void setModel(IFormModel m) {
    //Debug.noteln("IFP: setting model");
    //if ((m != null) && m.equals(model)) displayModelData();
    //else {
    IFormModel oldModel = model;
    model = m;
    if ((oldModel != null) && (model != null)) { //transfer listeners
      Iterator e = oldModel.getModelListeners().iterator();
      while (e.hasNext()) {
	model.addModelListener((FormModelListener)e.next());
      }
      e = oldModel.getDataListeners().iterator();
      while (e.hasNext()) {
	model.addDataListener((FormDataListener)e.next());
      }
    }
    this.ensurePanelBits(m);
    this.displayModelData();
    m.addModelListener(this);
    //}
  }
  /** Gets the panel's model. */
  public IFormModel getModel() { return model; }


  /**
   * Sets the given renderer for the given field.
   * If the field has an IXEditorPanel bit, sets the renderer for the panel.
   */
  public void setRenderer(String field, IXRenderer r) {
    //Debug.noteln("Setting renderer for field " + field);
    fieldRenderers.put(field, r);
    IXEditorPanel bit = (IXEditorPanel)getFieldBit(field);
    if (bit != null) {
      //Debug.noteln("Setting renderer for bit ");
      bit.setRenderer(r);
    }
  }

  public boolean isListField(String field) {
    Class fieldClass = model.getFieldClass(field);
    return List.class.isAssignableFrom(fieldClass);
  }
  public void setSimpleField(String field) {
    simpleFields.add(new String(field));
  }
  public void unsetSimpleField(String field) {
    simpleFields.remove(new String(field));
    //Debug.noteln("cut field " + field + " now: " + simpleFields.toString());
  }
  public boolean isSimpleField(String field) {
    return simpleFields.contains(new String(field));
  }

  /**
   * Ensures that the panel bits for the given model are made.
   * If the model has lazy fields, these are filled now, so ignore model 
   * updates during this method.
   */
  public void ensurePanelBits(IFormModel m) {
    if (m == null) return;
    else {
      model = m;
      m.startUpdate();
      Dimension size = getSize();
      //Debug.noteln("Panel size is " + size.toString());
      if (!checkNameBits()) this.makeNameBits();
      if (!checkDisplayBits()) this.makeDisplayBits();
      setSize(size);
      m.ignoreUpdate();
    }
  }

  /* Checks whether the model's name fields have panel bits */
  private boolean checkNameBits() {
    return checkBits(model.getNameFields(), nameBits);
  }			  
  /* Checks whether the model's display fields have panel bits */
  private boolean checkDisplayBits() {
    return checkBits(model.getDisplayFields(), displayBits);
  }			  
  /* Checks whether the given list of field names have bits in the given map 
  * and check that there are no superfluous ones.
  */
  private boolean checkBits(List fields, HashMap bits) {
    //both empty or null - ok
    if ((fields == null) && (bits == null)) return true;
    if ((fields == null) && (bits.size() == 0)) return true;
    if ((fields.size() == 0) && (bits == null)) return true;
    //sizes don't match
    if ((fields == null) || (bits == null)) return false;
    if (bits.size() != fields.size()) return false;

    //check all are there
    Iterator i = fields.iterator();
    while (i.hasNext()) {
      if (!bits.containsKey((String)i.next())) return false;
    }
    return true;
  }			  

  /**
   * Makes the name bits.
   * Gets the model's name fields, makes a ThingEditorPanel bit for each,
   * and adds the bit to its own panel, and sets the bit's renderer if there
   * is one.
   */
  private void makeNameBits() {
    clearBits(nameBits);
    List fields = model.getNameFields();
    if (fields.isEmpty()) return;
    else {
      int count = 0;
      Iterator i = fields.iterator();
      while (i.hasNext()) {
	String field = (String) i.next();
	makeNameBit(field, count);
	count++;
      }
    }
  }


  private void makeNameBit(final String field, int count) {
    JTextField jtf = new JTextField();
    IXEditorPanel bit = new ThingEditorPanel(this, 
					     model.getFieldName(field), jtf);
    jtf.addFocusListener(new FocusAdapter() {
	public void focusLost(FocusEvent fEvent) {
	  //if (!fEvent.isTemporary())  //only listen if focus within window
	    saveFieldData(field);
	};
      });
    nameBits.put(field, bit);
    addNameBit(field, (JPanel)bit, count);
  }

  /**
   * Makes the display bits.
   * Gets the model's display fields, makes a IXEditorPanel bit with a suitable
   * component for each field, and adds the bit to its own panel, and sets
   * the bit's renderer if there is one.
   */
  private void makeDisplayBits() {
    clearBits(displayBits);
    int width = model.getNameFields().size();
    List fields = model.getDisplayFields();
    if (fields.isEmpty()) return;
    else {
      //Debug.noteln("IFP: Making display bits",UIUtil.listToDisplay(fields));
      IXEditorPanel bit;
      int count = 1; //start with 1 to leave space for names...!
      Iterator i = fields.iterator();
      while (i.hasNext()) {
	String field = (String)i.next();
	//Debug.noteln("IFormPanel: Doing field " + field);
	//Debug.noteln(" class: " + model.getFieldClass(field).getName());
	bit = makeDisplayBit(field);
	IXRenderer r = (IXRenderer)fieldRenderers.get(field);
	if (r != null) {
	  //Debug.noteln("Setting renderer in makeDisplayBits");
	  bit.setRenderer(r); 
	}
	displayBits.put(field, bit);
	addDisplayBit(field, (JPanel)bit, count, width);
	count++;
      }
    }
  }
  /** If the given field has a renderer, it is set in the given bit. */
  private void addRenderer(String field, IXEditorPanel bit) {
    IXRenderer r = (IXRenderer)fieldRenderers.get(field);
    if (r != null) {
      //Debug.noteln("Setting renderer in makeDisplayBits");
      bit.setRenderer(r); 
    }
  }

  protected static void clearBits(HashMap bits) {
    Collection c = bits.values();
    if (c.isEmpty()) return;
    else {
      Iterator i = c.iterator();
      while (i.hasNext()) ((JComponent)i.next()).setVisible(false);
      bits.clear();
    }
  }


  /**
   * Adds the given bit of the given field to the name part of this panel in
   * the given position. Name bits all go in one row at the top of the panel.
   * Looks up the field's relative weight in the model to set x-portion.
   * @param field the field whose bit is added.
   * @param bit the bit (IXEditorPanel) that is added.
   * @param count the position of the bit in the name row.
   */
  private void addNameBit(String field, JComponent bit, int count) {
    Insets insets = new Insets(4,0,4,0);
    double weight = model.getFieldWeight(field);
    GridBagConstraints gbc = 
      new GridBagConstraints(count,0,1,1,weight,0.0,
        GridBagConstraints.NORTH,GridBagConstraints.HORIZONTAL,insets,0,0);
    this.add(bit,gbc);
  }
  /**
   * Adds the given bit of the given field to the display part of this panel
   * in the given position spanning the given number of columns. Display bits
   * each take up the whole width of the panel. 
   * Looks up the field's relative weight in the model to set y-portion.
   * @param field the field whose bit is added.
   * @param bit the bit (IXEditorPanel) that is added.
   * @param count the position of the bit in the display stack.
   * @param w the number of columns in the panel (i.e. the number of name bits)
   */
  private void addDisplayBit(String field, JComponent bit, int count, int w) {
    Insets insets = new Insets(4,0,4,0);
    double weight = model.getFieldWeight(field);
    GridBagConstraints gbc = 
      new GridBagConstraints(0,count,w,1,1.0,weight,
        GridBagConstraints.WEST,GridBagConstraints.BOTH,insets,0,0);
    this.add(bit,gbc);
  }

  /**
   * Makes a display panel with the name as its label and a suitable item.
   * The item to display the fields will be:
   * <ul>
   * <li> Number/Symbol/Boolean - a JTextField 
   * <li> Array/Collection/etc - a JList in a scroll panel with buttons
   * <li> Other - a JTextArea in a scroll panel
   * </ul>
   */
  protected IXEditorPanel makeDisplayBit(final String field) {
    String fieldName = model.getFieldName(field);
    Class fieldClass = model.getFieldClass(field);
    IXEditorPanel bit;
    //Debug.note("makeDisplayBit " + fieldName);
    //numbers - text field
    if ((Number.class.isAssignableFrom(fieldClass)) 
	|| (Symbol.class.isAssignableFrom(fieldClass))
	|| (Boolean.class.isAssignableFrom(fieldClass))) {
      JTextField jtf = new JTextField();
      bit = new ThingEditorPanel(this, fieldName, jtf);
      jtf.addFocusListener(new FocusAdapter() {
	  public void focusLost(FocusEvent fEvent) {
	    //Debug.noteln("may save data for field", field);
	    //Debug.noteln(" with event", fEvent);
	    //if (!fEvent.isTemporary()) { //only listen if focus within window
	      //Debug.noteln("  saving");
	      saveFieldData(field);
	      //}
	  };
	});
    }
    //arrays, collections - list in scroll panel
    else if (((Collection.class.isAssignableFrom(fieldClass))
	      ||(java.lang.reflect.Array.class.isAssignableFrom(fieldClass)))
	     && !isSimpleField(field)) { 
      String[] buttons = {"Add", "Edit", "Delete"};
      JList list = new JList();
      list.setVisibleRowCount(3);
      //list.setFixedCellHeight(list.getFont().getSize() + 10);
      JScrollPane jsp = new JScrollPane();
      jsp.getViewport().add(list);
      bit = new ListEditorPanel(this,fieldName,jsp,buttons);
    }
    //default - text area in scroll panel
    else {
      JScrollPane jsp = new JScrollPane();
      //jsp.setPreferredSize(new Dimension(400, 30));
      JTextArea jta = new JTextArea();
      jta.addFocusListener(new FocusAdapter() {
	  public void focusLost(FocusEvent fEvent) {
	    //if (!fEvent.isTemporary()) //only listen if focus within window
	      saveFieldData(field);
	  };
	});
      jsp.getViewport().add(jta);
      bit = new ThingEditorPanel(this,fieldName,jsp);
    }
    bit.setEnabled(model.isEditable(field));
    return bit;
  }


  /**
   * Loads the data of the current model into the panel.
   * Looks up the fields in the model.
   */
  public void displayModelData() {
    if (model == null) return;
    else {
      //Debug.noteln("IFormPanel: clearing model data");
      clearFields();
      if (model.getObject() == null) return;
      else {
	//Debug.noteln("IFormPanel: setting model fields");
	Iterator fields = getAllFields();
	while (fields.hasNext()) 
	  displayFieldData((String)fields.next());
      }
    }
  }
  /**XXXX
   * Loads the data of the given field into the panel.
   * Gets the value from the model, renders it, and sets the panel bit.
   */
  public void displayFieldData(String field) {
    //Debug.noteln("IFP: displaying field", field);
    clearField(field);
    Object value = model.getValue(field);
    //Debug.noteln("IFP: got value", value);
    //Debug.noteln(" for field", field);
    //(new Throwable()).printStackTrace();
    if (value == null) return;
    else 
      setFieldData(field, value);
  }

  /**
   * Sets the given data of the given field in the panel.
   * Renders the data first.
   */
  public void setFieldData(String field, Object value) {
    //Debug.noteln("IFP: Setting field data for ", field);
    IXEditorPanel bit = (IXEditorPanel)getFieldBit(field);
    if (bit != null) {
      //Debug.noteln(" Bit item is of class", 
      //	 bit.getItemComponent().getClass().getName());
      Object newValue = renderValue(field, value);
      bit.setData(newValue);
      //((JComponent)bit).revalidate();
      ((JComponent)bit).paintImmediately(((JComponent)bit).getBounds());
      //Debug.noteln("IFP: painted field", field);
    }
  }

  /**
   * Clears the panel bits of all fields.
   */
  public void clearFields() {
    Iterator fields = getAllFields();
    while (fields.hasNext()) {
      clearField((String)fields.next());
    }
  } 
  /**
   * Clears the panel bit of the given field.
   */
  public void clearField(String field) {
    //Debug.noteln("IFormPanel: clearing field " + field);
    IXEditorPanel panel = getFieldBit(field);
    if (panel == null) {
      //Debug.noteln("Cannot find panel for field " + field);
      //(new Throwable()).printStackTrace();    
      return;
    }
    //Debug.noteln("component type: " + panel.getItemComponent().getClass());
    panel.removeData();    
  }

  /**
   * Default rendering of the given value for the given field -
   * transforms collections to arrays. Used to provide data to the panel.
   */
  protected Object renderValue(String field, Object value) {
    if (value == null) return null;
    else if (value instanceof AbstractCollection)
      return ((AbstractCollection)value).toArray();
    else return value;
  }

  /**
   * Default de-rendering of the given value for the given field -
   * transforms arrays to collections. Used to store data from the panel.
   */
  protected Object deRenderValue(String field, Object data) {
    //Debug.noteln("IFormPanel: deRenderValue doing field " + field);
    if (data == null) {
      if (isListField(field)) return Lisp.NIL;
      else return null;
    }
    Class vc = model.getFieldClass(field);
    //Debug.noteln("IFP: de-render to class", vc.getName());
    Class dc = data.getClass();
    //Debug.noteln(" data class", dc.getName());
    if (vc.isAssignableFrom(dc)) return data;
    else if (dc.isArray()) {
      //Debug.noteln(" got array data");
      try {
	if (TypedList.class.isAssignableFrom(vc)) {
	  return UIUtil.newTypedList(vc, data);
	}
	else if (Collection.class.isAssignableFrom(vc)) {
	  Collection value = (Collection)vc.newInstance();
	  value.addAll(Arrays.asList((Object[])data));
	  return value;
	}
      }
      catch (Exception e) {
	Debug.noteln("deRenderValue: caught problem with field " + field);
	Debug.noteln("Data is of type " + data.getClass());
	Debug.noteException(e);
	return data;
      }
    }      
    return data;
  }

  /** 
   * Sets the renderers to be added to the panel items (lists) of fiels.
   * To be overridden by sub-classes.
   */
  protected void setRenderers() {}
			  
  public IXEditorPanel getFieldBit(String field) {
    IXEditorPanel bit = (IXEditorPanel) nameBits.get(field);
    if (bit == null)
      bit = (IXEditorPanel) displayBits.get(field);
    return bit;
  }

  /**
   * Gets the field's value from its display component.
   */
  public Object getFieldData(String field) {
    IXEditorPanel bit = getFieldBit(field);
    if (bit == null) return null;
    else {
      return bit.getData();
    }
  }

  /**
   * Saves the field's value from its display component into the model's
   * object.
   */
  public void saveFieldData(String field) {
    //Debug.noteln("IFP: saving data for field", field);
    Object data = getFieldData(field);
    Object value = deRenderValue(field, data);
    model.setValue(field, value);
  }

  /**
   * Saves the data from the display panels into the model's object.
   */
  public void saveToObject() {
    EditableObject object = model.getObject();
    List allFields = new LListCollector(model.getNameFields());
    allFields.addAll(model.getDisplayFields());
    Iterator i = allFields.iterator();
    while (i.hasNext()) {
      String field = (String)i.next();
      Object data = getFieldData(field);
      Object value = deRenderValue(field, data);
      model.setValue(field, value);
    }
  }
  /**
   * Saves the display information from the display panels into the given
   * object.
   */
  public void saveToObject(EditableObject targetObject) {
    //quietly set the target object to avoid updating the display
    List allFields = new LListCollector(model.getNameFields());
    allFields.addAll(model.getDisplayFields());
    Iterator i = allFields.iterator();
    while (i.hasNext()) {
      String field = (String)i.next();
      //Debug.noteln("IFP: saving field", field);
      Object data = getFieldData(field);
      //if (data != null) Debug.noteln(" data class", data.getClass());
      Object value = deRenderValue(field, data);
      //if (value != null) Debug.noteln(" deRendered class", value.getClass());
      model.setValue(targetObject, field, value);
    }
  }

  /**
   * Gets all fields in the model - name fields first.
   */
  private Iterator getAllFields() {
    List allFields = new LListCollector(model.getNameFields());
    allFields.addAll(model.getDisplayFields());
    return allFields.iterator();
  }

  /** Clears the data in the panel. */
  public void clear() {
    model.clearData();
  }

  /*-------------------------------------------------------------------------
   * reflect model changes
   *-------------------------------------------------------------------------
   */

  /**
   * The model changed, so update the display.
   * <ul>
   * <li> If the object changed, just update the display.
   * <li> If the field names changed, update the labels of bits.
   * <li> Otherwise, make a whole new panel.
   * </ul>
   */
  public void formModelChanged(String change, ObjectChangeEvent oce) {
    //Debug.noteln("IFP: got a model change: " + change);
    if (change.equals("object") || change.equals("allFields")) {
      //Debug.noteln("IFP: doing object change");
      displayModelData();
    }
    else if (change.equals("fieldNames")) {
      //for each display bit, set its label to the new name
      List fields = model.getNameFields();
      updateNames(fields, nameBits);
      fields = model.getDisplayFields();
      updateNames(fields, displayBits);
    }
    else if (change.equals("Field")) {
      String field = oce.getField();
      //Debug.noteln("IFP: got data change for field", field);
      //clearField(field);
      displayFieldData(field);
    }
    else { //e.g. "nameFields", "displayFields"
      //make a whole new panel
      model.startUpdate();
      this.setModel(model);
      model.ignoreUpdate();
    }
  }
			  
  /**
   * Change the labels in the given panels using the given fields.
   */
  public void updateNames(List fields, HashMap bits) {
    if ((fields == null) || (fields.isEmpty())) return;
    Iterator i = fields.iterator();
    while (i.hasNext()) {
      String field = (String)i.next();
      String label = model.getFieldName(field);
      if (label != null) {
	IXEditorPanel bit = (IXEditorPanel)bits.get(field);
	if (bit == null) Debug.noteln("Cannot find bit for " + label);
	else bit.setLabel(label);
      }
    }
  }


  /*-------------------------------------------------------------------------
   * Relay mouse events to listeners
   *-------------------------------------------------------------------------
   */

  public void mouseClicked(MouseEvent me) {
    //Debug.noteln("IFP: got mouse clicked");
    if (listeners.isEmpty()) return;
    else {      
      for (Iterator i = listeners.iterator(); i.hasNext(); ) {
	MouseListener l = (MouseListener)i.next();
	l.mouseClicked(me);
      }
    }
  }
  public void mouseEntered(MouseEvent me) {
    if (listeners.isEmpty()) return;
    else {
      for (Iterator i = listeners.iterator(); i.hasNext(); ) {
	MouseListener l = (MouseListener)i.next();
	l.mouseEntered(me);
      }
    }
 }
  public void mouseExited(MouseEvent me) {
    if (listeners.isEmpty()) return;
    else {
      for (Iterator i = listeners.iterator(); i.hasNext(); ) {
	MouseListener l = (MouseListener)i.next();
	l.mouseExited(me);
      }
    }
 }
  public void mousePressed(MouseEvent me) {
    if (listeners.isEmpty()) return;
    else {
      for (Iterator i = listeners.iterator(); i.hasNext(); ) {
	MouseListener l = (MouseListener)i.next();
	l.mousePressed(me);
      }
    }
 }
  public void mouseReleased(MouseEvent me) {
    if (listeners.isEmpty()) return;
    else {
      for (Iterator i = listeners.iterator(); i.hasNext(); ) {
	MouseListener l = (MouseListener)i.next();
	l.mouseReleased(me);
      }
    }
 }

  public void addFormMouseListener(MouseListener ml) {
    if (ml != null) listeners.add(ml);
  }
}
