/****************************************************************************
 * A panel with a combo-box and a card panel that can display other panels
 *
 * @author Jussi Stader
 * @version 1.0
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

/**
 * A panel with a combo-box and a card panel that can display other panels.
 *
 * Example of how to use a CardPanel:
 *  <PRE><code>
 * cardsPanel = new CardPanel();  
 * ...
 * cardsPanel.addPanel(panel1, name1);
 * cardsPanel.addPanel(panel2, name2);
 * cardsPanel.addPanel(panel3, name3);
 *
 * cardsPanel.setPanel(name2);
 *
 * </code> </PRE>
 * 
 * The CardPanel will look after selections made in the combo-box, i.e. it
 * will display the selected panel. However,
 * If changing panel should have side-effects (e.g. updating menus, enabling 
 * options, make the interested component implement an ItemListener, add it
 * to the CardPanel using 
 * <code> cardsPanel.addItemListener(theListener);</code>
 * and provide a method like this:
 * <PRE><code>
 *  public void itemStateChanged(ItemEvent event) {
 *    if (cardsPanel.isAdjusting) return;
 *    String panelName = (String)event.getItem();
 *    this.updateCurrentPanel(panelName);  
 *  }  
 * </code> </PRE>
 */
public class CardPanel extends JPanel implements ItemListener {
  JPanel cardsPanel;
  JComboBox panelNamesCB;
  public boolean isAdjusting = false;

    /** Creates a panel for a combo box and a card panel. */
  public CardPanel(){
      super(new BorderLayout());
      panelNamesCB = new JComboBox(new DefaultComboBoxModel());
      panelNamesCB.setEditable(false);
      panelNamesCB.addItemListener(this);
      this.add(panelNamesCB, BorderLayout.NORTH);
      
      cardsPanel = new JPanel(new CardLayout());
      this.add(cardsPanel, BorderLayout.CENTER);
    }
  public CardPanel(ItemListener listener){
    this();
    panelNamesCB.addItemListener(listener);
  }

  public JPanel getCardsPanel() {
    return cardsPanel;
  }

  public void addItemListener(ItemListener listener){
    panelNamesCB.addItemListener(listener);
  }

  public void addPanel(JPanel panel) {
    this.addPanel(panel, panel.getName());
  }

  public void addPanel(JPanel panel, String panelName) {
    cardsPanel.add(panel, panelName);
    this.isAdjusting = true;
    panelNamesCB.setEnabled(false);
    panelNamesCB.addItem(panelName);
    panelNamesCB.setEnabled(true);
    this.isAdjusting = false;
  }

  public void removePanel(JPanel panel, String panelName) {
    this.isAdjusting = true;
    cardsPanel.remove(panel);
    if (panelNamesCB.getItemCount() != 0)
      panelNamesCB.removeItem(panelName);    
    this.isAdjusting = false;
  }

  public void removePanel(JPanel panel) {
    removePanel(panel, panel.getName());
  }

  public void removeAllPanels() {
    this.isAdjusting = true;
    cardsPanel.removeAll();
    if (panelNamesCB.getItemCount() != 0) panelNamesCB.removeAllItems();
    this.isAdjusting = false;
  }

  public void setPanel(String panelName) {
    CardLayout cl = (CardLayout)(cardsPanel.getLayout());
    cl.show(cardsPanel, panelName);
    this.isAdjusting = true;
    panelNamesCB.setSelectedItem(panelName);
    this.isAdjusting = false;
  }


  public void itemStateChanged(ItemEvent event) {
    if (isAdjusting) return;
    CardLayout cl = (CardLayout)(cardsPanel.getLayout());
    cl.show(cardsPanel, (String)event.getItem());
  }
}
    
