/****************************************************************************
 * An interface model to be used to fill an IFormPanel.
 *
 * @author Jussi Stader
 * @version 4.0+
 * Updated: Mon Oct  2 10:50:24 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui;

import ix.*;
import ix.util.Debug;
import ix.util.lisp.*;
import ix.icore.event.*;
import ix.iface.ui.util.*;
import ix.iface.ui.event.*;

import java.util.*;
import java.lang.reflect.*;
//import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

/**
 * An abstract model class to be used to fill an IFormPanel.
 *
 */
public abstract class AbstractIFormModel 
  implements IFormModel, DataChangeListener 
{
  protected List nameFields = Lisp.NIL;
  protected List displayFields = Lisp.NIL;
  //a map, (<fieldName>, <displayName>), usually ("field", "Field")
  protected HashMap fieldNames = new HashMap();
  protected HashMap fieldClasses = new HashMap();
  protected EditableObject baseObject;
  protected Class baseObjectClass;
  protected HashSet listeners = new HashSet();
  protected HashSet dataListeners = new HashSet();
  protected boolean isUpdating = false;
  protected boolean settingData = false;

  public AbstractIFormModel(Class objectClass) {
    super();
    baseObjectClass = objectClass;
    setFields();
  }
  public AbstractIFormModel(EditableObject object) {
    this(object.getClass());
    baseObject = object;
    object.addDataChangeListener(this);
  }

  public void clearData() {
    setObject(null);
  }


  //---------------------- Setting up -----------------------------------------

  public String getFieldName(String field) {
    return (String) fieldNames.get(field);
  }
  public Class getFieldClass(String field) {
    return (Class) fieldClasses.get(field);
  }
  public double getFieldWeight(String field) {
    return 1.0;
  }


  public void setNameFields(List fields) {
    nameFields = fields;
    fieldNames.clear();
    fieldClasses.clear();
    deriveNames();
    deriveClasses();
    fireModelChanged("nameFields");
  }
  public void setNameFields(String[] fields) {
    List list = new LListCollector(Arrays.asList(fields));
    setNameFields(list);
  }
  public void setDisplayFields(List fields) {
    displayFields = fields;
    fieldNames.clear();
    fieldClasses.clear();
    deriveNames();
    deriveClasses();
    fireModelChanged("displayFields");
  }
  public void setDisplayFields(String[] fields) {
    List list = new LListCollector(Arrays.asList(fields));
    setDisplayFields(list);
  }

  /**
   * Get the fields that correspond to the object's name (e.g. "name", "id")
   */
  public List getNameFields() {
    return nameFields;
  }

  /**
   * Get the titles to use for the object's name fields (e.g. "Name", "Id").
   * Derive them if they are not there.
   */
  public List getNameNames() {
    return getFieldNames(nameFields);
  }

  /**
   * Get the classes of the object's name fields 
   * (e.g. String.class, Object.class)
   */
  public List getNameClasses() {
    return getFieldClasses(nameFields);
  }
  /**
   * Get the fields of the object that are to be displayed (exc. names)
   */
  public List getDisplayFields() {
    return displayFields;
  }

  /**
   * Get the titles to use for the object's display fields
   */
  public List getDisplayNames() {
    return getFieldNames(displayFields);
  }
  /**
   * Get the classes of the object's display fields 
   * (e.g. String.class, Object.class, Collection.class)
   */
  public List getDisplayClasses() {
    return getFieldClasses(displayFields);
  }

  /**
   * Set the class of the model's base object
   */
  public void setObjectClass(Class c) {
    if (baseObjectClass.equals(c)) return;
    if (baseObjectClass != null) clearFields();
    baseObjectClass = c;
    setFields(); //fires "allFields"
    //fireModelChanged("objectClass");
  }
  /**
   * Get the class of the model's base object
   */
  public Class getObjectClass() {
    if (baseObjectClass == null) 
      if (baseObject != null) 
	baseObjectClass = baseObject.getClass();
    return baseObjectClass;
  }



  //---------------------- Running -----------------------------------------

  public void setObject(EditableObject o) {
    EditableObject oldObject = baseObject;
    boolean changed = false;
    //Debug.note("form model: setting object");
    //Debug.noteln("Object class is " + baseObjectClass.getName());
    if ((o != null) && (!(o.getClass().equals(baseObjectClass)))) {
      setObjectClass(o.getClass());
      changed = true;
    }
    if (!((o == null) && (baseObject == null)) &&
	!((o != null) && o.equals(baseObject)))
      changed = true;
    baseObject = o;
    if (oldObject != null) oldObject.removeDataChangeListener(this);
    if (o != null) o.addDataChangeListener(this);
    if (changed) fireModelChanged("object"); //real change (different object)
    else if (baseObject != null)
      fireModelChanged("allFields"); //just in case there were any changes
  }

  public EditableObject getObject() {
    return baseObject;
  }


  /**
   * Get the current object's value for the given field.
   */
  public Object getValue(String field) {
    return UIUtil.getObjectFieldValue(baseObject, field);
  }
  public Object getValue(EditableObject object, String field) {
    return UIUtil.getObjectFieldValue(object, field);
  }
  

  public void setValue(String field, Object value) {
    settingData = true;
    Object oldValue = getValue(baseObject, field);
    if (baseObject == null) {
      Debug.noteln("AIFormModel: Trying to set value for null object");
      return;
    }
    if (baseObject.sameValue(field, oldValue, value)) return; //---->
    //Debug.noteln("AIFormModel: setting value for field", field);
    if (setValue(baseObject, field, value)) {
      fireDataChanged(field, baseObject, oldValue, value);
    }
    settingData = false;
  }
  /**
   * Get the object to change its value. 
   * @return true if the value was changed, false if it was not.
   */
  public boolean setValue(EditableObject object, String field, Object value) {
    return object.setValue(field, value);
    //Object oldValue = getValue(object, field);
    //Class fClass = getFieldClass(field);
    //UIUtil.setObjectFieldValue(object, field, fClass, value);
  }


  public Field stringToField(String field) throws 
                              NoSuchFieldException, IllegalAccessException {
    return UIUtil.stringToField(getObjectClass(), field);
  }



  /**
   * Derives the name fields, display fields and the field names and classes.
   */
  public void setFields() {
    if (baseObjectClass == null) {
      Debug.noteln("Cannot set fields for a null base object class.");
      return;
    }
    else {
      isUpdating = true;
      if (nameFields.isEmpty()) nameFields = this.deriveNameFields();
      if (displayFields.isEmpty()) displayFields = this.deriveDisplayFields();
      List allFields = new LListCollector(nameFields);
      allFields.addAll(displayFields);
      if (fieldNames.isEmpty()) deriveNames(allFields);
      if (fieldClasses.isEmpty()) deriveClasses(allFields);
      isUpdating = false;
      fireModelChanged("allFields");
    }
  }

  private void clearFields() {
    nameFields = Lisp.NIL;
    displayFields = Lisp.NIL;
    fieldNames.clear();
    fieldClasses.clear();
    if (baseObject != null) baseObject.removeDataChangeListener(this);
    baseObject = null;
  }


  protected List lookupFieldNames(List fields) {
    return lookupFieldMaps(fields, fieldNames);
  }
  protected List lookupFieldClasses(List fields) {
    return lookupFieldMaps(fields, fieldClasses);
  }

  public boolean isFieldShowing(String field) {
    return fieldNames.containsKey(field);
  }

  /**
   * Get the mappings of the given fields from the given list. 
   * Do not try to fill the map if it is not already there.
   */
  private static List lookupFieldMaps(List fields, HashMap map) {
    if (fields.isEmpty()) return Lisp.NIL;
    else {
      LListCollector values = new LListCollector();
      Iterator i = fields.iterator();
      while (i.hasNext()) {
	String field = (String)i.next();
	values.addElement(map.get(field));
      }
      return values;    
    }

  }


  /**
   * Get or derive the names of the given fields 
   */
  private List getFieldNames(List fields) {
    return lookupFieldNames(fields);
  }
  /**
   * Get or derive the classes of the given fields 
   */
  private List getFieldClasses(List fields) {
    return lookupFieldClasses(fields);
  }

  /** Collects name and display fields and calls deriveNames(fields). */
  protected void deriveNames() {
    List allFields = new LListCollector(nameFields);
    allFields.addAll(displayFields);
    deriveNames(allFields);
  }
  /** Collects name and display fields and calls deriveClasses(fields). */
  protected void deriveClasses() {
    List allFields = new LListCollector(nameFields);
    allFields.addAll(displayFields);
    deriveClasses(allFields);
  }

  /**
   * Derives the names of the given fields and puts them into
   * HashMap fieldNames.
   */
  abstract protected void deriveNames(List fields);

  /**
   * Derives the classes of the given fields and puts them into
   * HashMap fieldClasses.
   */
  abstract protected void deriveClasses(List fields);

  /**
   * Derives the fields to be used for displaying the object name(s)
   */
  abstract protected List deriveNameFields();

  /**
   * Derives the fields to be used for displaying the object information
   */
  abstract protected List deriveDisplayFields();

  public static Collection addIfField(Collection fields, 
				      Class c, String sField) {
    try {
      Field field = c.getField(sField);
      fields.add(sField);
      return fields;
    }
    catch (NoSuchFieldException fe) { return fields; }
  }

  public boolean isEditable(String field) {
    return true;
  }


  public void startUpdate() {
    isUpdating = true;
  }
  public void endUpdate() {
    isUpdating = false;
    fireModelChanged("allFields");
  }
  /** End update without firing */
  public void ignoreUpdate() {
    isUpdating = false;
  }


  public void fireModelChanged(String change, ObjectChangeEvent oce) {
    if (isUpdating) return; //doing my own updates, so ignore this
    //Debug.noteln("AbIFM: model change", change);
    for (Iterator e = listeners.iterator(); e.hasNext(); ) {
      FormModelListener l = (FormModelListener)e.next();
      l.formModelChanged(change, oce);
    }    
  }
  public void fireModelChanged(String change) {
    if (isUpdating) return;
    //Debug.noteln("AbIFM: model change", change);
    if (listeners.isEmpty()) return;
    ObjectChangeEvent oce = new ObjectChangeEvent(this, getObject());
    for (Iterator e = listeners.iterator(); e.hasNext(); ) {
      FormModelListener l = (FormModelListener)e.next();
      l.formModelChanged(change, oce);
    }    
  }
  /** 
   * Notify listeners that data has changed. Also fires ModelChanged.
   * Make sure you only call this if data really has changed! 
   */
  public void fireDataChanged(String field, Object editedObject, 
			      Object oldValue, Object newValue) {
    ObjectChangeEvent oce = 
      new ObjectChangeEvent(this, editedObject, field, oldValue, newValue);
    fireModelChanged("Field " + field);
    //Debug.noteln("AIFM: model changed field", field);
    //Debug.noteln("    from", oldValue);
    //Debug.noteln("    to", newValue);
    //(new Throwable()).printStackTrace(); 
    if (isUpdating) return;
    //Debug.noteln("AIFM not updating");
    for (Iterator e = dataListeners.iterator(); e.hasNext(); ) {
	FormDataListener l = (FormDataListener)e.next();
	l.formDataChanged(oce);
    }    
  }


  /**
   * Listening to changes to the object generated elsewhere.
   */
  public void dataChanged(ObjectChangeEvent oce) {
    if (settingData) return;  //ignore own data changes
    if (oce.getObject() == null) return;
    if (oce.getObject() == baseObject) { //only listen to this object!
      //Debug.noteln("AIFM: got data change in field", oce.getField());
      fireModelChanged("Field", oce); //pass on to listeners (panel)
    }
    else {
      Debug.noteln("AbIFM: Form Model is listening to the wrong object!",
		   oce.getObject().toString());
      (new Throwable()).printStackTrace();
    }
  }
  public void dataChanged(EditableObject eo, String field, 
			  Object oldValue, Object newValue) {
    if (settingData) return;  //ignore own data changes
    if (eo == null) return;
    if (eo == baseObject) { //only listen to this object!
      //Debug.noteln("AIFM: got old data change in field", field);
      fireModelChanged("Field " + field);
    }
    else {
      Debug.noteln("AbIFM: Form Model is listening to the wrong object!",
		  eo.toString());
      (new Throwable()).printStackTrace();
    }
  }

  public void addModelListener(FormModelListener l) {    
    listeners.remove(l);
    listeners.add(l);
  }
  public void addDataListener(FormDataListener l) {    
    dataListeners.remove(l);
    dataListeners.add(l);
  }

  public HashSet getModelListeners() {return listeners;}
  public HashSet getDataListeners() {return dataListeners;}

}
