/****************************************************************************
 * A simple panel with a label, a component, and optional buttons.
 *
 * @author Jussi Stader
 * @version 2.3+
 * Updated: Mon Oct 30 11:41:35 2006
 * Copyright: (c) 2001, AIAI, University of Edinburgh
 *
 *****************************************************************************
 */
package ix.iface.ui;

import java.awt.BorderLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.Dimension;
import javax.swing.*;
import javax.swing.border.TitledBorder;
import javax.swing.text.JTextComponent;
import javax.swing.table.*;
import java.util.EventListener;
import java.util.Collection;

import ix.util.Debug;
import ix.iface.ui.table.*;
import ix.iface.ui.util.UIUtil;
    
/**
 * A simple panel with a label (title), a component, and optional buttons.
 * The given label will be at the top, the given component at the left and
 * buttons with given labels on the right.
 * Adds the given listener to the buttons and the list.
 */
public abstract class AbstractEditorPanel extends JPanel 
implements IXEditorPanel {

  public boolean isBordered = false;
  /** A sub-panel containing buttons */
  public IXButPanel butPanel;
  /** The panel's label (title) */
  public JLabel nameLabel;
  /** The panel's border (if it has one) */
  public TitledBorder border;
  /** The panel's item that may contain data. There are methods for setting
   * and getting data for JList and JTextComponent items
   */
  public JComponent itemComponent;
        
  /**
   * Creates a panel with a label (top left) or a border (depending on the
   * default)
   * Other components (a JComponent and a button panel) are usually added via
   * one of the other constructors; the label text is set in other
   * constructors.
   */
  public AbstractEditorPanel(){
    super(new GridBagLayout());
    makePanel();
  }
  /**
   * Creates a panel as above but with a given border setting.
   * If there is a border, the title of the panel will appear in the border,
   * otherwise it will appear in a label at the top left of the panel.
   * Other components (a JComponent and a button panel) are usually added via
   * one of the other constructors; the label text is set in other
   * constructors.
   */
  public AbstractEditorPanel(boolean makeBorder) {
    super(new GridBagLayout());
    isBordered = makeBorder;
    makePanel();
  }
  /**
   * Creates a panel with a title and a component (left).
   * If the border-flag is set, the title will go into the border, otherwise
   * a label will be used (top left).
   * If the component is a JList or a JTextArea, it is made scrollable
   * Also used by AbstractEditorPanel constructors that have buttons.
   *
   * @param ml the listener that is interested in component events
   * @param makeBorder a flag determining whether the title is set in a border
   *   (true) or a label (false).
   * @param label a string that is used as the label (title of the panel).
   * @param component the component at the left of the panel
   */
  public AbstractEditorPanel(EventListener ml, boolean makeBorder,
			     String label, JComponent component){
    this(makeBorder);
    makePanelComponent(ml, component);
    setLabel(label);
    this.setName(label);
  }
  /**
   * Creates a panel with a title and a component (left) as above with default
   * border/label.
   *
   * @param ml the listener that is interested in component events
   * @param label a string that is used as the label (title of the panel).
   * @param component the component at the left of the panel
   */
  public AbstractEditorPanel(EventListener ml, String label,
			     JComponent component) {
    this();
    makePanelComponent(ml, component);
    setLabel(label);
    this.setName(label);
  }
  /**
   * Creates a panel with a label (top left) and buttons (right) only.
   *
   * @param ml the listener that is interested in button events
   * @param label a string that is used as the label (title of the panel).
   * @param buttons an array of strings that are button labels
   */
  public AbstractEditorPanel(EventListener ml, String label,
			     String[] buttons) {
    this();
    setLabel(label);
    makeButtons(ml, buttons);
  }
      
  /**
   * Creates a panel with a title, component (left) and buttons (right).
   * The title goes either into a label (top left) or into a border, depening
   * on the parameter.
   * Calls AbstractEditorPanel(EventListener, label, JComponent)
   * and makes the buttons itself.
   *
   * @param ml the listener that is interested in component events
   * @param makeBorder a flag determining whether the title is set in a border
   *   (true) or a label (false).
   * @param label a string that is used as the label (title of the panel).
   * @param component the component at the left of the panel
   * @param buttons an array of strings that are button labels
   */
  public AbstractEditorPanel(EventListener ml, boolean makeBorder,
			     String label, JComponent component, 
			     String[] buttons){
    this(ml, makeBorder, label, component);
    makeButtons(ml, buttons);
  }
  /**
   * Creates a panel as above but with the default border/label setting.
   *
   * @param ml the listener that is interested in component events
   * @param label a string that is used as the label (title of the panel).
   * @param component the component at the left of the panel
   * @param buttons an array of strings that are button labels
   */
  public AbstractEditorPanel(EventListener ml, String label, 
			     JComponent component, String[] buttons){
    this(ml, label, component);
    makeButtons(ml, buttons);
  }

  /**
   * Makes an empty label and adds it to the panel;
   * The label is the title of the border if there is a border,
   * otherwise the label is at the top left of the panel.
   */
  private void makePanel() {
    if (isBordered) {
      border = BorderFactory.createTitledBorder("some label");
      this.setBorder(border);
    }
    else {
      GridBagConstraints gbc = 
	new GridBagConstraints(0,0,  GridBagConstraints.REMAINDER,1, //x,y, w,h
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.NORTHWEST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,4,0,0),0,0); //insets and padding
      nameLabel = new JLabel("some label");
      this.add(nameLabel, gbc);
    }
  }

  /**
   * Puts the component into the panel (left) and sets the title.
   * If the border-flag is set, the title will go into the border, otherwise
   * a label will be used (top left).
   * If the component is a JList, JTable or a JTextArea, it is made scrollable
   * Also used by AbstractEditorPanel constructors that have buttons.
   *
   * @param ml the listener that is interested in component events
   * @param label a string that is used as the label (title of the panel).
   * @param component the component at the left of the panel
   */
  protected void makePanelComponent(EventListener ml, JComponent component) {
    boolean horizontal = false;
    int fill = GridBagConstraints.BOTH;

    if (JTextField.class.isInstance(component)) {
      horizontal = true;
      fill = GridBagConstraints.HORIZONTAL;
    }
    else if (JCheckBox.class.isInstance(component)) {
      horizontal = true;
      fill = GridBagConstraints.NONE;
    }
    else if (component instanceof IXButPanel) {
      //Debug.noteln("AEdP: got butPan", component);
      if (((IXButPanel)component).getOrientation() != BoxLayout.Y_AXIS) 
	horizontal = true;
      fill = GridBagConstraints.NONE;
    }
    //variables fill and horizontal are set now

    GridBagConstraints gbc;

    if (isBordered) 
      gbc = new GridBagConstraints(1,0, 1,1,  //x,y, width,height
				   2.0,1.0,  //weight x,y
				   GridBagConstraints.NORTHWEST, //anchor
				   fill,// fill
				   new Insets(0,2,0,2),1,0); //insets, padding
    else if (horizontal) {
      //Debug.noteln("AEdP: horizontal");
      //change the label constraint
      gbc = new GridBagConstraints(0,0,  1,1, //x,y, w,h
				   0.0,0.0,  //weight x,y
				   GridBagConstraints.NORTHWEST, //anchor
				   GridBagConstraints.NONE,  //fill
				   new Insets(0,4,0,0),0,0); //insets, pad
      this.add(nameLabel, gbc);
      
      //make gbc for component next to label      
      gbc = new GridBagConstraints(1,0, 1,1,  //x,y, width,height
				   2.0,1.0,  //weight x,y
				   GridBagConstraints.NORTHWEST, //anchor
				   fill,// fill
				   new Insets(0,2,0,2),1,0); //insets, padding

    }
    else 
      gbc = new GridBagConstraints(0,1, 1,1,  //x,y, width,height
				   2.0,1.0,  //weight x,y
				   GridBagConstraints.WEST, fill, //anchor,fill
				   new Insets(0,2,0,2),1,0); //insets, padding
    if (isScrollItem(component)) {
      // JLists and JTextAreas should be scrollable!
      //Debug.noteln("AEdP: scoll item", component);
      JScrollPane jsp = new JScrollPane();
      jsp.getViewport().add(component);
      this.add(jsp, gbc);
    }
    else this.add(component, gbc);

    //strictly this bit should be obsolete with the above
    JComponent insideComponent = null;
    if (JScrollPane.class.isInstance(component)) {
      //if the component is a scroll pane, the real component is the one inside
      try { //try to get the component inside
	insideComponent = (JComponent)
	  ((JScrollPane)component).getViewport().getComponent(0);
      }
      catch (ArrayIndexOutOfBoundsException e) {
      }
    }
    if (insideComponent == null) //no scroll pane or no inside component found
      itemComponent = component;
    else itemComponent = insideComponent;

    addComponentListener(ml);
  }
  private void makeButtons(EventListener ml, String[] buttons){
    if ((buttons != null) && (buttons.length > 0)) {
      butPanel = new IXButPanel(ml, (Object[])buttons);
      GridBagConstraints gbc = 
	new GridBagConstraints(1,1,  //x,y
			       1,1,   //width,height
			       0.0,0.0,  //weight x,y
			       GridBagConstraints.EAST, //anchor
			       GridBagConstraints.NONE,  //fill
			       new Insets(0,2,0,2),0,0); //insets and padding
      this.add(this.butPanel, gbc);
    }
  }

  public void addComponentListener(EventListener ml) {
      UIUtil.addListener(itemComponent, ml);
  }

  /** Sets the panel's label */
  public void setLabel(String label) {
    if (isBordered) border.setTitle(label);
    else nameLabel.setText(label);
  }
      
  /**
   * Gets the component of the panel.
   *
   * @return the JComponent that is at the left of the panel
   */
  public JComponent getItemComponent(){
    return this.itemComponent;
  }

  public boolean isListItem() {
    return isListItem(this.itemComponent);
  }
  public static boolean isListItem(JComponent component) {
    return ((component instanceof JList)
	    || (component instanceof JComboBox)
	    || (component instanceof JTable));
  }
  public static boolean isScrollItem(JComponent component) {
    return ((component instanceof JList)
	    || (component instanceof JTextArea)
	    || (component instanceof JTable));
  }

  /** Gets the panel's name (the label) */
  public String getName(){ return this.nameLabel.getText(); }

  /** Makes the item (not) editable and enables (disables) the buttons */
  public void setEnabled(boolean isEditable) {
    //if (isEditable) return; //by default things are enabled. Save some work.
    if (butPanel != null)
      this.butPanel.setEnabled(isEditable);
  }



  /**
   * Minimises the component and the rest of the editor
   */
  public void minimise() {
    Class iClass = itemComponent.getClass();
    Dimension dim1 = itemComponent.getSize();
    Debug.noteln("item size before minimise", dim1);
    if (itemComponent instanceof JList) minimiseList();
    if (itemComponent instanceof JTextArea) minimiseText();
    else if (itemComponent instanceof JTable) minimiseTable();      
    else if (itemComponent instanceof JTree) minimiseTree();   
    Dimension dim2 = itemComponent.getSize();
    Debug.noteln("item size after minimise", dim2);
    if (dim2.getHeight() < dim1.getHeight())
      dim2.setSize(dim2.getWidth(), dim2.getHeight()*2);
    this.setSize(dim2);
  }

  private void minimiseList() {
    ((JList)getItemComponent()).setVisibleRowCount(1);
  }

  //neither of these work!
  private void minimiseText() {
    JTextArea text = (JTextArea)getItemComponent();
    text.setRows(1);
    //text.setSize(text.getSize().getWidth(), text.getHeight());
    text.invalidate();
  }
  private void minimiseTree() {
    ((JTree)getItemComponent()).setVisibleRowCount(1);
  }
  private void minimiseTable() {
    JTable table = (JTable)getItemComponent();
    Dimension dim = table.getPreferredScrollableViewportSize();
    dim.setSize(dim.getWidth(), table.getRowHeight());
    table.setPreferredScrollableViewportSize(dim);
  }

  public boolean hasData() {
    Object data = getData();
    if (data == null) return false;
    Object[] d = {};
    if (data == d) return false;
    if (data instanceof Collection) return !((Collection)data).isEmpty();
    return true;
  }

  public abstract Object getData();
  public abstract void setData(Object data);
  public abstract void removeData();
  public abstract boolean setRenderer(ListCellRenderer r);

}
