;;; Author: Jeff Dalton
;;; Updated: Sun May  9 04:22:57 2004 by Jeff Dalton
;;; Copyright: (c) 2004, AIAI, University of Edinburgh

(in-package :oplan)

(defvar *id-to-ix-node-table* (make-hash-table :test #'equal))

(defstruct (ix-node (:print-function print-ix-node))
  id
  pattern
  begin
  end
  parent
  (children '()))

(defstruct (ix-node-end (:print-function print-ix-node-end))
  node-id
  end			; :begin or :end
  (predecessors '())
  (successors '()))

(defun main-draw-psgraph-for-ix ()
  (multiple-value-bind (nodes orderings) (read-ix-nodes-and-orderings)
    (let* ((outfile
	    (or (get-parameter :output)
		(generate-unique-filename
		   (namestring (temp-filename "graph"))
		   ".ps")))
	   (viewer (get-parameter :ps-viewer)))
      ;; Draw the graph
      (with-open-file (*standard-output* outfile :direction :output)
	(draw-psgraph-for-ix nodes orderings))
      ;; If an output file was specified, we're done;
      ;; otherwise, display the graph and delete the tmp file.
      (unless (get-parameter :output)
	(system
	 (concat-string
	  "(" viewer " " outfile "; /bin/rm " outfile ")&"))))))

(defun read-ix-nodes-and-orderings ()
  (let ((graph-file (get-parameter :graph)))
    (if (string= graph-file "-")
	(values (read) (read))
      (with-open-file (in graph-file :direction :input)
	(let* ((nodes (read in))
	       (orderings (read in)))
	  (values nodes orderings))))))


;;; We could convert the I-X information to an O-Plan end-graph,
;;; perhaps by first converting it to O-Plan node-descriptions,
;;; but then we'd have to make sure we used node-1 and node-2
;;; the way O-Plan did, and we'd have to understand the end-graph
;;; code when debugging, so that it's easier to construct our
;;; own representation.

(defvar *ix-minimals*)

(defun draw-psgraph-for-ix (nodes orderings)
  (make-ix-nodes nodes)
  (add-ix-orderings orderings)
  (add-ix-parent-child-orderings)
  (let ((*ix-minimals* (minimal-ix-node-ends)))
    (psgraph:psgraph
      (ix-root-node-end *ix-minimals*)	;root
      #'ix-node-end-graph-successors	;childf
      #'ix-node-end-info		;infof
      t					;shrink to one page
      nil				;not insert
      #'eq				;test
      t; nil				;don't remove redundant links
      nil)))				;no title (was "TPN")

(defun minimal-ix-node-ends ()
  (let ((result '()))
    (maphash #'(lambda (id node)
		 (declare (ignore id))
		 (let ((begin (ix-node-begin node)))
		   (when (null (ix-node-end-predecessors begin))
		     (push begin result))))
	     *id-to-ix-node-table*)
    result))

(defun ix-root-node-end (minimals)
  (if (length=1 minimals)
      (first minimals)
    :start))

(defun ix-node-end-graph-successors (ne)
  (if (eq ne :start)
      *ix-minimals*
    (let ((node (ix-node-end-node ne)))
      (if (and (eq (ix-node-end-end ne) :begin)
	       (ix-unit-node-p node))
	  (ix-node-end-successors (ix-node-end node))
	(ix-node-end-successors ne)))))

(defun ix-node-end-info (ne)
  (if (eq ne :start)
      '("START")
    (let ((node (ix-node-end-node ne)))
      (ecase (ix-node-end-end ne)
	(:begin
	 (if (ix-unit-node-p node)
	     (list (format-upcase "~A ACTION" (ix-node-id node))
		   (format-upcase "~A" (ix-node-pattern node)))
	   (list (format-upcase "~A BEGIN" (ix-node-id node))
		 (format-upcase "~A" (ix-node-pattern node)))))
	(:end
	 (if (ix-unit-node-p node)
	     (error "Shouldn't reach end of unit node ~S" ne)
	   (list (format-upcase "~A" (ix-node-id node))
		 "END")))))))

(defun format-upcase (format-string &rest args)
  (format nil "~:@(~?~)" format-string args))

;;;; Node and node-end access and construction
;;;

(defun ix-node-end-node (ne)		;pseudo-accessor
  (get-ix-node (ix-node-end-node-id ne)))

(defun print-ix-node-end (ne stream depth)
  (declare (ignore depth))
  (format stream "#<ix-node-end ~S ~S>"
	  (ix-node-end-node-id ne)
	  (ix-node-end-end ne)))

(defun print-ix-node (n stream depth)
  (declare (ignore depth))
  (format stream "#<ix-node ~S ~S>" (ix-node-id n) (ix-node-pattern n)))

(defun get-ix-node (id)
  (or (gethash id *id-to-ix-node-table*)
      (error "No node with id=~S" id)))

(defun get-ix-node-end (etag)
  ;; An etag is (node-id end-keyword).  That is the opposite
  ;; order from that in an I-X NodeEndRef, but we use the
  ;; O-Plan order in the description this program takes as input.
  (let ((node (get-ix-node (etag-node etag))))
    (ecase (etag-end etag)
      (:begin (ix-node-begin node))
      (:end (ix-node-end node)))))

(defun make-ix-nodes (nodes)
  ;; Create the nodes and node-ends
  (loop for (id pattern) in nodes do
    (let ((node (make-ix-node-and-ends id pattern)))
      (ix-link-before (ix-node-begin node) (ix-node-end node))
      (setf (gethash id *id-to-ix-node-table*)
	    node)))
  ;; Connect parents and children
  (dolist (node (hash-table-values *id-to-ix-node-table*))
    (let ((parent (find-ix-parent-from-child-id node)))
      (when parent
	(setf (ix-node-parent node) parent)
	(push node (ix-node-children parent))))))

(defun make-ix-node-and-ends (id pattern)
  (make-ix-node
    :id id
    :pattern pattern
    :begin (make-ix-node-end :node-id id :end :begin)
    :end  (make-ix-node-end :node-id id :end :end)))

(defun find-ix-parent-from-child-id (child)
  (let* ((child-id (ix-node-id child))
	 (parent-id (break-string-at-last #\- child-id)))
    (if (string-equal parent-id "node")
	nil
      (get-ix-node parent-id))))

(defun add-ix-orderings (orderings)
  (loop for (from-etag to-etag) in orderings do
    (let ((from (get-ix-node-end from-etag))
	  (to (get-ix-node-end to-etag)))
      (ix-link-before from to))))

(defun add-ix-parent-child-orderings ()
  (dolist (parent (hash-table-values *id-to-ix-node-table*))
    (let ((pb (ix-node-begin parent))
	  (pe (ix-node-end parent)))
      (dolist (child (ix-node-children parent))
	;; (format *debug-io* "Parent ~S, child ~S~%" parent child)
	(let ((cb (ix-node-begin child))
	      (ce (ix-node-end child)))
	  ;; /\/: By now, a child node-end might have predecessors
	  ;; or successors other than its siblings, and so the
	  ;; null tests below don't work.
	  (when t ; (null (ix-node-end-predecessors cb))
	    (ix-link-before pb cb))
	  (when t ; (null (ix-node-end-successors ce))
	    (ix-link-before ce pe)))))))

(defun ix-link-before (from to)
  ;; (format *debug-io* "-- ~S --> ~S~%" from to)
  (nconcf1 (ix-node-end-successors from) to)
  (nconcf1 (ix-node-end-predecessors to) from))

(defun ix-unit-node-p (node)
  ;; True if only one link from begin_of node and one to end_of node
  ;; and the one link directly links the two ends of the node.
  (let ((begin-successors (ix-node-end-successors (ix-node-begin node)))
	(end-predecessors (ix-node-end-predecessors (ix-node-end node))))
    (and (length=1 begin-successors)
	 (length=1 end-predecessors)
	 (equal (first begin-successors) (ix-node-end node))
	 (equal (first end-predecessors) (ix-node-begin node)))))

;;; End
