package RemoteService.agent;

import RemoteService.resource.*;
import RemoteService.agent.*;
import JavaAgent.agent.*;
import JavaAgent.resource.*;
import JavaAgent.context.ContextInterface;
import java.util.Hashtable;
import java.util.Vector;
import java.util.Enumeration;
import java.io.*;
import java.net.URL;
import java.awt.Panel;

/**
 * ServiceAgent: Subclass of Agent which manages a set of Service objects. 
 * When each Service is added a registration message is sent to the 
 * ServiceBroker providing both a description and gif thumbnail. Requests for
 * a specific Service will be responded to with the location of the appropriate
 * class file.<p>
 *
 * <hr>
 * Copyright (c) 1995, H. Robert Frost, Stanford University.
 * All rights reserved.<p>
 * Copyright (c) 1996, H. Robert Frost, Enterprise Integration Technologies,
 * Inc. All rights reserved.<p>
 *
 * RESTRICTED RIGHTS LEGEND: Use, duplication or disclosure by the 
 * Government is subject to restrictions as set forth in 
 * subparagraph(c)(1)(ii) of the Rights in Technical Data and Computer
 * Software clause at DFARS 252.227-7013 and in similar clauses in the
 * FAR and NASA FAR supplement.<p>
 * 
 * This software is bound by the terms and conditions listed in the
 * attached <a href="../LICENSE">LICENSE file</A>.
 * <hr>
 * @author <A HREF="http://cdr.stanford.edu/html/people/frost-bio.html"> Rob Frost</A>
 * @version 0.3 5/21/96 for Java(tm) Language Version 1.0.2
 */

public class ServiceAgent extends Agent {

  /**
   * Contructs a ServiceAgent.
   *
   * @param n String identifier for the Agent.
   * @param init_url URL for the initialization file.
   * @param shared_url URL for shared classes.
   * @param shared_dir Directory for local access to shared classes.
   * @param working_dir Directory for local files, will be null if the 
   * Agent is unable to write to the local file system.
   */

  public ServiceAgent(ContextInterface context,String n, URL init_url, 
		     URL shared_url, File shared_dir,File working_dir) {
    super(context, n, init_url, shared_url, shared_dir, working_dir);
  }

  /**
   * Creates the ServiceResourceManager, ServiceMessageHandler and reads in the
   * init_file.
   */
  
  public void init(){
    if(working_dir != null){
      io_enabled = true;
    }
    resources = new ServiceResourceManager(this);
    handler = new ServiceMessageHandler(this);
    loadInitFile(init_url);
  }

  /**
   * Called by a Resource object when the number of elements has changed.
   * @param type Type of the Resource.
   */

  public void resourceChanged(Resource r){
    String class_name = r.getClass().getName();
    String type = null;
    if(class_name.equals("RemoteService.resource.AvailableServices")){
      type = "available_service";
      context.resourceChanged(type);
    } else {
      super.resourceChanged(r);
    }
  }

  /**
   * Called by the ServiceFrame to add a Service. Adds the AvailableService
   * object to the ServiceAgent's Resources then sends a register-service
   * message to the ANS.
   * @param service_name Name of the service agent
   */
  
  public void addService(AvailableService service){
    getResource("available_service").addElement(service.name,service, true);   
    addService(service);
    KQMLmessage message = new KQMLmessage();
    message.addFieldValuePair("sender",getName());
    message.addFieldValuePair("receiver","ANS");
    message.addFieldValuePair("performative","evaluate");
    message.addFieldValuePair("language","KQML");
    message.addFieldValuePair("ontology","service");
    String file_url = getSharedClassesURL().toString();
    String path = file_url + File.separator + "files" + 
      File.separator;
    String descrip = path + service.descrip.getName();
    String gif = path + service.gif.getName();
    message.addFieldValuePair("content","(register-service :descrip " +
			      descrip + " :gif " + gif +")");
    sendMessage(message);
  } 

  /**
   * Called to remove a service. Sends out an unregister-service message.
   * @param name Name of the Service.
   */
  
  public void removeService(String name){
    try {
      AvailableServices available_services = 
	(AvailableServices)getResource("available_service");
      if(available_services.hasElement(name)){
	/* Remove service from the Resource */
	AvailableService service = 
	  (AvailableService)available_services.getElement(name,Resource.NO_RETRIEVAL);
	available_services.removeElement(service);
	KQMLmessage message = new KQMLmessage();
	message.addFieldValuePair("sender",getName());
	message.addFieldValuePair("receiver","ANS");
	message.addFieldValuePair("performative","evaluate");
	message.addFieldValuePair("language","KQML");
	message.addFieldValuePair("ontology","service");
	message.addFieldValuePair("content","(unregister-service :name " +
				name + ")");
	sendMessage(message);
      }
    } catch (ResourceException e){
      addSystemMessage("Unable to remove service.", e);
    }
  }
    
}







