package JavaAgent.resource.fopl;


import java.util.*;
import java.io.*;
import java.text.ParseException;

import JavaAgent.resource.cdl.KRSentence;

/**
** A LitLObject is a rather ugly side effect necessary to allow the
** inheritance from the JAT Language class. Literal cannot directly inherit 
** from the Language class because a Literal is a Formula. Hence,
** this class, LitLObject, inherits from Language and contains only 
** exactly one object, a Literal.
** @see JavaAgent.resource.Language
** @see Literal
*/

public class LitLObject extends JavaAgent.resource.Language 
    implements JavaAgent.resource.cdl.Parsable, KRSentence {

  /**
  ** This constructor creates an empty instance of a LitLObject.
  */
  public LitLObject() {
    theLit = null;
  }
    
  /**
  ** This constructor for a LitLObject parses the given String and
  ** creates the according Literal.
  ** @param input The string to be parsed.
  ** @exception ParseException An Exception can occur if parsing fails.
  */
  public LitLObject(String input) throws java.text.ParseException {
    theLit = Literal.parseLit(input);
  }

  /** 
  ** This function generates a copy of this LitLObject. The copy will contain
  ** a clone of the Literal contained in this one.
  ** @return an equal copy of this LitLObject
  */
  public Object clone() {
    LitLObject res = new LitLObject();
    res.theLit = (Literal)theLit.clone();
    return res;
  }

  /** 
  ** Returns a copy of this Literal with Variables replaced according to the 
  ** given Substitution. Symbols in the copy will be the same 
  ** as the original ones but Variables will be repalced by the respective 
  ** Terms in the given Substitution. A Variable not occurring in the 
  ** Substitution will be replaced by a new (different) Variable with the 
  ** same name. The given Substitution will be extended to include the 
  ** mappings of old Variables to new ones.
  ** <p> This function can also be used to generate a copy of this 
  ** Literal that contains a new set of Variables. The given Substitution 
  ** should be a new, empty Substitution in this case. Otherwise cloning
  ** can be seen as instantiating the Literal.
  ** @param s the Substitution that tells us how to replace Variables
  ** @return a new LitLObject that is an instance of this LitLObject; the 
  ** Substitution <tt>s</tt> will be extended with any new Variable 
  ** replacements introduced
  ** @exception IllegalArgumentException An exception can occur if the given
  ** Substitution is null. In this case <tt>clone()</tt> (above) should be
  ** used.
  */
  public LitLObject clone(Substitution s) 
      throws IllegalArgumentException {
    LitLObject res = new LitLObject();
    res.theLit = (Literal)theLit.clone(s);
    return res;
  }

  /**
  ** This function returns the Literal embedded in this LitLObject.
  ** @return the Literal embedded in this LitLObject
  */
  public Literal getLiteral() {
    return theLit;
  }

  /**
  ** This function returns a FOPLFormula that represents a conjunction of
  ** the Literals in the given Vector. Note that this is function does
  ** not return a LitLObject but a different kind of KRSentence.
  ** @param conjuncts a Vector of LitLObjects
  ** @return a FOPLFormula containing the conjunction of the given 
  **   LitLObjects
  */
  static public KRSentence getConjunction(Vector conjuncts) {
    if (conjuncts.size() < 2)
      return (KRSentence)conjuncts.firstElement();
    MultiConFormula res = new MultiConFormula(
        YYtokentypes.AND.intValue(), conjuncts.size()+1);
    for (Enumeration fe = conjuncts.elements(); fe.hasMoreElements(); )
      res.addFormula(((LitLObject)fe.nextElement()).theLit);
    FOPLFormula result = new FOPLFormula();
    result.theForm = res;
    return result;
  }

  /**
  ** This function returns the Class that implements KnowledgeBase for
  ** this KRSentence Class. It could be something much simpler but for 
  ** now it is FormulaSet.
  */
  static public Class getKBClass() {
    try {
      return Class.forName("JavaAgent.resource.fopl.FormulaSet");
    } catch (ClassNotFoundException cnfe) {
      throw new UnknownError("Failed to find class FormulaSet!");
    }
  }

  /**
  ** This function returns a FOPLFormula that contains the Literal in this
  ** LitLObject.
  */
  public FOPLFormula getFOPLFormula() {
    FOPLFormula result = new FOPLFormula();
    result.theForm = theLit;
    return result;
  }

  /**
  ** This function tests whether this and the given Object are equal. In
  ** general, two LitLObjecte are equal if they contain equal Literals.
  ** @param otherLit the LitLObject this one is to be compared to
  ** @return whether the given Object is equal to this one
  */
  public boolean equals(Object otherLit) {
    if (otherLit == null)
      return false;
    if (otherLit instanceof Literal)
      return theLit.equals(otherLit);
    if (! (otherLit instanceof LitLObject))
      return false;
    return theLit.equals(((LitLObject)otherLit).theLit);
  }

  /**
  ** This function returns the String that represents this LitLObject.
  */
  public String toString() {
    return theLit.toString();
  }

  /**
  ** This function should only be used on empty instances of a LitLObject.
  ** It will parse the given String and create the according Literal to
  ** be kept in this LitLObject.
  ** @param input The string to be parsed.
  */
  public void parseString(String input) {
    try {
      theLit = Literal.parseLit(input);
    }
    catch (ParseException pe) {
      // Language.parseString does not allow the throwing of exceptions!
    }
  }

  /**
  ** This function can be used to parse a given String that represents a 
  ** LitLObject. The syntax is as for a Literal.
  ** @exception IllegalArgumentException An exception will occur if the 
  ** supplied String is empty or null.
  ** @exception ParseException An exception can occur if parsing failed.
  ** Potential reasons include a syntax error or an I/O problem.
  */
  public static LitLObject parse(String s) 
      throws IllegalArgumentException, ParseException {
    LitLObject res = new LitLObject();
    res.theLit = Literal.parseLit(s);
    return res;
  }

  /**
  ** This function can be used to parse a given InputStream that represents 
  ** a Literal. The syntax is as for a Literal.
  */
  public static LitLObject parse(InputStream ist) 
      throws IllegalArgumentException, IOException, ParseException {
    LitLObject res = new LitLObject();
    res.theLit = Literal.parseLit(ist);
    return res;
  }

  /**
  ** The Literal contained in this LitLObject.
  */
  protected Literal theLit;

}
