package JavaAgent.resource.cdl;


import java.util.*;
import java.io.*;

/**
** A KnowledgeBase is a set of KRSentences. A KnowledgeBase, like a 
** KRSentence, does not enforce much functionality. The idea is rather that 
** if a function can be defined for Class implementing a KnowledgeBase then
** the function name and order of parameters can be assumed as described 
** below. This allows other classes to test for the existence of these
** functions and assume that they are not implemented if it cannot find 
** them. Thus, here is the intentional interface definition:
** <ul>
** <li> <b>public <i>KnowledgeBase</i>();</b>
**   <br> This constructor allows the construction of an empty KnowledgeBase
**   that does not contain any knowledge yet.
** <li> <b>public boolean isInconsistent();</b>
**   <br> This function tests whether this KnowledgeBase is inconsistent.
**   It return <tt>true</tt> iff this KnowledgeBase is inconsistent.
** <li> <b>public Hashtable evaluate(KRSentence sent, Vector vars);</b>
**   <br> This function is an extension of <tt>evaluate</tt> described below.
**   Instead of just testing whether the given KRSentence follows from this
**   KnowledgeBase this function tries to find a substitution for the Variables
**   in the given Vector such that the given KRSentence follows from this
**   KnowledgeBase. If no such substitution can be found it will return null.
** <li> <b>public void writeToStream(OutputStream os);</b>
**   <br> This function writes the content of this KnowledgeBase to the given
**   OutputStream.
** </ul>
** @see KRSentence
*/

public interface KnowledgeBase {
  /**
  ** This function generates a copy of this KnowledgeBase that holds the same
  ** knowledge in the same formalism.
  ** @return a copy of this KnowledgeBase
  */
  public Object clone();

  /**
  ** This function can be used to add knowledge to this KnowledgeBase.
  ** @param sent the KRSentence holding the new knowledge
  ** @exception IllegalArgumentException An exception will occur if the given
  ** KRSentence is null.
  ** @exception FormalismException An exception will occur if the given
  ** KRSentence is not of the right Formalism.
  */
  public void assert(KRSentence sent)
      throws IllegalArgumentException, FormalismException;

  /**
  ** This function takes a KRSentence and evaluates it against this
  ** KnowledgeBase.
  ** @param sent the KRSentence to be evaluated
  ** @return <tt>true</tt> iff the given KRSentence could be derived from 
  ** the KnowledgeBase
  ** @exception IllegalArgumentException An exception will occur if the given
  ** KRSentence is null.
  ** @exception FormalismException An exception will occur if the given
  ** KRSentence is not of the right Formalism.
  ** @exception IOException An exception can occur if writing to a given 
  ** OutputStream for tracing fails.
  */
  public boolean evaluate(KRSentence sent)
      throws IllegalArgumentException, FormalismException, IOException;

  /**
  ** This function returns an enumerator for the KRSentences in this
  ** KnowledgeBase.
  ** @return an Enumeration for the KRSentences in this KnowledgeBase
  */
  public Enumeration sentences();

}
