/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Nov  7 14:58:23 2006 by Jeff Dalton
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.iglobe.test;

import java.util.*;

import iglobe.plan.*;

import ix.icore.*;
import ix.icore.plan.*;
import ix.icore.domain.End;
import ix.icore.process.*;
import ix.ip2.*;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

/**
 * Turns the subplan rooted in a specified top-level activity
 * into a {@link RescuePlan}.  To use it, create an instance
 * and call its {@link #makeRescuePlan()} method.
 */
public class RescuePlanMaker {

    protected Ip2 ip2;
    protected ActivityItem root;

    protected List<RescueActivity> activities = 
	new LinkedList<RescueActivity>();

    protected List<PlanConstraint> constraints =
	new LinkedList<PlanConstraint>();

    protected Map<ActivityItem,RescueActivity> activityTranslation =
	new LinkedHashMap<ActivityItem,RescueActivity>();

    Gensym.Generator nameGen = new Gensym.Generator();

    public RescuePlanMaker(Ip2 ip2, ActivityItem root) {
	this.ip2 = ip2;
	this.root = root;
    }

    public RescuePlan makeRescuePlan() {
	RescuePlan plan = new RescuePlan();
	createRescueActivities();
	collectConstraints();
	if (!activities.isEmpty()) plan.setActivities(activities);
	if (!constraints.isEmpty()) plan.setConstraints(constraints);
	return plan;
    }

    /**
     * Make rescue-activities that correspond to activities in the plan.
     */
    void createRescueActivities() {
	mapActivityTree(root, "activity");
    }

    void mapActivityTree(ActivityItem root, String baseId) {
	// Visit root
	String id = nameGen.nextString(baseId);
	RescueActivity act = makeRescueActivity(root, id);
	Debug.noteln("ACT", act);
	activities.add(act);
	activityTranslation.put(root, act);
	// Map children
	for (Iterator i = root.getChildren().iterator(); i.hasNext();) {
	    ActivityItem child = (ActivityItem)i.next();
	    mapActivityTree(child, id);
	}
    }

    RescueActivity makeRescueActivity(ActivityItem item, String id) {
	LList pattern = (LList)Variable.removeVars(item.getPattern());
	String description = Lisp.elementsToString(pattern);
	String verb = pattern.get(0).toString();
	RescueActivity act = (verb.equals("transport"))
	    ? makeTransportActivity(pattern)
	    : new AbstractActivity();
	act.setId(id);
	act.setDescription(description);
	return act;
    }

    TransportActivity makeTransportActivity(LList pattern) {
	TransportActivity act = new TransportActivity();
	LList syntax = Lisp.elementsFromString
	    ("transport ?number ?resource ?location");
	MatchEnv env = SimpleMatcher.mustMatch(syntax, pattern);
	Long n = (Long)matchValue(env, "?number", Long.class);
	Symbol resource = (Symbol)matchValue(env, "?resource", Symbol.class);
	Symbol location = (Symbol)matchValue(env, "?location", Symbol.class);
	act.setAmount(n.intValue());
	act.setResource(resource.toString());
	act.setDestination(location.toString());
	return act;
    }

    Object matchValue(MatchEnv env, String var, Class c) {
	Object value = env.get(Symbol.intern(var));
	return Util.mustBe(c, value);
    }

    /**
     * Collect ordering constraints that apply to the rescue activities.
     */
    void collectConstraints() {
	for (Map.Entry<ActivityItem,RescueActivity> e
		 : activityTranslation.entrySet()) {
	    ActivityItem item = e.getKey();
	    RescueActivity act = e.getValue();
	    PNodeEnd itemBegin = item.getBegin();
	    for (PNodeEnd pre : (List<PNodeEnd>)itemBegin.getPredecessors()) {
		if (pre.getEnd() != End.END)
		    continue;
		ActivityItem preItem = (ActivityItem)pre.getNode();
		if (!preItem.isDescendentOf(root))
		    continue;
		RescueActivity preAct = activityTranslation.get(preItem);
		Debug.expect(preAct != null, "Can't find", preItem);
		OrderConstraint ord = new OrderConstraint();
		ord.setFromId(preAct.getId());
		ord.setToId(act.getId());
		constraints.add(ord);
	    }
	}
    }

}
