/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Mar  1 12:30:22 2007 by Jeff Dalton
 * Copyright: (c) 2006, 2007, AIAI, University of Edinburgh
 */

package ix.iglobe.test;

import javax.swing.*;
import java.awt.Color;
import java.awt.event.*;
import java.util.*;

import iglobe.plan.*;

import ix.iglobe.test.shared.*;
import ix.icore.Activity;
import ix.icore.domain.Constraint;
import ix.icore.domain.PatternAssignment;

import ix.ip2.*;
import ix.iplan.IPlanOptionManager;

import ix.iface.util.IFUtil;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;

/**
 * An I-P2 customised for use in an I-Globe {@link RescuePlanner}.
 */
public class IGlobeIp2 extends Ip2 implements IGlobeIXAgent {

    IGlobeCallback callback;

    boolean handleAutomatically = false;

    Map<Name,RescueTask> activityIdToTask =
	new LinkedHashMap<Name,RescueTask>();

    XMLTreeViewFrame treeViewer = new XMLTreeViewFrame(this, "Rescue Plan");

    public IGlobeIp2() {
	super();

	// Change some defaults
	logoLine1 = "I-Globe I-X Process Panel";
	logoImage = "ip2-logo.gif";

    }

    /**
     * Main program for testing this class independently.
     */
    public static void main(String[] argv) {
	Util.printGreeting("I-Globe I-P2");
	new IGlobeIp2().mainStartup(argv);
    }

    public void setIGlobeCallback(IGlobeCallback callback) {
	this.callback = callback;
    }

    /* --- Methods in the IGlobeIXAgent interface ---> */

    public void newTask(RescueTask task) {

	Debug.noteln("New rescue task", task);

	// Create an activity that will expand into a way to
	// handle the task.
	Symbol verb = Symbol.intern("deal-with-accident");
	Symbol location = Symbol.intern(task.getLocation());
	int numberInjured = task.getNumberInjured();
	LList pattern = Lisp.list(verb, new Long(numberInjured), location);
	Activity act = new Activity(pattern);
	act.ensureId();
	activityIdToTask.put(act.getId(), task);

	// Add the activity.
	// /\/: We now do this too directly for it to work with options.
	// Used to do: handleInput(new IPC.BasicInputMessage(act));
	ActivityItem node = controller.addActivity(act);
        node.setIsNew(true);

	// Make constrains that will prevent use of the unavailable
	// resources in the plan for the task.
	List<String> unavailableResources = task.getUnavailableResources();
	if (unavailableResources != null) {
	    List<Constraint> constraints =
		makeUnavailableResourceConstraints(unavailableResources);
	    // Add the constraints.
	    getIp2ModelManager().addConstraints(node, constraints);
	}

	if (handleAutomatically)
	    planFor(node, task);

    }

    public void schedulingFailure(RescueTask task) {
	// /\/: For now ...
	Debug.noteln("Scheduling failure for", task);
	newTask(task);
    }

    public void viewPlan(RescuePlan plan) {
	treeViewer.editObject(plan);
	treeViewer.setVisible(true);
    }

    @Override
    public void exit() {
	frame.dispose();	// need to get rid of tools etc too /\/
	callback.exit();
    }

    /* ---> end of methods in the IGlobeIXAgent interface */

    List<Constraint> makeUnavailableResourceConstraints
	                 (List<String> unavailableResources) {
	List<Constraint> result = new LinkedList<Constraint>();
	for (String r: unavailableResources) {
	    PatternAssignment pv = new PatternAssignment
		(Lisp.list(Symbol.intern(r)), new Long(0)); // (r) = 0
	    Constraint c = new Constraint
		 ("resource", "overall", Lisp.list(pv));
	    result.add(c);
	}
	return result;
    }

    void planFor(ActivityItem item, RescueTask task) {
	// Get a plan.
	IPlanOptionManager optMan = getOptionManager();
	if (!optMan.plan()) {
	    // No plan found
	    callback.noPlan(task);
	    return;
	}

	// Construct a RescuePlan
	Name actId = item.getAbout().getId();
	ActivityItem inPlan = 
	    (ActivityItem)controller.getActivityAgenda().getItem(actId);
	RescuePlanMaker planMaker = new RescuePlanMaker(this, inPlan);
	RescuePlan p = planMaker.makeRescuePlan();

	callback.takePlan(p, task);
    }

    @Override
    protected Ip2Frame makeIp2Frame() {
	final Ip2Frame frame = new ModifiedIp2Frame(this);

	int rgb = Parameters.getInt("background-color", 0);
	if (rgb !=0) {
	    Color c = new Color(rgb);
	    JComponent content = (JComponent)frame.getContentPane();
	    content.setBorder
		(BorderFactory.createLineBorder(c, 3));
	}

	frame.addTest("Generate rescue task", new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		String unavailable = JOptionPane.showInputDialog
		    (frame, "Enter unavailable resources separated by commas");
		if (unavailable == null)
		    return;	// user cancelled
		else if (unavailable.equals(""))
		    callback.generateTask();
		else
		    callback.generateTask(Strings.breakAt(",", unavailable));
	    }
	});

	frame.addTest("Set to automatic", new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		AbstractButton button = (AbstractButton)event.getSource();
		String command = event.getActionCommand();
		if (command.endsWith("automatic")) {
		    handleAutomatically = true;
		    button.setText("Set to manual");
		}
		else if (command.endsWith("manual")) {
		    handleAutomatically = false;
		    button.setText("Set to automatic");
		}
		else
		    throw new ConsistencyException
			("Invalid test action", command);
	    }
	});

	return frame;
    }

    class ModifiedIp2Frame extends NewIp2Frame {

	ModifiedIp2Frame(Ip2 ip2) {
	    super(ip2);
	}

	@Override
	public void adjustAgendaItemPopup(AbstractAgendaItemPopupMenu menu,
					  AgendaItem item) {
	    if (item.getLevel() == 0 && item instanceof ActivityItem) {
		if (!item.getChildren().isEmpty()) {
		    JMenuItem sendItem =
			IFUtil.makeMenuItem
		            ("Send to HUMRED",
			     new SendSubtreeHandler((ActivityItem)item));
		    menu.add(sendItem);
		}
		JMenuItem noPlanItem =
		    IFUtil.makeMenuItem
		        ("Tell HUMRED no plan was found",
			 new SendNoPlanHandler((ActivityItem)item));
		menu.add(noPlanItem);
	    }
	}

    }

    class SendSubtreeHandler implements ActionListener {

	ActivityItem item;

	SendSubtreeHandler(ActivityItem item) {
	    this.item = item;
	}

	public void actionPerformed(ActionEvent event) {
	    RescuePlanMaker planMaker =
		new RescuePlanMaker(IGlobeIp2.this, item);
	    RescuePlan plan = planMaker.makeRescuePlan();

	    // So, now we have our plan.

	    Debug.noteln("Plan", XML.objectToXMLString(plan));

	    RescueTask task = activityIdToTask.get(item.getAbout().getId());
	    callback.takePlan(plan, task);

	}

    }

    class SendNoPlanHandler implements ActionListener {
	ActivityItem item;
	SendNoPlanHandler(ActivityItem item) {
	    this.item = item;
	}
	public void actionPerformed(ActionEvent event) {
	    RescueTask task = activityIdToTask.get(item.getAbout().getId());
	    callback.noPlan(task);
	}
    }

}
