
package across.visio.oldvisio;

import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import across.data.AgentData;
import across.data.Batch;
import across.data.ColumnData;
import across.data.Page;
import across.data.PublicParams;
import across.data.RelationInfo;
import across.data.TransportBatch;
import across.data.TransportCfp;
import across.simulation.constants.AcrossVisioConstants;
import aglobe.container.transport.Address;
import aglobe.service.gis.server.GISTopicServerListener;
import aglobe.visio3D.Position2D;
import aglobe.visio3D.VisioCallbackListener;
import aglobe.visio3D.VisioConnection;
import aglobe.visio3D.VisioConstants;
import aglobe.visio3D.ontology.Chart;
import aglobex.simulation.global.EntitiesPositionInfo;
import aglobex.simulation.global.EntityRecord;
import aglobex.simulation.ontology.Configuration;
import aglobex.simulation.ontology.Param;

/**
 * <p>Title: Across</p>
 * <p>Description: Simulates position of the all nodepods. Runs in own thread</p>
 * <p>Copyright: Copyright (c) 2004</p>
 * <p>Company: Gerstner Laboratory</p>
 * @author David Sislak
 * @version $Revision: 1.5 $ $Date: 2007/03/05 14:47:59 $
 */

public class Visualizer implements VisioCallbackListener, GISTopicServerListener {


  private final static String VISIO_TYPE_PARAM = "Type";

  /**
   * reference to the owner
   */
  private VisioConnectionAgent owner;

  /**
   * List of entities in the world. If null, there isn't any map information
   */
  private List<EntityRecord> entityRecords = new ArrayList<EntityRecord>();

  /**
   * Hash map of the entities where key is containerName and values are EntityRecord objects
   */
  private Map<String,EntityRecord> entityRecordByName = new HashMap<String,EntityRecord>();

  /**
   * Hash map of the entities where key is containerAddress and values are EntityRecord object.
   * Used by handleCommunicationInfo to find apropriate entities
   */
  private Map<Address,EntityRecord> entityRecordByAddress = new HashMap<Address,EntityRecord>();

    /**
   * connection to the visios
   */
  public static VisioConnection visio;

  private boolean relInfoTopicSubscribed = false;
  private HashSet<String> relInfoSubscribers = new HashSet<String>();
  private HashMap<String,LinkedList> relInfoCache = new HashMap<String,LinkedList>();
  private HashMap<String, ColumnData> checker = new HashMap<String,ColumnData>();

  /**
   * Map containing actual info (params) about agents.
   * Key = String - agent name
   * Value = Map - params
   */
  Map<String,Map<String,String>> agentInfoParams = new HashMap<String,Map<String,String>>();
  
  /**
   * World configuration
   */
  private Configuration conf;
  
  /**
   * Holds last positions of entities in the system.
   */
  private EntitiesPositionInfo lastPos;
  
  /**
   * Mapping of slot indexes (as got from the simulation) to the visio indexes.
   */
  private short[] slotIndexToVisioIndex;
  
  /**
   * Positions of objects in the visio.
   */
  private Position2D[] pos; 
  
  /**
   *
   * @param owner NodePodAgent - owner of this NodePodSimulator
   */
  private Visualizer(VisioConnectionAgent owner, Configuration conf) {
    this.owner = owner;
    this.conf = conf;
	if(slotIndexToVisioIndex==null) {
		slotIndexToVisioIndex = new short[conf.getMaxEntities()];
		Arrays.fill(slotIndexToVisioIndex,(short) -1);
	}
	pos = new Position2D[conf.getMaxEntities()];
	for (int i = 0; i < pos.length; i++) {
		pos[i] = new Position2D((short) -1,0);
		pos[i].logged = false;
	}
    try {
    	String visioConf = null;
    	for (Param param : (List<Param>) conf.getParam()) {
			if(param.getName().equals(AcrossVisioConstants.VISIO_CONFIGURATION)) {
				visioConf = param.getValue();
			}
		}
      visio = new VisioConnection(visioConf);
      visio.sendReset();

      visio.registerCallbackListener(this);
    }
    catch (Exception ex1) {
      owner.getLogger().warning(ex1.toString());
    }
  }

  /**
   * Factory method - this method ensures that there is not a partly intialized Visualizer
   * in the system.
   * @param owner
   * @param conf
   * @return
   */
  public static Visualizer newInstance(VisioConnectionAgent owner, Configuration conf) {
	  return new Visualizer(owner,conf);
  }
  
  /**
   * Adds a new entity to the visio.
   * @param er
   */
  synchronized void entityLogged(EntityRecord er) {
	  if(!entityRecords.contains(er)) {
		  entityRecords.add(er);
		  entityRecordByAddress.put(er.address, er);
		  entityRecordByName.put(er.containerName,er);
		  createEntityInVisio(er);
	  }
  }

  /**
   * Creates the entity in visio and assigns it a number (visioId).
   * @param er
   */
  private void createEntityInVisio(EntityRecord er) {
	  if(visio!=null) {
		  // TODO - visibility
		  short visInd = visio.sendCreateObject(er.entityDescriptor.confParamsString.get(AcrossVisioConstants.ENTITY_VISIO_NAME), er.entityDescriptor.confParamsString.get(AcrossVisioConstants.ENTITY_VISIO_OBJECT), er.entityDescriptor.confParamsString.get(AcrossVisioConstants.ENTITY_VISIO_TEXTURE), -1);
		  slotIndexToVisioIndex[er.indexToSlotArray]=visInd;
		  pos[visInd].logged = true;
		  pos[visInd].objectIndex = visInd;
	  }
  }
  
  /**
   * Removes an entity from the visio.
   * @param er
   */
  synchronized void entityLogout(EntityRecord er) {
  	  short ind = slotIndexToVisioIndex[er.indexToSlotArray];
	  slotIndexToVisioIndex[er.indexToSlotArray] = (short) -1;
	  entityRecords.remove(er);
	  entityRecordByAddress.remove(er.address);
	  entityRecordByName.remove(er.containerName);
	  visio.sendDestroyObject(ind);
  }
  
  /**
   * Entity crashed.
   * @param contName
   */
  synchronized void entityCrashed(String contName) {
	  final EntityRecord er = entityRecordByName.get(contName); 
	  short ind = slotIndexToVisioIndex[er.indexToSlotArray];
	  visio.sendShowAction(contName, ind, VisioConnectionAgent.ACTION_GOODS_STOLEN);
	  owner.scheduleEvent(new Runnable() {
		  
		  public void run() {
			  Visualizer.this.entityLogout(er);
		  }
		  
	  }, 1000);
  }
  
  /**
   * Updates the positions of the objects in the visio. 
   * @param epi
   */
  synchronized void positionUpdate(EntitiesPositionInfo epi) {
	  if(visio!=null) {
		  for (EntityRecord er : entityRecords) {
			 short ind = slotIndexToVisioIndex[er.indexToSlotArray];
			 if(ind!=-1) {
				 pos[ind].positionUpdated = true;
				 pos[ind].X = epi.x[er.indexToSlotArray];
				 pos[ind].Y = epi.y[er.indexToSlotArray];
				 visio.sendPostionUpdate(pos);
				 pos[ind].positionUpdated = false;
			 }
		  }
		  lastPos = epi;
	  }
  }
  
 /**
   * Handles incoming communication info. Resend it to the visio.
   * @param from Address - message sender
   * @param to Address - message receiver
   */
  synchronized void incomingCommunication(Address from, Address to) {
    EntityRecord f = entityRecordByAddress.get(from.deriveContainerAddress());
    EntityRecord t = entityRecordByAddress.get(to.deriveContainerAddress());
    if ((f != null) && (t != null) && (visio != null)) {
    	short fromInd = slotIndexToVisioIndex[f.indexToSlotArray];
    	short toInd = slotIndexToVisioIndex[t.indexToSlotArray];
    	if(fromInd!=-1 && toInd!=-1) {
    		visio.sendCommunication(fromInd,toInd,VisioConstants.COMMUNICATION_COLOR_GREEN);
    	}
    }
  }

  /**
   * Sends code transfer to visio
   * @param from String
   * @param to Address
   */
  synchronized void sendCodeTransferToVisio(String from, Address to) {
    EntityRecord f = entityRecordByName.get(from);
    EntityRecord t = entityRecordByAddress.get(to.deriveContainerAddress());
    if ((f != null) && (t != null) && (visio != null)) {
    	short fromInd = slotIndexToVisioIndex[f.indexToSlotArray];
    	short toInd = slotIndexToVisioIndex[t.indexToSlotArray];
    	if(fromInd!=-1 && toInd!=-1) {
    		visio.sendCommunication(fromInd,toInd,VisioConstants.COMMUNICATION_COLOR_VIOLET);
    	}
    }
  }

   /**
   * Handles incoming agent info from Yellow pages and other agents.
   * Updates agentInfoParams map and send actual info to the visio
   *
   * If Param.getValue is empty String then the old parameter is only removed.
   *
   * @param page Page - information about agent
   */

  synchronized void incomingAgentInfo(Page page) {

      Address entityAddress = page.getAddress().deriveContainerAddress();
      EntityRecord en = entityRecordByAddress.get(entityAddress);
      
      if ( (en != null) && (visio != null)) {
    	  
    	  // parameters to be sent to visio
          Map<String,String> params = agentInfoParams.get(page.getName());
          if (params == null) { // if this is first agentInfo about this agent
              params = new HashMap<String,String>();
              agentInfoParams.put(page.getName(), params);
          }
          
          // public parameters sent with the page
          PublicParams pubPar = page.getPublicParams();
          
          if (!pubPar.getType().equals("")) {
              params.put(VISIO_TYPE_PARAM, pubPar.getType());
          }

          for (across.data.Param param : pubPar.getParam()) {
              if (!param.getValue().equals("")) {
                  params.put(param.getName(), param.getValue());
              } else {
                  params.remove(param.getName());
              }
          }
          short ind = slotIndexToVisioIndex[en.indexToSlotArray];
          if(ind!=-1) {
        	  visio.sendAgentInfo(page.getName(),ind,params);
          }
      }
  }

  /**
   * incomingAgentInfoRemoved
   *
   * @param agentAddress Address
   */
  synchronized void incomingAgentInfoRemoved(Address agentAddress) {
    Address entityAddress = agentAddress.deriveContainerAddress();
    EntityRecord en = (EntityRecord) entityRecordByAddress.get(entityAddress);
    if ( (en != null) && (visio != null)) {
     agentInfoParams.remove(agentAddress.getName());
     visio.sendRemoveAgent(agentAddress.getName());
     owner.getLogger().fine("Send agent info removed: "+agentAddress);
   }
  }

  /**
   * Sets new visio info box text for the apropriate entity
   * @param entityName String
   * @param newText String
   */
  synchronized void updateVisioInfo(String entityName, String newText) {
    EntityRecord en = entityRecordByName.get(entityName);
    if (en != null) {
      if (visio != null) {
    	  short ind = slotIndexToVisioIndex[en.indexToSlotArray];
          if(ind!=-1) {
        	 visio.sendObjectInfo(ind,newText);
          }
      }
    } else {
      owner.getLogger().warning("Receiving update visio text for unknown entity name: "+entityName);
    }
  }

  /**
   * updateVisioInfo
   *
   * @param entityName String
   * @param load byte
   */
  synchronized void updateVisioInfo(String entityName, byte load) {
    EntityRecord en = (EntityRecord)entityRecordByName.get(entityName);
    if ((en != null)) {
      if (visio != null) {
    	  short ind = slotIndexToVisioIndex[en.indexToSlotArray];
          if(ind!=-1) {
        	  visio.sendSpecialCarLoad(ind,load);
          }
      }
    } else {
      owner.getLogger().warning("Receiving load update for unknown entity name: "+entityName);
    }
  }

  /**
   * setNewMapArcs
   *
   * @param entityName String
   * @param agentAddress Address
   * @param action byte
   */
  synchronized void showVisioAction(String entityName, Address agentAddress,
                     byte action) {
    EntityRecord en = (EntityRecord)entityRecordByName.get(entityName);
    if ((en != null)) {
      if (visio != null) {
    	  short ind = slotIndexToVisioIndex[en.indexToSlotArray];
          if(ind!=-1) {
        	  visio.sendShowAction(agentAddress.getName(),ind,action);
          }
      }
    } else {
      owner.getLogger().warning("Receiving load update for unknown entity name: "+entityName);
    }
  }

  /**
   * Create/Update chart in the visio
   * @param entityName String
   * @param chartName String
   * @param chartDefinition Chart
   */
  synchronized void visioSetChart(String entityName, String chartName, Chart chartDefinition) {
    EntityRecord en = (EntityRecord)entityRecordByName.get(entityName);
    if ((en != null)) {
      if (visio != null) {
    	  short ind = slotIndexToVisioIndex[en.indexToSlotArray];
          if(ind!=-1) {
        	  visio.sendSetChart(ind,chartName,chartDefinition);
          }
      }
    } else {
      owner.getLogger().warning("Receiving set chart for unknown entity name: "+entityName);
    }
  }

  /**
   * Remove chart in the visio
   * @param entityName String
   * @param chartName String
   */
  synchronized void visioRemoveChart(String entityName, String chartName) {
    EntityRecord en = (EntityRecord)entityRecordByName.get(entityName);
    if ((en != null)) {
      if (visio != null) {
    	  short ind = slotIndexToVisioIndex[en.indexToSlotArray];
          if(ind!=-1) {
        	  visio.sendRemoveChart(ind,chartName);
          }
      }
    } else {
      owner.getLogger().warning("Receiving remove chart for unknown entity name: "+entityName);
    }
  }

  public void visioPlanStart(short objectIndex) {
  }

  public void visioPlanStop(short objectIndex) {
  }

  public void visioSetSimulationSpeed(short speed) {
  }

  public void visioReceivedScreenShotData(short screenShotID, BufferedImage screenShot) {
  }

  public void visioRelInfoStart(String agentName, short containerID) {

    if (!relInfoTopicSubscribed) {
      relInfoTopicSubscribed = true;
     // petr relInfoCache.clear();
      owner.gisShell.subscribeTopic(VisioConnectionAgent.TOPIC_VISIO_REL_INFO,this);
      owner.gisShell.broadcastTopic(VisioConnectionAgent.TOPIC_VISIO_REL_INFO,"SEND UPDATE PLEASE");
    }
    if (relInfoSubscribers.contains(agentName+containerID)) {
      // nothig happens, this situation should not occur
      return;
    } else {
      relInfoSubscribers.add(agentName+containerID);
      sendRelInfoToVisio(agentName,containerID);
    }

  }

  public void visioRelInfoEnd(String agentName, short containerID) {
    if (!relInfoSubscribers.contains(agentName+containerID)) {
      // nothig happens, this situation should not occur
      return;
    } else {
      relInfoSubscribers.remove(agentName+containerID);
    }

  }

  /**
   * parseRelInfoUpdate
   *
   * @param content RelationInfo
   */

  private synchronized void parseRelInfoUpdate(RelationInfo relInfoUpdate, String remoteContainerName) {

      LinkedList curRelInfo = relInfoCache.get(relInfoUpdate.getOwnerAgentName());

      owner.logSevere(relInfoUpdate.toString());

      if (curRelInfo == null) {
        curRelInfo = new LinkedList();
        relInfoCache.put(relInfoUpdate.getOwnerAgentName(), curRelInfo);
      }

      for (Iterator iter = relInfoUpdate.getColumnData().iterator(); iter.hasNext(); ) {
        ColumnData item = (ColumnData) iter.next();

        if (checker.get(relInfoUpdate.getOwnerAgentName() + item.getColumnTitle()) != null) {
          //column is already in database

          int pos = -1;
          int position = -1;
          for (Iterator iterator = curRelInfo.iterator(); iterator.hasNext(); ) {
            pos++;
            ColumnData cd = (ColumnData) iterator.next();
            if (cd.getColumnTitle().equals(item.getColumnTitle())) {
              // comumn was found in the list
              position = pos;
              break;
            }
          }

          if (position == -1) {
            System.out.println("conceptual error in column cache");
          }
          curRelInfo.remove(position);

          curRelInfo.add(position, item);
        }
        else {
          curRelInfo.add(item);

        }
        checker.put(relInfoUpdate.getOwnerAgentName() + item.getColumnTitle(),
                    item);
      }


// send info only if it should be displayed

      sendRelInfoToVisio(relInfoUpdate.getOwnerAgentName(),slotIndexToVisioIndex[entityRecordByName.get(remoteContainerName).indexToSlotArray]);

  }

  private synchronized void sendRelInfoToVisio(String agentName,short visioObjectID) {
      List curRelInfo = relInfoCache.get(agentName);
       if (curRelInfo == null) {
         return;
       }

       if (visio != null) {
         int colNum = curRelInfo.size();

         String[] aColumnTitle = new String[colNum];
         String[][] aAgentName = new String[colNum][];
         short[][] aContainerID = new short[colNum][];
         String[][] aInfoString = new String[colNum][];
         byte[][] infoColor = new byte[colNum][];

         int i = 0;

         for (Iterator iter  = curRelInfo.iterator(); iter.hasNext(); ) {

           ColumnData column = (ColumnData)iter.next();
           int agtNum = column.getAgentData().size();

           aAgentName[i] = new String[agtNum];
           aContainerID[i] = new short[agtNum];
           aInfoString[i] = new String[agtNum];
           infoColor[i] = new byte[agtNum];

           aColumnTitle[i] = column.getColumnTitle();
           int j = 0;
           for (AgentData elem1 : (List<AgentData>)column.getAgentData()) {
             aAgentName[i][j] = elem1.getAgentName();
             aContainerID[i][j] = slotIndexToVisioIndex[entityRecordByName.get(elem1.getContainerName()).indexToSlotArray];
             aInfoString[i][j] = elem1.getInfoString();
             infoColor[i][j] = elem1.getInfoColor();
             j++;
           }
           i++;
         }


         visio.sendRelationInfoUpdate(visioObjectID,
                                      agentName,
                                      aColumnTitle,
                                      aAgentName,
                                      aContainerID,
                                      aInfoString,
                                      infoColor);
       }
  }

  
  private HashMap<String, HashSet<Long>> contracts = new HashMap<String, HashSet<Long>>();
  private HashMap batchIdToRequestId = new HashMap();
  private final int contractsToDisplay = 15;
  private LinkedList<String> contractIdsForDisplay = new LinkedList<String>();
  
  
  private HashMap<String, LinkedList<Batch>> batchInContainer = new HashMap<String, LinkedList<Batch>>();

  private short humagentContainerId;
  private Address humagentAddress = null;
  private String humagentContainer = null;

  private boolean contractsIdsVisible = false;
  private String nowShowing = null;
  
  public synchronized void visioSendRequestsIDsStart() {
    contractsIdsVisible = true;
    visioSendRequestsIDs();
    if (nowShowing != null) {
        displayBatchList();
    }
  }
  
  public synchronized void visioSendRequestsIDsEnd() {
      contractsIdsVisible = false;
  }
  
  public synchronized void visioSendRequestsIDs() {
     visio.sendShowRequestIDs(humagentContainerId, contractIdsForDisplay);
  }
  
  public synchronized void setHumanitarianContainer(Address humAddress, String humContainer) {
    humagentContainerId = slotIndexToVisioIndex[entityRecordByName.get(humContainer).indexToSlotArray]; 
    humagentAddress = humAddress;
    humagentContainer = humContainer;
  }
  
  public synchronized void visioBatchInfoStart(String requestID) {
      nowShowing = requestID;
      displayBatchList();
  }
  
  public void visioBatchInfoEnd(String requestID) {
      nowShowing = null;
  }

  private void fillText(String containerName) {
      String text="";
      int rows = 0;
      
      LinkedList visibleList = batchInContainer.get(containerName);
      
      if (visibleList != null) {
          if (visibleList.size() != 0) {
              for (Iterator i = visibleList.iterator(); i.hasNext(); ) {
                  Batch batch = (Batch)i.next();
                  String requestID = (String)batchIdToRequestId.get(batch.getBatchid());
                  if (requestID != null) {
                      if (requestID.equals(nowShowing)) {
                          text += batch.getComodityName()+" ("+batch.getCount()+")<n>";
                          rows++;
                      }
                  }
              }
              if (rows != 0) {
                  text = "<b>"+containerName+":<n><r>"+text;
              }
          }
      }
      short ind = slotIndexToVisioIndex[entityRecordByName.get(containerName).indexToSlotArray];
      if(ind!=-1) {
    	  visio.sendBatchInfo(ind, text);
      }
  }
  
private void fillStolenBatches(Position pos, LinkedList visibleList) {
      String text="";
      int rows = 0;

      if (visibleList != null) {
          if (visibleList.size() != 0) {
              for (Iterator i = visibleList.iterator(); i.hasNext(); ) {
                  Batch batch = (Batch)i.next();
                  text = text+batch.getComodityName()+" ("+batch.getCount()+")<n>";
                  rows++;
              }
              if (rows != 0) {
                  text = "<b>stolen <r>from <b>"+pos.containerName+":<n><r>"+text;
              }
              visio.sendBatchInfo(pos.x,pos.y, text);
          }
      }
  }  
  

  private void displayBatchList() {
      Set keys = batchInContainer.keySet();
      
      for (Iterator i = keys.iterator(); i.hasNext(); ) {
          String container = (String)i.next();
          fillText(container);
      }
      
      //fillStolenBatches
      
      for (Iterator i = leftBatches.iterator(); i.hasNext(); ) {
          Position pos = (Position)i.next();
          LinkedList completeBatchList = pos.batchList;
          LinkedList stolenToDisplay = new LinkedList();
          for (Iterator j = completeBatchList.iterator(); j.hasNext();) {
              Batch b = (Batch)j.next();
              String reqId = (String)batchIdToRequestId.get(b.getBatchid());
              if (reqId != null) {
                  if (reqId.equals(nowShowing)) {
                      stolenToDisplay.add(b);
                  }
              }
          }
          fillStolenBatches(pos, stolenToDisplay);
          
      }   
  }
  
  
  private class Position {
      public double x;
      public double y;
      public String containerName;
      public LinkedList batchList;
              
      public Position(double x, double y, String containerName, LinkedList batchList) {
          this.x = x;
          this.y = y;
          this.containerName = containerName;
          this.batchList = batchList;
      }
  }
  
  private LinkedList leftBatches = new LinkedList();
  
  
  private LinkedList convertBatches(LinkedList batches) {
      LinkedList val = new LinkedList();
      for (Iterator i = batches.iterator(); i.hasNext(); ) {
          val.addLast(convertLoadBatch((Batch)i.next()));
      }
      return val;
  }
  
  public synchronized void batchesLeft(LinkedList stolenBatches, String containerName) {
      stolenBatches = convertBatches(stolenBatches);
      
      EntityRecord entity = (EntityRecord) entityRecordByName.get(containerName);
      if (entity == null) {
          owner.logSevere("Unknown container: "+containerName);
          return;
      }
      Position pos = new Position(lastPos.y[entity.indexToSlotArray],lastPos.x[entity.indexToSlotArray], containerName, stolenBatches);
      leftBatches.add(pos);
      
   LinkedList batchesInContainer = batchInContainer.get(containerName);
   
   for (Iterator j = stolenBatches.iterator(); j.hasNext();) {
       Batch b = (Batch)j.next();
       
      Batch unloadedBatch = null;
      if (batchesInContainer != null) {
          for (Iterator i = batchesInContainer.iterator(); i.hasNext(); ) {
              Batch b2 = (Batch)i.next();
              if ((b2.getBatchid() == b.getBatchid()) && (b2.getCount() == b.getCount())) {
                  unloadedBatch = b2;
                  break;
              }
          }
          if (unloadedBatch != null)   {
              batchesInContainer.remove(unloadedBatch);
          } else {
              System.out.println("no such batch error");
          }
      }      
      
   } 
   displayBatchList();
     
      
  }
  
  
  public synchronized void addNewContract(TransportCfp tcfp) {

	  if (!contractIdsForDisplay.contains(tcfp.getRequestid())) {
          contractIdsForDisplay.addLast(tcfp.getRequestid());
          if (contractIdsForDisplay.size() > contractsToDisplay) {
              contractIdsForDisplay.removeFirst();
              contracts.remove(tcfp.getRequestid());
          }
          if (contractsIdsVisible) {
              visioSendRequestsIDs();
          }
      }
 
    HashSet<Long> batches = contracts.get(tcfp.getRequestid());
    if (batches == null) {
        batches = new  HashSet<Long>();
        contracts.put(tcfp.getRequestid(), batches);        
    }    
        
    for (Iterator i=tcfp.getTransportBatch().iterator(); i.hasNext(); ) {
      TransportBatch b2 = (TransportBatch)i.next();
      Batch b = new Batch();
      b.setBatchid(b2.getBatchid());
      b.setComodityName(b2.getComodityName());
      b.setCount(b2.getCount());
      
      b = convertLoadBatch(b);
      
      batchIdToRequestId.put(b.getBatchid(), tcfp.getRequestid());
      batches.add(b.getBatchid());
      if (humagentAddress != null) {
          if (humagentAddress.equals(b2.getStartAddress())) {
              LinkedList<Batch> bts = batchInContainer.get(humagentContainer);
              if (bts == null) {
                  bts = new LinkedList<Batch>();
                  batchInContainer.put(humagentContainer, bts);
              }
              bts.add(b);
          }
      }
      
    }
    displayBatchList();
  }
  
 
  public synchronized void addNewContract(List batchList, String requestID) {

      if (!contractIdsForDisplay.contains(requestID)) {
          contractIdsForDisplay.addLast(requestID);
          if (contractIdsForDisplay.size() > contractsToDisplay) {
              contractIdsForDisplay.removeFirst();
              contracts.remove(requestID);
          }
          if (contractsIdsVisible) {
              visioSendRequestsIDs();
          }
      }      
 
    HashSet<Long> batches = contracts.get(requestID);
    if (batches == null) {
        batches = new  HashSet<Long>();
        contracts.put(requestID, batches);        
    }    
        
    for (Iterator i= batchList.iterator(); i.hasNext(); ) {
      TransportBatch b2 = (TransportBatch)i.next();
      Batch b = new Batch();
      b.setBatchid(b2.getBatchid());
      b.setComodityName(b2.getComodityName());
      b.setCount(b2.getCount());
      
      b = convertLoadBatch(b);
      
      batchIdToRequestId.put(b.getBatchid(), requestID);
      batches.add(b.getBatchid());
      if (humagentAddress != null) {
          if (humagentAddress.equals(b2.getStartAddress())) {
              LinkedList<Batch> bts = batchInContainer.get(humagentContainer);
              if (bts == null) {
                  bts = new LinkedList<Batch>();
                  batchInContainer.put(humagentContainer, bts);
              }
              bts.add(b);
          }
      }
      
    }
    displayBatchList();
  }

  
  public synchronized void batchLoadedOnContainer(Batch b, String containerName) {
    b = convertLoadBatch(b);
      
    System.out.println("LOADED BATCH ON "+containerName+" "+b.toString());  
      
    String request = (String)batchIdToRequestId.get(b.getBatchid());
    LinkedList batchesInContainer = batchInContainer.get(containerName);
    if (batchesInContainer == null) {
        batchesInContainer = new LinkedList<Batch>();
        batchInContainer.put(containerName, batchesInContainer);
    }
    batchesInContainer.add(b);
    displayBatchList();
  }
  
  private Batch convertLoadBatch(Batch b) {
      Batch val = new Batch();
      val.setCount(Math.abs(b.getCount()));
      val.setBatchid(b.getBatchid());
      val.setComodityName(b.getComodityName());
      return val;      
  }
  
  
  private Batch convertUnloadBatch(Batch b) {
      Batch val = new Batch();
      val.setCount(-Math.abs(b.getCount()));
      val.setBatchid(b.getBatchid());
      val.setComodityName(b.getComodityName());
      return val;      
  }
  
  
  public synchronized void batchUnloadedFromContainer(Batch b, String containerName) {
    b = convertUnloadBatch(b);
    String request = (String)batchIdToRequestId.get(b.getBatchid());
    LinkedList batchesInContainer = batchInContainer.get(containerName);
    if (batchesInContainer == null) {
        batchesInContainer = new LinkedList<Batch>();
        batchInContainer.put(containerName, batchesInContainer);
    }
    batchesInContainer.add(b);
    displayBatchList();
      
  }    
      
  /**
   * Called by GISServer service to infrom that some client service subscribe
   * for specific topic info.
   *
   * @param topic String - topic name
   * @param remoteContainerName String - remote container name, if null it is
   *   local subscriber
   * @param remoteContainerAddress Address - remote container address, if null
   *   it is local subscriber
   * @todo Implement this aglobe.service.gis.server.GISTopicServerListener
   *   method
   */
  public void handleLoginTopic(String topic, String remoteContainerName,
                               Address remoteContainerAddress) {
  }

  /**
   * Called by GISServer service to infrom that some client service unsubscribe
   * for specific topic info.
   *
   * @param topic String - topic name
   * @param remoteContainerName String - remote container name, if null it is
   *   local subscriber
   * @todo Implement this aglobe.service.gis.server.GISTopicServerListener
   *   method
   */
  public void handleLogoutTopic(String topic, String remoteContainerName) {
  }

  /**
   * Called by the GISServer service to distribute information received from
   * entities on client containers.
   *
   * @param topic type of the information (Ex. WEATHER)
   * @param content content of the information (Ex. 10 degree, rain,
   *   70%humidity, tornado,...)
   * @param reason String
   * @param remoteContainerName - name of the remote container who send the
   *   topic, null - topic is send from the server
   * @param remoteClientAddress address of the GISclient service, null - topic
   *   is send from the server
   * @todo Implement this aglobe.service.gis.server.GISTopicServerListener
   *   method
   */
  public void handleTopic(String topic, Object content, String reason,
                          String remoteContainerName,
                          Address remoteClientAddress) {
	  if (topic.equals(VisioConnectionAgent.TOPIC_VISIO_REL_INFO)) {

		  if (remoteContainerName != null) {
			  parseRelInfoUpdate((RelationInfo) content, remoteContainerName);
		  }
		  return;
	  }
    owner.logSevere("Unexpected incoming topic: "+topic);
  }

  /**
   *
   * @param e Runnable
   * @todo Implement this aglobe.container.EventReceiver method
   */
  public void addEvent(Runnable e) {
    owner.addEvent(e);
  }
 
  /**
   * Don't use this!
   */
  public void visioAircraftSelected(short containerID) {
	throw new RuntimeException("Don't use me!");
  }
  
}
