package across.util.skn.util.membership;

import java.util.ArrayList;

import across.util.skn.util.FuzzySet;

/**
 * This class implements the membership for type 2 fuzzy sets.
 *
 * In this case, the membership itself is a fuzzy set. It is defined by points and we assume the membership to be linear between these points.
 * Fuzzy number:
 *  - is defined on the interval [0,1]
 *  - must have non-empty core
 *  - must be convex
 *
 * Due to the lack of multiple inheritance in java, this class must be derived from IterativeFuzzyMembership. However,
 * the function that mapps iterative properties to the fuzzy number shape is not implemented.
 */
public class FuzzyNumberMembership<MemberObject> extends IterativeFuzzyMembership<MemberObject>{

    //stores the points defining the fuzzy number
    protected ArrayList<Point> points = new ArrayList<Point>();

    public FuzzyNumberMembership(MemberObject o, FuzzySet set, Point... shape) {
        super(o, set);
        if (null != shape) {
            for (Point cpoint : shape) {
                points.add(cpoint);
            }
        }
    }
    public FuzzyNumberMembership(MemberObject o, FuzzySet set, IterativeFuzzyMembership<MemberObject> original) {
    	super(o,set,original);
	}
	/**
     * We don't do anything here. This function is to be redefined by derived classes.
     */
    public void applyChangesToShape()
    {
        //To change body of implemented methods use File | Settings | File Templates.
    }

    /**
     * Returns true if incidence with the set is nonzero, i.e. if membership function is not zero for this member.
     * If there is
     */
    public boolean isMember() {
        if (points.size() > 0)
        {
            for (Point point : points) {
                if (point.getX() > 0 && point.getY() > 0) return true;
            }
        }
        return false;
    }

    /**
     * Returns true if incidence with the core of the set is nonzero, i.e. if membership function is maximum, typically 1
     * For fuzzy numbers, returns true if the memebrship value of 1 in the memebrship function is 1.
     * It means that the last point must be (1,1)
     */
    public boolean isCoreMember() {

        if (points.size() > 0 && points.get(points.size()-1).getX() == 1 && points.get(points.size()-1).getY() == 1)
        {
            return true;
        }
        return false;
    }

    /**
     * Returns the center of the membership function.
     * @return center of the function, shape specific. Bounded to [0,1] interval.
     */
    public double getMembershipFunctionCenter() {
        return getCoreCenter();
    }

    /**
     * Returns the uncertainity of the membership function.
     */
    public double getMembershipFunctionUncertainity() {
        return getTotalSurface();
    }

    public double getCenter()
    {
        return getCoreCenter();
    }

    public double getCoreCenter()
    {
        Point coreStart = null;
        Point coreEnd = null;
        for (Point point : points) {
            if (null == coreStart && point.getY() == 1)
            {
                coreStart = point;
                coreEnd = point;
            }
            else if (coreEnd != null && point.getY() == 1)
            {
                coreEnd = point;
            }
            else if (coreEnd != null && point.getY() < 1) break;
        }
        if (null != coreEnd) return (coreEnd.getX() - coreStart.getX())/2 + coreStart.getX();
        return 0;
    }

    /**
     * Returns the value of the minimum memebership- if X is not 0, then y is 0
     * @return x - minimum point.
     */
    public Point getMin()
    {
        // we assume that the points structure is ordered
        if (points.size() > 0)
        {
            return points.get(0);
        }
        return new Point(0,0);
    }
    /**
     * Returns the value of the maximum membership- if X is not 1, then y is 0
     * @return x - maximum point.
     */
    public Point getMax()
    {
        // we assume that the points structure is ordered
        if (points.size() > 0)
        {
            return points.get(points.size() -1);
        }
        return new Point(0,0);
    }

    /**
     * Returns the total surface under the membership function.
     * Roughly, bigger the size, bigger the uncertainity...
     * @return total surface under the membership function
     */
    public double getTotalSurface()
    {
        return getSurfaceUnder(1);
    }

    /**
     * Returns the total surface under the membership function cut at the selected level...
     * @param limit X axis limit. we do count only the surface under the limit
     * @return total surface under the membership function cut at the given level
     */
    public double getSurfaceUnder(double limit)
    {
        double surface = 0;
        Point last = null;
        for (Point point : points) {
            // skip the first point
            if (null != last) {
                surface += computeSurface(last, point, limit);
                if (point.getX() >= limit) break;
            }
            last = point;
        }
        return surface;
    }

    /**
     * @param limit X axis limit. we do count only the surface under the limit
     * @return pair of double, x value is asurface under the limit, y is surface above the limit
     */
    public PairDouble getSurfaceUnderAbove(double limit)
    {
        double surfaceUnder = 0;
        double surfaceAbove = 0;
        Point last = null;
        double surf, sl;
        for (Point point : points) {
            // skip the first point
            if (null != last) {
                surf = computeSurface(last, point);
                sl = computeSurface(last, point, limit);
                surfaceUnder += sl;
                surfaceAbove += surf - sl;
            }
            last = point;
        }
        return new PairDouble(surfaceUnder, surfaceAbove);
    }

    /** Calculates the surface between the y=0 and the line determined by two points.*/
    private static double computeSurface(Point left, Point right)
    {
        if (left.getX() > right.getX())
        {
            return computeSurface(right, left);
        }
        double minp = Math.min(left.getY(), right.getY());
        double maxp = Math.max(left.getY(), right.getY());
        double res = (minp + (maxp - minp)/2) * (right.getX() - left.getX());
        return res;
    }

    /** Calculates the surface between the y=0 and the line determined by two points to the left of the x=limit.*/
    private static double computeSurface(Point left, Point right, double limit)
    {
        if (left.getX() > right.getX())
        {
            return computeSurface(right, left, limit);
        }
        if (limit < left.getX()) {return 0;}
        if (limit > right.getX()) {return computeSurface(left, right);}
        double limVal = left.getY() + (right.getY() - left.getY())*(limit-left.getX());
        return computeSurface(left, new Point(limit, limVal));
    }

    private static double getLinearFunctionValueAtPoint(Point left, Point right, double xValue)
    {
        if (left.getX() > right.getX())
        {
            return getLinearFunctionValueAtPoint(right, left, xValue);
        }
        if (left.getX() <= xValue && xValue <= right.getX()) {
            if ((right.getX() - left.getX()) < 0.0000001)
            {
                return Math.max(left.getY(), right.getY());
            } 
            return left.getY() + ((right.getY() - left.getY())/(right.getX() - left.getX()))*(xValue-left.getX());
        }
        return 0;
    }



    private static Point getMaxCrossing(Point left1, Point right1, Point left2, Point right2)
    {
        // a1 is lower, to avoid degenerated interval /0 problem
        double b1 = left1.getY();
        double c1 = left1.getX();
        // a2 is lower, to avoid degenerated interval /0 problem
        double b2 = left2.getY();
        double c2 = left2.getX();
        // check for zero length intervals (peaks)...
        // first and both first...
        if (left1.getX() == right1.getX())
        {
            if (left2.getX() == right2.getX())
            {
                // two peaks, check if it is the same x coordinate
                if (c1 == c2)
                {
                    return new Point(c1, Math.min(Math.max(b1, right1.getY()), Math.max(b2, right2.getY())));
                }
                return null; // no overlap possible...
            }
            // first peak, second not
            if ( c2 <= c1 && c1 <= right2.getX())
            {
            	return new Point(c1, Math.min(getLinearFunctionValueAtPoint(left2, right2, c1), Math.max(b1, right1.getY())));
            }
            return null; // no overleap, peak outside the second interval
        }
        // second next - no check for both, covered by the first case...
        if (left2.getX() == right2.getX())
        {
            // second peak, first interval
            if ( c1 <= c2 && c2 <= right1.getX())
            {
                return new Point(c1, Math.min(getLinearFunctionValueAtPoint(left1, right1, c1), Math.max(b2, right2.getY())));
            }
            return null; // no overleap, peak outside the second interval
        }
        // determine the a now, when ther is no /0 threat
        double a1 = (right1.getY() - left1.getY()) / (right1.getX() - c1);
        double a2 = (right2.getY() - left2.getY()) / (right2.getX() - c2);
        // parallel lines handling
        if (a1 == a2)
        {
            // same line
            if (b1 - a1*c1 == b2 - a2*c2)
            {
                if (left1.getY() > right1.getY()) return left1;
                return right1;
            }
            return null;// no intersection
        }
        // intersection exists
        double resx = ( a1*c1 - a2*c2 - b1 + b2)/(a1-a2);
        // we return the result only if it falls between the left1 and right1
        if ( left1.getX() <= resx && right1.getX() >= resx && left2.getX() <= resx && right2.getX() >= resx)
        {
            double y1 = getLinearFunctionValueAtPoint(left1, right1, resx);
            double y2 = getLinearFunctionValueAtPoint(left2, right2, resx);
            // check for the floating point arithemetics errors for functions that are too steep
            if (Math.abs(a1) > 1E6 || Math.abs(a2) > 1E6)
            {
                if (Math.abs(a2) < 1E6)
                {
                    y1 = y2;// use the value that is not on a vertical segment... intersection with the perpendicular}
                }
                else if (Math.abs(a1) < 1E6)
                {
                    y2 = y1;
                }
                else
                {
                    // two steep lines
                    y1 = Math.min(Math.max(left2.getY(), right2.getY()), Math.max(left2.getY(), right2.getY()));
                    y2 = y1;
                }
            }

            if (!(Math.abs(y1-y2) < 0.00002))
            {
                System.out.println("Bad EQUATION: " + left1 + right1 + left2 + right2 + " res: " + resx + y1 + y2);
                System.out.println("Bad EQUATION details: " + left1 + right1 + left2 + right2 + " res: " + resx + " a1: " +a1 + " a2: " + a2);
            }
            //System.out.println("eq. inputs: "+ " " + a1 + " " + b1+ " " +c1+ " " +a2+ " " +b2+ " " +c2+ " " + resx + " " + y1 + " " + y2 + left1 + right1 + left2 + right2 );
            return new Point(resx, y1);// watch out for previous y1 modifs...
        }
        return null;
    }

    /**
     * Returns a height of an intersection of two normal fuzzy numbers.
     * @param other the number to make intersection with
     * @return height of the intersection
     */
    public double getIntersectionHeight(FuzzyNumberMembership other)
    {
        ArrayList<Point> opoints = other.points;// this is introduced to solve the java bug
        // determine the boundaries of the intersection
        double maxLeft = Math.max(points.get(0).getX(), opoints.get(0).getX());
        double minRight = Math.min(points.get(points.size()-1).getX(), opoints.get(other.points.size()-1).getX());

        double sup = 0.0;// current intersection supremum
        Point left = null;

        for (Point right : points) {
            // skip the first point
            if (null != left) {
                // check whether we are in the intersection zone
                if (right.getX() < maxLeft) {continue;}
                if (left.getX() > minRight) {continue;}
                sup = Math.max(sup, other.getMaxCrossingWithLine(left, right));// we call the function due to the list passing bug
                if (1== sup) return 1.0;
                // TODO - not important now, but optimize later to avoid double cycle
            }
            left = right;
        }
        return sup;
    }

    private double getMaxCrossingWithLine(Point left, Point right)
    {
        double sup = 0;
        Point oleft = null;
        for (Point oright : points) {
            if (null != oleft)
            {
                // check whether the two sections do have non-void intersection
                if (left.getX() > oright.getX()) {continue;}
                if (right.getX() < oleft.getX()) {continue;}
                // find the line intersection in the section delimited by left, right
                if (left.getX() > right.getX() )
                {
                    System.out.println( "######## Left on the Right: l: " + left + " r: " + right);
                }
                if (oleft.getX() > oright.getX() )
                {
                    System.out.println( "######## Left on the Right: l: " + oleft + " r: " + oright + " in: " + this);
                }
                Point cross = getMaxCrossing(left, right, oleft, oright);
                if (null != cross)
                {
                    sup = Math.max(sup, cross.getY());
                    if (1== sup) return 1;
                }
            }
            oleft = oright;
        }
        return sup;
    }

    /** Returns the intersection with single value - membership value of the memebrship function at this point. */
    public double getIntersectionHeight(SimpleFuzzyMembership other)
    {
        Point last = null;
        for (Point point : points) {
            // skip the first point
            if (null != last) {
                if (last.getX() <= other.membership && point.getX() >= other.membership)
                {
                    return getLinearFunctionValueAtPoint(last, point, other.membership);
                }
            }
            last = point;
        }
        return 0;
    }

    /**
     * Compares this Fuzzy number whose X values were shifted by multiplication with coeff with the other one.
     * Note that by multiplication with coeff smaller than 1 we shrink the uncertainity and vice versa.
     * @param coeff coefficient used to multiply the x values of the current number
     * @param other membership function to compare with
     * @return 1 if bigger than other, -1 if smaller than other, 0 if incomparable or equivalent
     */
    public int compareWithCoefficient(double coeff, IterativeFuzzyMembership other )
    {
        FuzzyNumberMembership<MemberObject> shifted = shiftByMultiplication(coeff);
        //return shifted.compareTo((FuzzyNumberMembership)other);
        return shifted.compareTo(other);
    }

    /**
     * Shifts the fuzzy number X values by multiplying them with coefficient.
     * @param coeff coefficient used to multiply the x values of the current membership function
     * @return new FuzzyNumber - references the same MemberObject instance, but NO SET
     */
    protected FuzzyNumberMembership<MemberObject> shiftByMultiplication(double coeff)
    {
        ArrayList<Point> np = new ArrayList<Point>();
        for (Point point : points) {
            np.add(new Point(point.getX()*coeff, point.getY()));
        }
        Point[] npa = null;
        np.toArray(npa);
        return new FuzzyNumberMembership<MemberObject>(member, null, npa);
    }

    /**
     *
     * @param o the Object to be compared.
     * @return a negative integer, zero, or a positive integer as this object
     *         is less than, equal to, or greater than the specified object.
     * @throws ClassCastException if the specified object's type prevents it
     *                            from being compared to this Object.
     */
    public int compareTo(Object o) {
        if (null == o)
        {
            return 1;// null is smaller than anything else...
        }
        if (o instanceof SimpleFuzzyMembership)
        {
            SimpleFuzzyMembership<MemberObject> sfm = (SimpleFuzzyMembership<MemberObject>) o;
            if ( sfm.getMembershipFunctionCenter() <= getCenter()) { return 1;}
            return -1;
        }
        if (o instanceof FuzzyNumberMembership)
        {
            FuzzyNumberMembership<MemberObject> fnm = (FuzzyNumberMembership<MemberObject>) o;
            if (fnm.getMax().getX() < getMin().getX()) { return 1;}
            if (fnm.getMin().getX() > getMax().getX()) { return -1;}
            // check if there is more of the others surface under my center than vice-versa
            double othersSurfaceUnder = fnm.getSurfaceUnder(getCenter());
            double mySurfaceUnder = getSurfaceUnder(fnm.getCenter());
            if (othersSurfaceUnder > mySurfaceUnder)
            {
                return 1;
            }
            else if (mySurfaceUnder == othersSurfaceUnder)
            {
                return 0;
            }
            else
            {
                return -1;
            }
        }
        throw new ClassCastException("Can not compare with this class: " + o.getClass().getName());
    }

}


