package across.util.skn.util;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;

import across.util.skn.util.membership.Membership;

/**
 * This class represents a set of entities with membership values.
 * MemberObject parameter determines the type of the members of the set.
 */
public class FuzzySet<MemberObject> {

    protected ArrayList<Membership<MemberObject>> members = new ArrayList<Membership<MemberObject>>();

    public void removeAllMembers() {
      members = new ArrayList<Membership<MemberObject>>();
    }

    /**
     * Returns all members of the set, with any membership function > 0.
     * Implements o level alpha cut.
     * @return List of MemberObjects
     */

    public List<MemberObject> getAllMembers()
    {
        List<MemberObject> res =  new LinkedList<MemberObject>();
        for (Membership<MemberObject> member : members) {
            // check whether the subject belongs at least a bit to the set
            if (member.isMember()) {
                res.add(member.getMember());
            }
        }
        return res;
    };

    /**
     * Returns all members of the set core, with any membership function == 1.
     * Implements 1 level alpha cut.
     * @return List of MemberObjects
     */
    public List<MemberObject> getAllCoreMembers()
    {
        List<MemberObject> res =  new LinkedList<MemberObject>();
        for (Membership<MemberObject> member : members) {
            // check whether the subject belongs to the core of the set
            if (member.isCoreMember()) {
                res.add(member.getMember());
            }
        }
        return res;
    };

    /**
     * Returns all members of the set, with any membership function > alphaLevel 1.
     * Implements general level alpha cut.
     * @param alphaLevel membership in the set must be above this to belong to the set
     * @return List of MemberObjects
     */
    public List<MemberObject> getAllMembersAbove(Membership<MemberObject> alphaLevel)
    {
        List<MemberObject> res =  new LinkedList<MemberObject>();
        for (Membership<MemberObject> member : members) {
            // check whether the subject belongs at least a bit to the set
            if (0 < member.compareTo(alphaLevel)) {
                res.add(member.getMember());
            }
        }
        return res;
    };
    /**
     * Returns the count of all set members with membership bigger then alphaLevel
     * @param alphaLevel lower bound on the membership of members to include
     * @return count of the members with membership above alphaLevel
     */
    public long getMembersCount(Membership<MemberObject> alphaLevel)
    {
        if (0 < members.size())
        {
            long count = 0;
            for (Membership<MemberObject> member : members) {
                // check whether the subject belongs at least a bit to the set
                if (0 < member.compareTo(alphaLevel)) {
                    count++;
                }
            }
            return count;
        }
        return 0;
    }

    /**
     * Returns the count of all members of the set.
     * @return count of all set members with non-sero membership
     */
    public long getAllMembersCount()
    {
        if (0 < members.size())
        {
            long count = 0;
            for (Membership<MemberObject> member : members) {
                // check whether the subject belongs at least a bit to the set
                if (member.isMember()) {
                    count++;
                }
            }
            return count;
        }
        return 0;
    }
    /**
     * Returns the count of core members of the set.
     * @return count of the core members
     */
    public long getCoreMembersCount()
    {
        if (0 < members.size())
        {
            long count = 0;
            for (Membership<MemberObject> member : members) {
                // check whether the subject belongs at least a bit to the set
                if (member.isCoreMember()) {
                    count++;
                }
            }
            return count;
        }
        return 0;
    }


    /**
     * Returns randomly picked member from the list.
     * @return Member structure, type defined by template, null if there are no elements in the list
     */
    public  MemberObject pickRandomMember()
    {
        if ( 0 < members.size())
        {
            long rand = Math.round(Math.floor(Math.random() * (members.size() - 1)));
            return members.get((int)rand).getMember();
        }
        return null;
    }

    /**
     * Returns the best member from the set
     * @return Member structure, type defined by template, null if there are no elements in the list
     */
    public MemberObject pickBestMember()
    {
        Membership<MemberObject> res = null;
        if ( 0 < members.size())
        {
            for (Membership<MemberObject> membership : members) {
                if (0 < membership.compareTo(res))
                {
                    res = membership;
                }
            }
        }
        return res.getMember();
    }

    /**
     * DON'T call this function. It is called by Membership constructor. Adds the member with the membership function to the set.
     * @param membership Member's membership function of the member, including member reference
     */
    public void addMember(Membership<MemberObject> membership)
    {
        // just to make sure that the same entity is not twice in the set, with two different memberships
        removeMember(membership.getMember());
        members.add(membership);
    }

    /**
     * Removes a member specified by membership reference from the set by erasing it from the set.
     * @param membership reference
     * @return true if removed, false if not found
     */
    public boolean removeMember(Membership<MemberObject> membership)
    {
        return members.remove(membership);
    }

    /**
     * Removes a member specified by member reference from the set by erasing it from the set.
     * @param member Member
     * @return true if removed, false if not found
     */
    public boolean removeMember( MemberObject member)
    {
        Membership membership = getMembership(member);
        if (null != membership)
        {
            return members.remove(membership);
        }
        return false;
    }

    /**
     * Checks if the queriedMember is already a member of the set and returns its membership function.
     * @param queriedMember memebr to check
     * @return The membership function of the queriedMember, null if not a member.
     */
    public Membership<MemberObject> getMembership(MemberObject queriedMember)
    {
        for (Membership<MemberObject> membership : members) {
            if (membership.getMember().equals(queriedMember))
            {
                return membership;
            }
        }
        return null;
    }

    /**
     * Checks if the queriedMember is already a member of the set
     * @param queriedMember memebr to check the membership
     * @return true if member, false otherwise
     */
    public boolean containsMember(MemberObject queriedMember)
    {
        for (Membership<MemberObject> membership : members) {
            if (membership.getMember().equals(queriedMember))
            {
                return true;
            }
        }
        return false;
    }
    /** Returns an iterator over membership List */
    public ListIterator<Membership<MemberObject>> getMembershipIterator()
    {
    	return members.listIterator();
    }
}
