package across.util.skn.util;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

import across.util.skn.AgentKnowledge;
import across.util.skn.util.membership.Membership;
import aglobe.container.transport.Address;

/**
 * Special Fuzzy Set, with Members of the AgentKnowledge Type.
 * Includes methods to obtain directly the addresses of agents in lists and to avoid including specified agents in the lists.
 */
public class AgentSet extends FuzzySet<AgentKnowledge>
{
    /**
     * Returns all accessible agents that belong at least partially to the set, except the agent to exclude passed in the param agentToAvoid.
     * @param agentToAvoid agent to remove from the list (for example the caller.) May be null.
     * @return List of AgentKnowledge structures
     */
    public Collection<AgentKnowledge> getAllAccessibleAgents(Address agentToAvoid)
    {
        List<AgentKnowledge> res = new LinkedList<AgentKnowledge>();
        for (Membership<AgentKnowledge> membership : members) {
            if (membership.isMember() && membership.getMember().isAccessible() && membership.getMember().address != agentToAvoid)
            {
                res.add(membership.getMember());
            }
        }
        return res;
    }

    /**
     * Returns all accessible agents that belong to the core of the set, except the agent to exclude passed in the param agentToAvoid.
     * @param agentToAvoid agent to remove from the returnd list (for example the caller.) May be null.
     * @return List of AgentKnowledge structures
     */
    public Collection<AgentKnowledge> getCoreAccessibleAgents(Address agentToAvoid)
    {
        List<AgentKnowledge> res = new LinkedList<AgentKnowledge>();
        for (Membership<AgentKnowledge> membership : members) {
            if (membership.isCoreMember() && membership.getMember().isAccessible() && membership.getMember().address != agentToAvoid)
            {
                res.add(membership.getMember());
            }
        }
        return res;
    }

    /**
     * Returns addresses of all accessible agents that belong at least partially to the set, except the agent to exclude passed in the param agentToAvoid.
     * @param agentToAvoid agent to remove from the list (for example the caller.) May be null.
     * @return List of AgentKnowledge structures
     */
    public Collection<Address> getAllAccessibleAgentsAddress(Address agentToAvoid)
    {
        List<Address> res = new LinkedList<Address>();
        for (Membership<AgentKnowledge> membership : members) {
            if (membership.isMember() && membership.getMember().isAccessible() && membership.getMember().address != agentToAvoid)
            {
                res.add(membership.getMember().address);
            }
        }
        return res;
    }

    /**
     * Returns addresses of all agents that belong at least partially to the set, except the agent to exclude passed in the param agentToAvoid.
     * @param agentToAvoid agent to remove from the list (for example the caller.) May be null.
     * @return List of AgentKnowledge structures
     */
    public Collection<Address> getAllAgentsAddress(Address agentToAvoid)
    {
        List<Address> res = new LinkedList<Address>();
        for (Membership<AgentKnowledge> membership : members) {
            if (membership.isMember() && membership.getMember().address != agentToAvoid)
            {
                res.add(membership.getMember().address);
            }
        }
        return res;
    }

    /**
     * Returns addresses of all agents that belong to the set core, except the agent to exclude passed in the param agentToAvoid.
     * @param agentToAvoid agent to remove from the list (for example the caller.) May be null.
     * @return List of AgentKnowledge structures
     */
    public Collection<Address> getCoreAgentsAddress(Address agentToAvoid)
    {
        List<Address> res = new LinkedList<Address>();
        for (Membership<AgentKnowledge> membership : members) {
            if (membership.isCoreMember() && membership.getMember().address != agentToAvoid)
            {
                res.add(membership.getMember().address);
            }
        }
        return res;
    }

    public Collection<Membership<AgentKnowledge>> getSortedByTrust()
    {
        SortedSet<Membership<AgentKnowledge>> sorted = new TreeSet<Membership<AgentKnowledge>>();
        for (Membership<AgentKnowledge> membership : members) {
            if (membership.isMember())
            {
                // ATTENTION - members ordered by trust !!!!!!!!
                sorted.add(membership.getMember().getTrust());
            }
        }
        return sorted;
    };
}
