package across.util.skn.update;

import across.util.skn.CommunityKnowledge;
import aglobe.container.EventReceiver;
import aglobe.container.sysservice.directory.DirectoryListener;
import aglobe.container.sysservice.directory.DirectoryRecord;
import aglobe.container.transport.MessageTransport;
import aglobe.ontology.VisibilityUpdate;
import aglobe.service.gis.client.GISTopicListener;

/**
 * This Listener shall be registered with DirectoryService on agent's container to update social knowledge in the agent structure.
 * User must create and subscribe/unsubscribe this structure with GISClient service visibility topic and with DirectoryService.
 */
public class UpdateListener implements DirectoryListener, GISTopicListener
{

    /** Social knowledge structure to be updated by the listener */
    private final CommunityKnowledge ckn;
    /** Reference to the owning agent or service */
    private final EventReceiver owner;

    /**
     * Creates the update class.
     * @param ckn reference to the client's social knowledge structure to be maintained.
     */
    public UpdateListener(CommunityKnowledge ckn, EventReceiver owner)
    {
        this.ckn = ckn;
        this.owner = owner;
    }

    /**
     * Handles new registration in the directory. New registration of an agent/service can be called
     * only if the agent is not visible yet.
     * @param containerName String - container name where the agent/service have been registred
     * @param records DirectoryRecord[] - array of the agents/services which are registred. An agent/service can
     * handle its own registration too.
     * @param matchingFilter String - matching filter to which is this callback. Can be used if an someone
     * has more subscriptions to the same listener with cross filter. In this case registration info about one agent/service
     * can be handled two times
     */
    public void handleNewRegister(String containerName, DirectoryRecord[] records, String matchingFilter) {
        for (int i = 0; i < records.length; i++) {
            DirectoryRecord record = records[i];
            ckn.agentRegistered(record);
        }
    }

    /**
     * Handles deregistration of the agent/service. New registration of an agent/service can be called
     * only if the agent is not visible now.
     * @param containerName String - container name where the agent/service have been deregistred
     * @param records DirectoryRecord[] - array of the agents/services which are deregistred. Only address and containerName
     * fields are valid; services field couldn't have valid data. An agent/service can
     * handle its own deregistration too.
     * @param matchingFilter String - matching filter to which is this callback. Can be used if an someone
     * has more subscriptions to the same listener with cross filter.
     */
    public void handleDeregister(String containerName, DirectoryRecord[] records, String matchingFilter)
    {
        for (int i = 0; i < records.length; i++)
        {
            DirectoryRecord record = records[i];
            ckn.agentDeRegistered(record);
        }
    }

    /**
     * Handles that some agents/services becomes visible to me. Visible of an agent/service can be called
     * only if the agent is previously known to me. (register called before)
     * @param containerName String - container name where agents/services become visible
     * @param records DirectoryRecord[] - array of the agents/services which become visible. Only address and containerName
     * fields are valid; services field couldn't have valid data. An agent/service can
     * handle its own visibility too.
     * @param matchingFilter String - matching filter to which is this callback. Can be used if an someone
     * has more subscriptions to the same listener with cross filter.
     */
    public void handleVisible(String containerName, DirectoryRecord[] records, String matchingFilter)
    {
        // visibility is managed by visibility topic on container level
    }

    /**
     * Handles that some agents/services becomes invisible to me. Invisible of an agent/service can be called
     * only if the agent is previously known to me and was visible to me. (register called before)
     * @param containerName String - container name where agents/services become invisible
     * @param records DirectoryRecord[] - array of the agents/services which become invisible. Only address and containerName
     * fields are valid; services field couldn't have valid data. An agent/service can
     * handle its own invisibility too.
     * @param matchingFilter String - matching filter to which is this callback. Can be used if an someone
     * has more subscriptions to the same listener with cross filter.
     */
    public void handleInvisible(String containerName, DirectoryRecord[] records, String matchingFilter)
    {
        // visibility is managed by visibility topic on container level
    }

    /**
     * Called by the GISClient service to distribute information received from master entities.
     * @param topic type of the information  (Ex. WEATHER)
     * @param content content of the information (Ex. 10 degree, rain, 70%humidity, tornado,...)
     */
    public void handleTopic(String topic, Object content, String reason)
    {
        // we expect to receive container visibility updates here
        if (topic.equalsIgnoreCase(MessageTransport.TOPIC_VISIBILITY_UPDATES))
        {
            VisibilityUpdate gi = (VisibilityUpdate) content;
            ckn.updateContainerVisibility(gi.getVisibleContainerAddress());
        }
    }

    public void addEvent(Runnable e)
    {
        owner.addEvent(e);
    }
}
