package across.util.skn.plan.across;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;

import across.data.Node;
import across.data.Resource;
import across.data.TransportPlan;
import across.data.TransportPlanBase;
import across.util.skn.AgentKnowledge;
import across.util.skn.CommunityKnowledge;
import across.util.skn.plan.Action;
import across.util.skn.plan.ActionType;
import across.util.skn.plan.ResourceKnowledge;

/**
 * This class holds simple abstract actions used by the agent for the planning of transports 
 * - it contains the elements of plan templates, resources of individual agents and the knowledge about resources.
 */
public class PlanBase {

	protected LinkedList<AcPlanTemplate> templates = new LinkedList<AcPlanTemplate>();
	
	/** Contains the meta-information about Actions - action successfulness, */
	protected LinkedHashMap<String, ActionType> actions = new LinkedHashMap<String, ActionType>();
	
	/** References to resources. */
	protected LinkedHashMap<String, ResourceKnowledge> resources = new LinkedHashMap<String, ResourceKnowledge>();

	private CommunityKnowledge caller;
	
	/** Nodes of the map used for distance estimate.*/ 
	protected HashMap<String, Node> nodes = null;
	
	/** Loads the plan templates from the xml file */
	public void loadFromXml(TransportPlanBase tpb)
	{
		for (Iterator it = tpb.getTransportPlan().iterator(); it.hasNext();) {
			TransportPlan tpxml = (TransportPlan) it.next();
			AcPlanTemplate acpt = new AcPlanTemplate(tpxml);
			templates.add(acpt);
		}
	}
	
	
	/** Returns the plan templates from the plan base that carry the cargo from startNode to targetNode. */
	public List<AcPlanTemplate> selectPlans(String startNode, String targetNode)
	{
		LinkedList<AcPlanTemplate> res = new LinkedList<AcPlanTemplate>();
		for (Iterator cit = templates.iterator(); cit.hasNext();) {
			AcPlanTemplate pt = (AcPlanTemplate) cit.next();
			for (Iterator pit = pt.getPrerequisites().iterator(); pit.hasNext();) {
				ObGoodsStored pob = (ObGoodsStored) pit.next();
				// check the start match first
				if (pob.getStorageNode().equalsIgnoreCase(startNode))
				{
					// if the start matches, check the target
					for (Iterator ait = pt.getAllows().iterator(); ait.hasNext();) {
						ObGoodsStored aob = (ObGoodsStored) ait.next();
						if (aob.getStorageNode().equalsIgnoreCase(targetNode))
						{
							res.add(pt);
						}
					}
				}
			}
		}
		return res;
	}//TODO - rewrite in a more general manner, push into PlanTemplate
	
	/** Combines all the input templates into a big global plan */
	public AcPlanTemplate combinePlans (List<AcPlanTemplate> plansToCombine)
	{
		//* contains all Actions (Objectives) with the same descriptor
		LinkedHashMap<String, List<Action>> actionsByDescriptor = new LinkedHashMap<String, List<Action>>();
		LinkedHashMap<String, List<Action>> objectivesByDescriptor = new LinkedHashMap<String, List<Action>>();
		for (AcPlanTemplate pt : plansToCombine) {
			// actions first
			for (Action ac : pt.getActions()) {
				List<Action> collection = actionsByDescriptor.get(ac.getActionDescriptor());
				if ( null != collection)
				{
					collection.add(ac);
				}
				else
				{
					collection = new LinkedList<Action>();
					collection.add(ac);
					actionsByDescriptor.put(ac.getActionDescriptor(), collection);
				}
			}
			//objectives next
			for (Action ob : pt.getObjectives()) {
				List<Action> collection = objectivesByDescriptor.get(ob.getActionDescriptor());
				if ( null != collection)
				{
					collection.add(ob);
				}
				else
				{
					collection = new LinkedList<Action>();
					collection.add(ob);
					objectivesByDescriptor.put(ob.getActionDescriptor(), collection);
				}
			}
		}
		// merge the lists we have gathered before
		LinkedHashMap<String, AcTransport> uniqueActions = new LinkedHashMap<String, AcTransport>();
		LinkedHashMap<String, AcTransport> uniqueObjectives = new LinkedHashMap<String, AcTransport>();
		// actions
		for (List<Action> abd : actionsByDescriptor.values()) {
			uniqueActions.put(abd.get(0).getActionDescriptor(), new AcTransport(abd));
		}
		// objectives
		for (List<Action> obd : objectivesByDescriptor.values()) {
			uniqueObjectives.put(obd.get(0).getActionDescriptor(), new ObGoodsStored(obd));
		}
		// create the new, complete combined plan
		return new AcPlanTemplate(uniqueActions.values(), uniqueObjectives.values());
	}
	
	/**
	 * 
	 */
	public PlanBase(CommunityKnowledge caller) {
		super();
		this.caller = caller;
	}


	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		for (AcPlanTemplate acp : templates) {
			sb.append(acp.toString());
		}
		return sb.toString();
	}

	/**
	 * Registers a new provider (resource) with specific action. If the action does not exist yet, it is created.
	 * @param ad action
	 * @param knowledge provider
	 * @return 
	 */
	public ActionType updateOrCreateActionType(String ad, ResourceKnowledge knowledge) {
		ActionType act = actions.get(ad);
		if (null == act)
		{
			// create new
			act = new ActionType(ad,knowledge);
			actions.put(ad,act);
		}
		else
		{
			act.addCompatibleResource(knowledge);
		}
		return act;
	}
	
	/**
	 * Registers a new provider (resource) with specific action. if the action does not exist yet, it is created.
	 *
	 */
	public ResourceKnowledge updateOrCreateResource(AgentKnowledge ak, Resource res, boolean isAvailable) 
	{
		ResourceKnowledge rk = resources.get(res.getResid());
		if (null == rk)
		{
			rk = new ResourceKnowledge(this, ak, res, isAvailable);
			resources.put(res.getResid(), rk);
//			System.out.println(caller.getOwnerAgentKnowledge().getName() + ": Resource created:" + rk.getResourceID() + " av: " + rk.isAvailable() + " own: " +rk.getResourceOwner().getName()+" applicable actions size: "+rk.getApplicableActions().size());
		}
		else
		{
			rk.updateResource(res, isAvailable);
//			System.out.println(caller.getOwnerAgentKnowledge().getName() + ": Resource updated:" + rk.getResourceID() + " av: " + rk.isAvailable() + " own: " +rk.getResourceOwner().getName()+" applicable actions size: "+rk.getApplicableActions().size());
		}
		return rk;
	}


	public LinkedHashMap<String, ResourceKnowledge> getResources() {
		return resources;
	}


	public LinkedHashMap<String, ActionType> getActions() {
		return actions;
	}
	
	public  ActionType getActionType(String descriptor) {
		return actions.get(descriptor);
	}
	
	public  List<ResourceKnowledge> getActionTypeResources(String descriptor) {
		ActionType actionType = actions.get(descriptor);
		if (null != actionType)
		{
			return actionType.getCompatibleResourcesList();
		}
		return new LinkedList<ResourceKnowledge>();
	}


	public CommunityKnowledge getCommunity() {
		return caller;
	}
	
	/**
	 * Returns an estimate of distance between two nodes, using their X,Y coordinates.
	 * @param startNode start node
	 * @param targetNode target node
	 * @return Distance estimate (symmetrical), big number if at least one node is unknown 
	 */
	public double getDistanceEstimate(String startNode, String targetNode)
	{
		if (nodes != null) {
			Node start = nodes.get(startNode);
			Node target = nodes.get(targetNode);
			if (null != start && null != target) {
				return Math.sqrt((start.getX() - target.getX()) * (start.getX() - target.getX()) +  (start.getY() - target.getY()) * (start.getY() - target.getY()));
			}
		}
		// unknown node implies very big distance
		return 5000000;
	}
	
	/**
	 * Returns an estimate of distance between of the transport action.
	 * @return Distance estimate (symmetrical), big number if at least one node is unknown 
	 */
	public double getDistanceEstimate(AcTransport transAction)
	{
		return getDistanceEstimate(transAction.getStartNode(), transAction.getTargetNode());
	}

	public void setupNodes(HashMap<String, Node> nodes2) {
		nodes = nodes2;
	}
	
//  NOT needed - would be too domain dependent...	
//	/**
//	 * Receives the semi-private params and creates corresponding knowledge structures.
//	 */
//	public void handleSemiPrivate(Address address, SemiPrivateParams spp)
//	{
//		AgentKnowledge ak = caller.getAgent(address.getName());
//		if (null != ak)
//		{
//			for (Iterator iter = spp.getResource().iterator(); iter.hasNext();) {
//				Resource res = (Resource) iter.next();
//				updateOrCreateResource(ak, res);
//			}
//		}
//	}
	
	
	

}
