package across.util.skn.plan;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import across.data.Resource;
import across.data.TransportBatch;
import across.util.GoodsConstants;
import across.util.ListBatchTools;
import across.util.skn.AgentKnowledge;
import across.util.skn.SocialSubject;
import across.util.skn.plan.across.PlanBase;

/** Corresponds to single resource or aggregate of zero or more resources with shared characteristics.
 */

public class ResourceKnowledge extends SocialSubject {

	/** References the plan base to which the resource is attached */
	PlanBase planBase;

	/** Owner of the resource. */
	AgentKnowledge resourceOwner;

	/** Resource Agent. Filled in only if the resopurce is represented by an agent (not the owner) and the agent is known to knowledge owner.*/
	AgentKnowledge resourceAgent;
	/** Unique ID of the resource */
	String resourceID;
	/** Maximum resource capacity */
	long maxCapacity = 0;
	/** Currently available resource capacity */
	long capacity = 0;
	/** True if the resource can be used for planning */
	boolean isAvailable = false;

	/** Resource information */
	Resource resInfo;

	/** Lists all elementary actions to which the resource can be assigned. */
	List<ActionType> applicableActions = new LinkedList<ActionType>();
	//TODO - membership with trustfulness ?? - or rather take this from agent and context ?


	public ResourceKnowledge(PlanBase planBase, AgentKnowledge owner, Resource xmlInfo, boolean resourceAvailable) {
		super();
		this.planBase = planBase;
		updateResource(xmlInfo, resourceAvailable);
		resourceOwner = owner;
		resourceOwner.addResource(this);
		for (String ad : (Collection<String>)xmlInfo.getAvailableAction()) {
			ActionType at = planBase.updateOrCreateActionType(ad, this);
			applicableActions.add(at);
		}
	}

	public ResourceKnowledge(PlanBase planBase, AgentKnowledge owner, AgentKnowledge resourceAg, Resource xmlInfo, boolean resourceAvailable) {
		super();
		this.planBase = planBase;
		updateResource(xmlInfo, resourceAvailable);
		resourceOwner = owner;
		resourceOwner.addResource(this);
		for (String ad : (Collection<String>)xmlInfo.getAvailableAction()) {
			ActionType at = planBase.updateOrCreateActionType(ad, this);
			applicableActions.add(at);
		}
		resourceAgent = resourceAg;
	}

	@Override
	public boolean isAccessible() {
		return true;
	}

	/** Updates the information about resource */
	public void updateResource(Resource res, boolean resourceAvailable) {
		resourceID = res.getResid();
		long rescap = res.getCapacity();
		if (rescap > maxCapacity)
		{
			maxCapacity = rescap;
		}
		capacity = rescap;
		resInfo = res;
		isAvailable = resourceAvailable;

		for (String ad : (Collection<String>)res.getAvailableAction()) {
			ActionType at = planBase.updateOrCreateActionType(ad, this);
			applicableActions.add(at);
		}
		// update available actions
		//TODO - actions are static for now, no need for the code so far
	}

	public List<ActionType> getApplicableActions() {
		return applicableActions;
	}

	public String getResourceID() {
		return resourceID;
	}

	public boolean isAvailable() {
		return isAvailable;
	}

	public double getCapacity() {
		return capacity;
	}

	public String getType() {
		return resInfo.getType();
	}

	public AgentKnowledge getResourceOwner() {
		return resourceOwner;
	}

	public double getTrustworthiness() {
		return resourceOwner.getApplicableTrustworthiness().getMembershipFunctionCenter();
	}

	/** Returns a complement of the resource trustfulness */
	public double getDistrustfulness() {
		return (1 - resourceOwner.getApplicableTrustworthiness().getMembershipFunctionCenter());
	}
	
	public boolean canCarryType(TransportBatch batch) {
		if (getType().equalsIgnoreCase(GoodsConstants.resolveType(batch.getComodityName())))
			return true;
		return false;
	}

	/** Used to assign resources and their parts to batches and other entities */
	public class ResourceAssignment
	{
		protected long assignedCapacity;

		public ResourceAssignment(long capacity) {
			super();
			assignedCapacity = capacity;
		}

		public ResourceAssignment() {}

		public long getAssignedCapacity() {
			return assignedCapacity;
		}

		public void setAssignedCapacity(long assignedCapacity) {
			this.assignedCapacity = assignedCapacity;
		}

		public long getMaxCapacity() {
			return ResourceKnowledge.this.maxCapacity;
		}

		public long getFreeCapacity() {
			return ResourceKnowledge.this.maxCapacity - assignedCapacity;
		}

		public ResourceKnowledge getRk() {
			return ResourceKnowledge.this;
		}
	}

	/** Used to assign resources and their parts to batches and other entities */
	public class BatchResourceAssignment extends ResourceAssignment
	{
		public BatchResourceAssignment(TransportBatch firstBatch, long batchAssigned) {
			super();
			assignBatch(firstBatch,batchAssigned);
		}

		public BatchResourceAssignment()
		{
			super();
		}

		LinkedHashMap<TransportBatch, Long> batches = new LinkedHashMap<TransportBatch, Long>();

		/** This will never assign more than free capacity
		 * @return really assigned capacity. */
		public long assignBatch(TransportBatch tb, long batchAssigned) {
			batchAssigned = Math.min(getFreeCapacity(),batchAssigned);
			batches.put(tb,batchAssigned);
			assignedCapacity += batchAssigned;
			return batchAssigned;
		}

		public LinkedHashMap<TransportBatch, Long> getBatches() {
			return batches;
		}

		public LinkedList<TransportBatch> getBatchesWithAssignedQuantities() {
			LinkedList<TransportBatch> res = new LinkedList<TransportBatch>();
			for (Map.Entry<TransportBatch, Long> ent : batches.entrySet()) {
				res.add(ListBatchTools.copyTransportBatch(ent.getKey(),ent.getValue()));
			}
			return res;
		}
	}

	/** Returns true if the Resource can be assigend (assigned in planning sense, of cource) to the action described by descriptor desc*/
	public boolean isCompatibleWith(String desc) {
		for (ActionType action : applicableActions) {
			if (action.actionDescriptor.equalsIgnoreCase(desc))
				return true;
		}
		return false;
	}

	public void setAvailable(boolean newAvailability) {
		isAvailable = newAvailability;
	}
}
