package across.util.skn;

import java.lang.reflect.Constructor;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;

import across.data.Param;
import across.data.PublicParams;
import across.data.Resource;
import across.data.SemiPrivateParams;
import across.data.trust.AgentTrustfulness;
import across.simulation.constants.AcrossDataConstants;
import across.util.skn.plan.ResourceKnowledge;
import across.util.skn.util.AgentSet;
import across.util.skn.util.membership.CrispMembership;
import across.util.skn.util.membership.IterativeFuzzyMembership;
import across.util.skn.util.membership.Membership;
import across.util.skn.util.membership.iterative.IterativeTwoCategories;
import across.util.skn.util.membership.iterative.SigmaMembership;
import aglobe.container.sysservice.directory.DirectoryRecord;
import aglobe.container.transport.Address;

/**
 * Contains intruder knowledge about one agent.
 */
public class AgentKnowledge extends SocialSubject
{
    /** Address of the agent */
    public Address address;

    /** Public information about the agent as obtained from yellow pages. */
    public PublicParams publicParams;
    /** Public information about the agent as obtained from alliance members. */
    public SemiPrivateParams semiPrivateParams;
    /** Agent private information as observed, recieved from meta-agent and deduced...  */
    public PublicParams privateParams;
    /** Reference to container where agent resides*/
    public ContainerKnowledge container;
    /** Reference to alliance to which agent belongs. It is obtained from it's public information.*/
    public AllianceKnowledge alliance;
    /** Contains references to services provided by agent */
    public List<ServiceKnowledge> servicesProvided = new LinkedList<ServiceKnowledge>();
    /** Resources offered by the agent - with all restrictions and current capacities. Based on infor from SemiPrivateParams. */
    public List<ResourceKnowledge> resources = new LinkedList<ResourceKnowledge>();
    /** Reference to the owner object */
    private final CommunityKnowledge caller;

    /** Represents the trust in this agent. Specifically, it is the value of agent's membership in the set of trusted agents*/
    protected IterativeFuzzyMembership<AgentKnowledge> trust;

    /**  Holds the aggregate reputation of the agent from received observations ABOUT the agent. - does not include values data*/
    protected IterativeFuzzyMembership<AgentKnowledge> aggregateReputation;
    /** References the agents who have provided reputation values for this agent. */
    protected AgentSet reputationSources = new AgentSet();
    /** Reputation of other agents as provided by agent. */
    protected AgentSet reputation = new AgentSet();
    /** Trustfulness categories as provided by the agent. */
    protected IterativeTwoCategories reputationCategories = new IterativeTwoCategories();
    /** Used as return values for functions that provide trusting decisions */
    public enum TrustDecision{TRUSTED, DISTRUSTED, NOT_ENOUGH_DATA, NO_GAUGE_DATA};

    public AgentKnowledge(Address agentAddress, PublicParams pp, CommunityKnowledge caller)
    {
        this.caller = caller;
        address = agentAddress;
        publicInfoUpdated(agentAddress, pp, null);
        initTrust();
    }

    public AgentKnowledge(Address agentAddress, PublicParams pp, String containerName, CommunityKnowledge caller)
    {
        this.caller = caller;
        address = agentAddress;
        publicInfoUpdated(agentAddress, pp, containerName);
        initTrust();
    }

    public AgentKnowledge(DirectoryRecord dr, CommunityKnowledge caller)
    {
        this.caller = caller;
        address = dr.address;
        container = caller.updateOrCreateContainer(this.address.deriveContainerAddress(), this);
        for (int i = 0; i < dr.getServices().length; i++)
        {
            servicesProvided.add(caller.updateOrCreateService(dr.getServices()[i], this));
        }
        initTrust();
    }

    /** The simplest init - used only for the agents that are neither registered, nor have the Page with info.
     * Avoid if possible as container and service info are not initialized. */
    public AgentKnowledge(Address sender, CommunityKnowledge caller) {
    	this.caller = caller;
        address = sender;
        initTrust();
    }

	/** Handles the registration of the agent as received from directory service */
    public void directoryRegistration(DirectoryRecord dr)
    {
        if (null != container)
        {
            // remove from old and add to the new container - this means migration without deregistration
            if ( !container.address.equals(dr.address.deriveContainerAddress()))
            {
                // container change
                container.removeAgent(this);
                container = caller.updateOrCreateContainer(this.address.deriveContainerAddress() , this);
            }
        }
        else
        {
            // first container
            container = caller.updateOrCreateContainer(this.address.deriveContainerAddress() , this);
        }
        // pass through the new services and either find the match (no dereg before migration) or register
        for (int i = 0; i < dr.getServices().length; i++)
        {
            String sname = dr.getServices()[i];
            boolean match = false;
            for (Iterator<ServiceKnowledge> sit = servicesProvided.iterator(); sit.hasNext();)
            {
                ServiceKnowledge sk = sit.next();
                if(sname.equalsIgnoreCase(sk.serviceName))
                {
                    // found match - still providing this service
                    match = true;
                    break;
                }
            }
            if ( !match )
            {
                // no match in current, we must register the agent as service provider
                servicesProvided.add(caller.updateOrCreateService(dr.getServices()[i], this));
            }
        }
    }

    /** Handles the deregistration of the agent as received from directory service */
    public void directoryDeRegistration(DirectoryRecord dr)
    {
        // pass through the deregistered services and find the match
        for (int i = 0; i < dr.getServices().length; i++)
        {
            String sname = dr.getServices()[i];
            for (Iterator<ServiceKnowledge> sit = servicesProvided.iterator(); sit.hasNext();)
            {
                ServiceKnowledge sk = sit.next();
                if(sname.equalsIgnoreCase(sk.serviceName))
                {
                    // found match - deregister from this service
                    sk.removeMember(this);
                    // remove it from data model
                    caller.serviceProvidersDataModel.update(sk);
                    // remove it from agent list
                    sit.remove();
                    break;
                }
            }
        }
        // just verify that we didn't miss migration or registration...
        if (null != container && !container.address.equals(address.deriveContainerAddress()))
        {
            // container change
            container.removeAgent(this);
            container = caller.updateOrCreateContainer(this.address.deriveContainerAddress() , this);
        }
    }
    /** Call this to rgister agent as a service provider for some service */
    public void addServiceProvided(String serviceDescriptor)
    {
    	servicesProvided.add(caller.updateOrCreateService(serviceDescriptor, this));
    };

    /**
     * Updates existing agent public information using the new page recieved from yellow pages.
     * @param containerName name of the container where the agent is located - may be NULL !
     * @param page recieved page, contains agent's public information
     */
    public void publicInfoUpdated(Address address, PublicParams publicParams, String containerName)
    {

        // update service providers data model, because public params can be changed
        for (Iterator<ServiceKnowledge> iter = servicesProvided.iterator(); iter.hasNext(); ) {
          ServiceKnowledge item = iter.next();
          caller.serviceProvidersDataModel.update(item);
        }

        Address cad = address.deriveContainerAddress();
        // remove from old and add to the new container
		if (null != container && !container.address.equals(cad))
        {
            // container change
            container.removeAgent(this);
            container = caller.updateOrCreateContainer(cad, this);
        }
        else
        {
            // first container
        	container = caller.updateOrCreateContainer(cad, this);
        }
        // identify alliance name, if there is any...
        String currentAlliance = null;
        for (Iterator<Param> i = publicParams.getParam().iterator(); i.hasNext();)
        {
            Param param = i.next();
            if (param.getName().equalsIgnoreCase(AcrossDataConstants.TRANSPORTER_PARAM_ALLIANCE_NAME))
            {
                currentAlliance = param.getValue();
                break;
            }
        }
        // check all cases - first alliance, changing, leaving...
        enterAlliance(currentAlliance);
        publicIdentity = publicParams;
        this.publicParams = publicParams;
    }

    /** Initializes agent's trust membership function. picks from various possible types.
     *  The type to chose is determined by from caller properties.
     */
    public void initTrust()
    {
//        System.out.println(address.toString()+" "+caller.getTrustMembership().getName());
        try {
          Constructor con = caller.getTrustMembership().getConstructor(new Class[] {AgentKnowledge.class, across.util.skn.util.FuzzySet.class});
          trust = (IterativeFuzzyMembership<AgentKnowledge>)con.newInstance(new Object[] {this, caller.trustedAgents});
        }
        catch (Exception e) {
          trust = new SigmaMembership<AgentKnowledge>(this, caller.trustedAgents);
        }
        // initialize the aggregate reputation
        aggregateReputation = initTrust(this,caller.aggregateReputationAgents);
    }

    /** Initializes agent's trust membership function. picks from various possible types.
     *  The type to chose is determined by from caller properties.
     */
    public IterativeFuzzyMembership<AgentKnowledge> initTrust(AgentKnowledge trustedAgent, AgentSet trustSet)
    {
    	try {
          Constructor con = caller.getTrustMembership().getConstructor(new Class[] {AgentKnowledge.class, across.util.skn.util.FuzzySet.class});
          return (IterativeFuzzyMembership<AgentKnowledge>) con.newInstance(new Object[] {trustedAgent, trustSet});
        }
        catch (Exception e) {
        	return new SigmaMembership<AgentKnowledge>(trustedAgent, trustSet);
        }
    }

    public void updateSemiPrivateParams(SemiPrivateParams sp) {
    	
    	this.semiPrivateParams = sp;
    	if (alliance != null) {
    		caller.alliDataModel.update(alliance);
    		caller.myAllianceDataModel.update(alliance);
    	}
    	// update service providers data model, because public params can be changed
    	for (Iterator iter = servicesProvided.iterator(); iter.hasNext(); ) {
    		ServiceKnowledge item = (ServiceKnowledge)iter.next();
    		caller.serviceProvidersDataModel.update(item);
    	}
    	// update the resources and respective info
    	// - only for others, owners manage their resources in private params
    	if (! isOwner()) {
    		Collection<Resource> ress = new LinkedList<Resource>(sp.getResource());
    		// update the currently known resources with actual capacity
    		for (ResourceKnowledge rk : getResources()) {
    			boolean upd = false;
    			for (Iterator iter = ress.iterator(); iter.hasNext();) {
        			Resource res = (Resource) iter.next();
        			if (rk.getResourceID().equalsIgnoreCase(res.getResid()))
        			{
        				rk.updateResource(res, true);
        				upd = true;
        				// remove the resource from 
        				iter.remove();
        				break;
        			}
        		}
    			// check whether the resource wasn't excluded from the ress as it is not available...
    			if (! upd)
    			{
    				rk.setAvailable(false);
    			}
			}
    		// handle the previously unknown resources...
    		for (Iterator iter = ress.iterator(); iter.hasNext();) {
    			Resource res = (Resource) iter.next();
    			// update the plan base in caller - ActionTypes are created automatically
    			caller.updateOrCreateResource(this, res, res.getCapacity() > 0);
    			// ResourceKnowledge adds itself to the owner ak resource list
    		}
    	}
    }

    /** Agent will be assigned an alliance passed in the parameter. If this alliance does not exist, it is created.
     *
     * @param newAlliance
     */
    public void enterAlliance( String newAlliance)
    {
        if (null != newAlliance)
        {
            if(null != alliance)
            {
                if(!newAlliance.equalsIgnoreCase(alliance.name))
                {
                    alliance.removeMember(this);

                    caller.alliDataModel.remove(this);
                    caller.myAllianceDataModel.remove(this);

                    // test if number of alliance members is 0 then remove alliance
                    if (alliance.getAllMembersCount() == 0) {
                      caller.alliances.remove(alliance.name);
                      caller.alliDataModel.remove(alliance);
                      caller.myAllianceDataModel.remove(alliance);
                    }
                    alliance = caller.updateOrCreateAlliance(newAlliance, this);
                }
            }
            else
            {
                // first time entry
                alliance = caller.updateOrCreateAlliance(newAlliance, this);
            }
        }
        else
        {
            // remove agent from old alliance
            if (null !=alliance)
            {
                alliance.removeMember(this);
                caller.alliDataModel.remove(this);
                // test if number of alliance members is 0 then remove alliance
                if (alliance.getAllMembersCount() == 0) {
                  caller.alliances.remove(alliance.name);
                  caller.alliDataModel.remove(alliance);
                  caller.myAllianceDataModel.remove(alliance);
                }
            }
        }
    }

    /**
     * Returns true if given entity is accessible, false if not. We consider the groups of agents to
     * be accessible if at least one member is accessible.
     *
     * @return true if accessible, false if not
     */
    public boolean isAccessible() {
        if (null != container)
        {
            return container.isAccessible();
        }
        return false;
    }

    /**
     * Updates the trust in the agent using the data from one interaction, for example coalition.
     * @param eventTrust trust in the agent in the context of this single event. Accepts values bigger then 1 and crops them to 1.
     */
    public void updateTrustWithNewEvent(double eventTrust) {
        eventTrust = Math.min(1, eventTrust);
        trust.newSample(eventTrust);
        // update the trust categories
        if (isOwner())
        {
            caller.trustCategories.newSample(eventTrust);
        }
//        System.out.println(caller.getOwnerAgentKnowledge().address.getName() + " has updated the trust in " + address.getName() + " to " + trust.getMembershipFunctionCenter() + " " + trust.getMembershipFunctionUncertainity());
        List data = new LinkedList();
        data.add(caller.getOwnerAgentKnowledge().address.getName());
        data.add(address.getName());
        data.add(Double.toString(trust.getMembershipFunctionCenter()));
        data.add(Double.toString(trust.getMembershipFunctionUncertainity()));
        data.add((this.isTrusted() == TrustDecision.TRUSTED ? "Y" : "N")); // TODO: update all when self-trust changed
//        System.out.println
    }

    /** Call this function upon the reception of the new trust observation from the agent
     * @param evaluatedAgent agent whose trustworthiness is evaluated
     * @param observedEventTrust trustworthiness observation
     */
    public void reputationObservationReceivedFromAgent(AgentKnowledge evaluatedAgent, double observedEventTrust)
    {
    	IterativeFuzzyMembership<AgentKnowledge> mem = ((IterativeFuzzyMembership<AgentKnowledge>)reputation.getMembership(evaluatedAgent));
    	if (null == mem)
    	{
    		mem = initTrust(evaluatedAgent, reputation);
    		evaluatedAgent.registerReputationOpinion(this);
    	}
		mem.newSample(observedEventTrust);
		//manage the agent's self trust - trustDomains
		if (evaluatedAgent.equals(this))
		{
			reputationCategories.newSample(observedEventTrust);
		}
    }
    /**
     * Sets the reputation value of evaluatedAgent (as perceived by this agent) to the new value.
     * @param agt trustfulness value
     */
    public void reputationReceivedFromAgent(AgentTrustfulness agt)
    {
    	AgentKnowledge evaluatedAgent = caller.getAgent(agt.getAgentName());
    	if (null != evaluatedAgent)
    	{
    		IterativeFuzzyMembership<AgentKnowledge> mem = ((IterativeFuzzyMembership<AgentKnowledge>)reputation.getMembership(evaluatedAgent));
    		if (null == mem)
    		{
    			mem = initTrust(evaluatedAgent, reputation);
    			evaluatedAgent.registerReputationOpinion(this);
    			//caller.getOwnerAgent().getLogger().severe(caller.getOwnerAgent().getName() + ": repValueCreated: " + mem);
    		}
    		mem.setData(agt);
    		//manage the agent's self trust - trustDomains
    		if (evaluatedAgent.equals(this))
    		{
    			reputationCategories.setData(agt);
    		}
    	}
    	else
    	{
    		caller.getOwnerAgent().getLogger().severe( caller.getOwnerAgent().getName() + " : reputation values received for unknown agent: " + agt.getAgentName());
    	}
    }

    /** Registers an opinion provider about this agent */
    public void registerReputationOpinion(AgentKnowledge appraisingAgent)
    {
    	if (! reputationSources.containsMember(appraisingAgent)){
    		new CrispMembership<AgentKnowledge>(appraisingAgent,reputationSources);
    	}
    }

    /** Returns owner agent trust in the subject agent represented by this AgentKnowledge */
    public IterativeFuzzyMembership<AgentKnowledge> getTrust()
    {
        return trust;
    }

    /** Returns the aggregate reputation of the agent as determined from received observations -
     * does NOT include trust values data from others or own experience
     * @return fuzzy number representing the aggregate reputation
     */
    public IterativeFuzzyMembership<AgentKnowledge> getAggregateReputation()
    {
        return aggregateReputation;
    }

    /** Returns true if the name of the agent is the same.
     * We DON'T use address comparison, as agent's address may change during migration.
     * */
    public boolean equals(Object obj) {
        AgentKnowledge other = (AgentKnowledge) obj;
        return address.getName().equalsIgnoreCase(other.address.getName());
    }

    public String getName()
    {
        return address.getName();
    }

    /** Returns true if this AgentKnowledge represents the self-knowledge about agent*/
    public boolean isOwner()
    {
        if (caller.getOwnerAgentKnowledge() == this)
        {
            return true;
        }
        return false;
    }

    public boolean providesService(String serviceName)
    {
    	for (ServiceKnowledge service : servicesProvided) {
			if ( service.serviceName.equalsIgnoreCase(serviceName))
				return true;
		}
    	return false;
    }

    /** Returns true if this agent is trustworthy. Trustworthiness is evaluated using the caller trustCategories...*/
    public TrustDecision isTrusted()
    {
        if (trust.hasEnoughData()) {
			if (caller.trustCategories.isHigherThanLower(this.trust)) {
				return TrustDecision.TRUSTED;
			}
			return TrustDecision.DISTRUSTED;
		}
        return TrustDecision.NOT_ENOUGH_DATA;
    }

    /** If the agent has provided its self-trust or necessary observation data, use this data to evaluate trustworthiness */
    public TrustDecision trusts(AgentKnowledge trustedAgent)
    {
    	if (hasEnoughTrustDataAbout(trustedAgent))
    	{
    		if (reputationCategories.hasEnoughData()) {
				if (reputationCategories
						.isHigherThanLower((IterativeFuzzyMembership<AgentKnowledge>) reputation
								.getMembership(trustedAgent))) {
					return TrustDecision.TRUSTED;
				}
				return TrustDecision.DISTRUSTED;
			}
    		return TrustDecision.NO_GAUGE_DATA;
    	}
    	return TrustDecision.NOT_ENOUGH_DATA;
    }

    /**
     * If we don't have enough self-trust data from the agent, use this function that determines the trustfulness of the evaluated trustedAgent
     * using our trust in the evaluating agent (represented by AgentKnowledge instance).
     * @param trustedAgent agent whose trustworthiness is evaluated
     * @return whether the evaluator would trust trustedAgent were our trustfulness data about evaluator correct
     */
    public TrustDecision shouldTrust(AgentKnowledge trustedAgent)
    {
    	if (trust.hasEnoughData()) {
			if (hasEnoughTrustDataAbout(trustedAgent)) {
				// generate the categories from trust
				IterativeTwoCategories repCat = new IterativeTwoCategories(
						trust);
				if (repCat
						.isHigherThanLower((IterativeFuzzyMembership<AgentKnowledge>) reputation
								.getMembership(trustedAgent))) {
					return TrustDecision.TRUSTED;
				}
				return TrustDecision.DISTRUSTED;
			}
			return TrustDecision.NOT_ENOUGH_DATA;
		}
    	return TrustDecision.NO_GAUGE_DATA;
    }

    /**
     * Checks whether the trust evaluation is based on sufficiently big data sample.
     * @param trustedAgent agent whose trustworthiness is evaluated
     * @return true if the trustworthiness evaluation is based on sufficiently large experience
     */
    public boolean hasEnoughTrustDataAbout(AgentKnowledge trustedAgent)
    {
    	IterativeFuzzyMembership<AgentKnowledge> mem = ((IterativeFuzzyMembership<AgentKnowledge>)reputation.getMembership(trustedAgent));
		if ( null != mem && mem.hasEnoughData())
    	{
    		return true;
    	}
    	return false;
    }

    /**
     * Generates a reputation opinion about the agent. Makes a weighted aggregation of trustfulness evaluations, using both
     * the trustfulness of the source and the uncertainty reported by the source.
     * Uses the reputation VALUES received from other agents.
     * @return boolean trustfulness value
     */
    public TrustDecision hasGoodReputation()
    {
    	double trustingWeight = 0;
    	double distrustingWeight = 0;
    	// iterrate over reputation sources
    	for (ListIterator<Membership<AgentKnowledge>> it = reputationSources.getMembershipIterator(); it.hasNext();) {
			IterativeFuzzyMembership<AgentKnowledge> sourceMem = (IterativeFuzzyMembership<AgentKnowledge>) it.next();
			// get their opinion and trustfulness
			AgentKnowledge trustor = sourceMem.getMember();
			double indWeight = 0;
			IterativeFuzzyMembership<AgentKnowledge> trusteeEval = (IterativeFuzzyMembership<AgentKnowledge>) trustor.reputation.getMembership(this);
			// check whether we have self-trust of the trustor
			TrustDecision td = trustor.trusts(this);
			indWeight = trustor.trust.getMembershipFunctionCenter() * trusteeEval.getMembershipFunctionUncertainity();
			switch (td) {
			case TRUSTED:
				trustingWeight += indWeight;
				break;
			case DISTRUSTED:
				distrustingWeight += indWeight;
				break;
			case NOT_ENOUGH_DATA:
				// do nothing here, no opinion
				break;
			case NO_GAUGE_DATA:
				// use our perception of trustor's trustfulness as a gauge
				// 0.5 discount it compared with previous one... one half...
				TrustDecision etd = trustor.shouldTrust(this);
				switch (etd) {
				case TRUSTED:
					trustingWeight += 0.5 * indWeight;
					break;
				case DISTRUSTED:
					distrustingWeight += 0.5 * indWeight;
					break;
				}
				break;
			}
    	}
    	if (trustingWeight + distrustingWeight > 0)
    	{
    		if (trustingWeight >= distrustingWeight)
    			return TrustDecision.TRUSTED;
    		return TrustDecision.DISTRUSTED;
    	}
    	return TrustDecision.NOT_ENOUGH_DATA;
    }

    /** Returns the average numeric value of agent's reputation as received from others. */
	public double getAverageReputation() {
		double sampleUnc = 0;
		double cummulatedAvgValue = 0;
    	for (ListIterator<Membership<AgentKnowledge>> it = reputationSources.getMembershipIterator(); it.hasNext();) {
			IterativeFuzzyMembership<AgentKnowledge> sourceMem = (IterativeFuzzyMembership<AgentKnowledge>) it.next();
			// get their opinion and trustfulness
			AgentKnowledge trustor = sourceMem.getMember();
			IterativeFuzzyMembership<AgentKnowledge> trusteeEval = (IterativeFuzzyMembership<AgentKnowledge>) trustor.reputation.getMembership(this);
			cummulatedAvgValue += trusteeEval.getMembershipFunctionCenter() / trusteeEval.getMembershipFunctionUncertainity();
			sampleUnc += 1 /trusteeEval.getMembershipFunctionUncertainity();
		}
    	if (sampleUnc > 0)
    		return cummulatedAvgValue / sampleUnc;
    	return 0.5;//TODO - find something better...
	}

	public Membership getApplicableTrustworthiness() {
		return getTrust();
		//TODO - update to cover reputation and other modes...
	}

	public List<ResourceKnowledge> getResources() {
		return resources;
	}

	/**
	 * Adds a new resource to the list of resources maintained for the agent.
	 * @param knowledge
	 */
	public void addResource(ResourceKnowledge knowledge) {
		if (!resources.contains(knowledge))
		{
			resources.add(knowledge);
		}
	}
}
